#include "LS013B7DH03.h"
#include "SILABS_RHT.h"
#include "mbed_logo_thermo.h"
#include "Animation.h"

/******************** Define I/O *****************************/

InterruptIn in(SW0);

#define SCK     PE12
#define MOSI 	PE10

DigitalOut CS(PA10);
DigitalOut EXTCOM(PF3);
DigitalOut DISP(PA8);

SPI displaySPI(MOSI, NC, SCK);
silabs::LS013B7DH03 display(&displaySPI, &CS, &EXTCOM);

I2C sensorI2C(PD6, PD7);
DigitalOut SENS_EN(PC8);
silabs::SILABS_RHT rhtSensor(&sensorI2C);

/******************** Define Timers *****************************/

LowPowerTicker timeKeeping;

/***************** Define global variables **********************/
#define INIT_SECONDS		17600
#define TEST_DURATION		10

volatile uint32_t prevSeconds = INIT_SECONDS, seconds = INIT_SECONDS;
volatile bool refreshed = false;
volatile bool clockShown = false;
volatile bool measured = false;

typedef enum {
	CLOCK,
	DODECA
} display_mode_t;

volatile display_mode_t currentMode = CLOCK;
uint8_t currentFrame = 0;

/***************** Define callback handlers *********************/
void secondsCallback(void);
void in_handler();
void toggleCallback(void);

/***************** Define helper functions ***********************/
void drawTemperature(int32_t temperature);

void drawTemperature(int32_t temperature) {
    uint32_t line = 20;
    int8_t int_temp = temperature / 500;

    for(int8_t iterator = 80; iterator > -6; iterator--) {
        if(int_temp >= iterator) display.fill(108,line,10,1,Black);
        else display.fill(108,line,10,1,White);

        line += 1;
    }
}

/**
 * Callback for pushbutton interrupt
 */
void in_handler() {
    if(currentMode == CLOCK) {
    	currentMode = DODECA;
    } else {
    	currentMode = CLOCK;
    	clockShown = false;
    }
}

void secondsCallback(void) {
	seconds++;
}

/**
 * Callback for refresh completion
 */
void refreshCallback(void) {
	refreshed = true;
}

/**
 * Callback for measurement completion
 */
void measureCallback(void) {
	if(rhtSensor.get_active()) {
		measured = true;
	}
}

/*************************** MAIN *******************************/
int main() {
	// Initialize pushbutton handler
	in.rise(NULL);
	in.fall(in_handler);

    // Enable the LCD
	DISP = 1;

	// Enable the I2C RHT sensor
	SENS_EN = 1;

	// Start generating the 1Hz call for the timekeeping
	timeKeeping.attach(&secondsCallback, 1.0f);

	// Reset the LCD to a blank state. (All white)
	refreshed = false;
	display.clearImmediate(refreshCallback);
	while(refreshed == false) sleep();

	printf("Initialization done! \n");

	// Apply mbed logo bitmap to the pixel buffer
	display.showBMP((uint8_t*)mbed_logo_thermometer, 128, 128, 0, 0);
	display.background(Black);
	display.foreground(White);
	display.printf("I like MBED!");

	// Push update to the display
	uint32_t refreshCount = display.getRefreshTicks();
	refreshed = false;
	display.update(refreshCallback);

	// Sleep while doing the transmit
	while(refreshed == false) sleep();

	// Calculate and print refresh duration
	refreshCount = display.getRefreshTicks() - refreshCount;
	printf("Refresh duration: %d cycles @ 125Hz \n", (int)refreshCount);

	// Perform a measurement
	rhtSensor.check_availability(si7021, measureCallback);
	while(measured == false);

	/* Main loop */
	while(1) {
		sleep();

		// In clock mode, only update once per second
		if((currentMode == CLOCK) && (prevSeconds != seconds)) {
			/* Redraw background when coming from video mode */
		    if(clockShown == false) {
				display.showBMP((uint8_t*)mbed_logo_thermometer, 128, 128, 0, 0);
				clockShown = true;
			}

			/* Show numeric measurements */
			display.locate(0,13);
			display.printf("%02d.%01d degC", rhtSensor.get_temperature() / 1000, (rhtSensor.get_temperature() % 1000) / 100);
			display.locate(0,14);
			display.printf("%03d.%03d%%H", rhtSensor.get_humidity() / 1000, rhtSensor.get_humidity() % 1000);

			/* Show clock */
			display.locate(4,15);
			display.printf("%02d:%02d:%02d", (seconds / 1200) % 24, (seconds / 60) % 60, seconds % 60);

			/* Update the graphical thermometer */
			drawTemperature(rhtSensor.get_temperature());

			if(refreshed == true) {
				prevSeconds = seconds;
				refreshed = false;

				/* Perform both I/O tasks simultaneously */
				display.update(refreshCallback);
				rhtSensor.measure(si7021);
			}
		}
		// In dodecaeder mode, show frames as fast as possible
		else if(currentMode == DODECA) {
			if(refreshed == true) {
				display.showBMP((uint8_t*)(frames[currentFrame]->pData), 128, 128, 0, 0);
				currentFrame++;
				if( currentFrame >= NUM_FRAMES ) currentFrame = 0;
				refreshed = false;
				display.update(refreshCallback);
			}
		}
	}

	//notify_completion(true);
}

