// test FastCRC performance, hardware vs avr softcrc
//FastCRC
//Benchmark
//
//(c) Frank Boesing 2014

#include "mbed.h"
#include "FastCRC.h"
#include "crc16.h"

Timer tmr;
#define micros tmr.read_us

#define BUFSIZE 16384


FastCRC8 CRC8;
FastCRC16 CRC16;
FastCRC32 CRC32;

uint8_t buf[BUFSIZE] __attribute__((aligned(4)));

// validation
void printva(char * name, uint32_t check, uint32_t val){
    printf("%s",name);
    if (check == val)
        printf(" is ok");
    else
        printf(" is NOT ok");
    printf("\n");
}

void validate() {
    uint32_t crc;
    uint8_t buf[9] = {'1','2','3','4','5','6','7','8','9'};


  printf("CRC Validation\n");

  crc = CRC8.smbus(buf, sizeof(buf));
  printva("SMBUS", 0xf4, crc);

  crc = CRC8.maxim(buf, sizeof(buf));
  printva("Maxim", 0xa1, crc);

  crc = CRC16.ccitt(buf, sizeof(buf));
  printva("CCITT", 0x29b1, crc);

  crc = CRC16.mcrf4xx(buf, sizeof(buf));
  printva("MCRF4XX", 0x6f91, crc);

  crc = CRC16.modbus(buf, sizeof(buf));
  printva("MODBUS", 0x4b37, crc);

  crc = CRC16.kermit(buf, sizeof(buf));
  printva("KERMIT", 0x2189, crc);

  crc = CRC16.xmodem(buf, sizeof(buf));
  printva("XMODEM", 0x31c3, crc);

  crc = CRC16.x25(buf, sizeof(buf));
  printva("X.25", 0x906e, crc);

  crc = CRC32.crc32(buf, sizeof(buf));
  printva("CRC32", 0xcbf43926, crc);

  crc = CRC32.cksum(buf, sizeof(buf));
  printva("CKSUM", 0x765e7680, crc);
}


// Supporting functions for Software CRC

inline uint16_t softcrc(uint16_t seed, uint8_t *data, uint16_t datalen) {
    for (uint16_t i=0; i<datalen; i++) {
        seed = _crc16_update(seed,  data[i]);
    }
    return seed;
}

inline uint16_t softcrcIbutton(uint16_t seed, uint8_t *data, uint16_t datalen) {
    for (uint16_t i=0; i<datalen; i++) {
        seed = _crc_ibutton_update(seed,  data[i]);
    }
    return seed;
}

inline uint16_t softcrcCCIT(uint16_t seed, uint8_t *data, uint16_t datalen) {
    for (uint16_t i=0; i<datalen; i++) {
        seed = _crc_ccitt_update(seed,  data[i]);
    }
    return seed;
}

inline uint16_t softcrcXMODEM(uint16_t seed, uint8_t *data, uint16_t datalen) {
    for (uint16_t i=0; i<datalen; i++) {
        seed = _crc_xmodem_update(seed,  data[i]);
    }
    return seed;
}


void printVals(char * name, uint32_t crc, uint32_t time) {
    printf("%s Value:0x%x, Time: %d us\n",name,crc,time);
}

main() {
    uint32_t time;
    uint32_t crc;

  tmr.start();
  printf("\nSystemCoreClock %d %s %s\n",SystemCoreClock,__TIME__,__DATE__);
  validate();
  printf("\nCRC Benchmark %d bytes\n",sizeof(buf));

  //Fill array with data
  for (int i=0; i<BUFSIZE; i++) {
    buf[i] = (i+1) & 0xff;
  }


  time = micros();
  crc = CRC8.maxim(buf, BUFSIZE);
  time = micros() - time;
  printVals("Maxim (iButton) FastCRC:",crc,time);

  time = micros();
  crc = softcrcIbutton(0, buf, BUFSIZE);
  time = micros() - time;
  printVals("Maxim (iButton) builtin:",crc,time);


  time = micros();
  crc = CRC16.modbus(buf, BUFSIZE);
  time = micros() - time;
  printVals("MODBUS FastCRC:",crc,time);

  time = micros();
  crc = softcrc(0xffff, buf, BUFSIZE);
  time = micros() - time;
  printVals("MODBUS builtin:",crc,time);


  time = micros();
  crc = CRC16.xmodem(buf, BUFSIZE);
  time = micros() - time;
  printVals("XMODEM FastCRC:",crc,time);

  time = micros();
  crc = softcrcXMODEM(0, buf, BUFSIZE);
  time = micros() - time;
  printVals("XMODEM builtin:",crc,time);

  time = micros();
  crc = CRC16.mcrf4xx(buf,BUFSIZE);
  time = micros() - time;
  printVals("MCRF4XX FastCRC:",crc,time);

  time = micros();
  crc = softcrcCCIT(0xffff, buf, BUFSIZE);
  time = micros() - time;
  printVals("MCRF4XX builtin:",crc,time);


  time = micros();
  crc = CRC16.kermit(buf, BUFSIZE);
  time = micros() - time;
  printVals("KERMIT FastCRC:",crc,time);

  time = micros();
  crc = CRC32.crc32(buf, BUFSIZE);
  time = micros() - time;
  printVals("Ethernet FastCRC:",crc,time);
}
