
/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "MicroBit.h"
#include "mbed.h"
#include "ble/BLE.h"
//#include "TMP_nrf51/TMP_nrf51.h"
//nrf51822 temperature
#include "nrf.h"
#include "nrf_temp.h"

#define APP_SPECIFIC_ID_TEST 0x0059 //0xFEFE

#pragma pack(1)
struct ApplicationData_t {
    uint16_t applicationSpecificId;             /* An ID used to identify temperature value
                                                   in the manufacture specific AD data field */
    uint8_t tmpSensorValue; /* User defined application data */
    uint8_t btnA_Value;
    uint8_t btnB_Value;
};
#pragma pack()

MicroBit ubit;
BLE ble;
//TMP_nrf51 tempSensor;
DigitalOut alivenessLED(LED1, 1);

DigitalIn buttonA(MICROBIT_PIN_BUTTON_A);
DigitalIn buttonB(MICROBIT_PIN_BUTTON_B);

static bool triggerTempValueUpdate = false;
uint8_t temp;
uint8_t bA;
uint8_t bB;

void periodicCallback(void)
{
    /* Do blinky on LED1 while we're waiting for BLE events */
    alivenessLED = !alivenessLED;
    triggerTempValueUpdate = true;
}

void accumulateApplicationData(ApplicationData_t &appData)
{
//nrf51822 temperature
    NRF_TEMP->TASKS_START = 1; /* Start the temperature measurement */
    while ((NRF_TEMP->EVENTS_DATARDY & TEMP_INTENSET_DATARDY_Msk) != (TEMP_INTENSET_DATARDY_Set << TEMP_INTENSET_DATARDY_Pos)) {}
    NRF_TEMP->EVENTS_DATARDY = 0;
    temp = nrf_temp_read()/4;
    NRF_TEMP->TASKS_STOP = 1; /* Stop the temperature measurement */

    appData.applicationSpecificId = APP_SPECIFIC_ID_TEST;
    /* Read a new temperature value */
    appData.tmpSensorValue = temp;
    appData.btnA_Value = bA;
    appData.btnB_Value = bB;
}

void temperatureValueAdvertising(void)
{
    ApplicationData_t appData;

    accumulateApplicationData(appData);
    //printf("Temp is %f\r\n", (float)appData.tmpSensorValue);

    /* Setup advertising payload */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE); /* Set flag */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_THERMOMETER); /* Set appearance */

    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,(const uint8_t *)"Mammoth", sizeof("Mammoth") - 1);

    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)&appData, sizeof(ApplicationData_t)); /* Set data */
    /* Setup advertising parameters */
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_NON_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(500);
    /* Start advertising */
    ble.gap().startAdvertising();
}

void updateSensorValueInAdvPayload(void)
{
    ApplicationData_t appData;

    accumulateApplicationData(appData);

    /* Stop advertising first */
    ble.gap().stopAdvertising();
    /* Only update temperature value field */
    ble.gap().updateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)&appData, sizeof(ApplicationData_t));
    /* Start advertising again */
    ble.gap().startAdvertising();
}

int main(void)
{
//nrf51822 temperature
    nrf_temp_init();

    Ticker ticker;
    /* Enable trigger every 2 seconds */
    ticker.attach(periodicCallback, 1);

    ble.init();
    /* Start temperature advertising */
    temperatureValueAdvertising();

    while (true) {
        if (triggerTempValueUpdate) {
            /* Update temperature value */
            updateSensorValueInAdvPayload();
            triggerTempValueUpdate = false;
        }
        if(buttonA == 0x00) {
            bA = 0x00;
        } else if(buttonA == 0x01) {
            bA = 0x01;
        }
        if(buttonB == 0x00) {
            bB = 0x00;
        } else if(buttonB == 0x01) {
            bB = 0x01;
        }
        ble.waitForEvent();
    }
}
