/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "ble/DiscoveredCharacteristic.h"
#include "ble/DiscoveredService.h"

DigitalOut alivenessLED(LED1, 1);
DigitalOut connectLED(P0_22, 0);

Serial pc(USBTX, USBRX);

bool                     triggerLedCharacteristic = false;
DiscoveredCharacteristic ledCharacteristic;
uint8_t bVal = 0;

//Функции обработки событий кнопок
void triggerfall_1();   //Button1 falling interrupt function
void triggerrise_1();   //Button1 rising interrupt function
void triggerfall_2();   //Button2 falling interrupt function
void triggerrise_2();   //Button2 rising interrupt function

DigitalIn  sw1(BUTTON1);
DigitalIn  sw2(BUTTON2);

//Initiate input interrupts
InterruptIn sw1Press(BUTTON1);
InterruptIn sw2Press(BUTTON2);

BLE        ble;
Ticker ticker;

//событие по таймеру
void periodicCallback(void) {
//Мигаем светодиодом
    alivenessLED = !alivenessLED; /* Do blinky on LED1 while we're waiting for BLE events */
}
//результат сканирования
void advertisementCallback(const Gap::AdvertisementCallbackParams_t *params) {
/*    if (params->peerAddr[0] != 0x37) { // !ALERT! Alter this filter to suit your device.
        return;
    }*/
    printf("adv peerAddr[%02x %02x %02x %02x %02x %02x] rssi %d, isScanResponse %u, AdvertisementType %u\r\n",
           params->peerAddr[5], params->peerAddr[4], params->peerAddr[3], params->peerAddr[2], params->peerAddr[1], params->peerAddr[0],
           params->rssi, params->isScanResponse, params->type);

    BLE::Instance().gap().connect(params->peerAddr, Gap::ADDR_TYPE_RANDOM_STATIC, NULL, NULL);
}
//поиск сервисов
void serviceDiscoveryCallback(const DiscoveredService *service) {
    printf("TEST!!!!!!!!\r\n");
    printf("S UUID-%x\r\n", service->getUUID().getShortUUID());
    if (service->getUUID().shortOrLong() == UUID::UUID_TYPE_SHORT) {
        printf("S UUID-%x attrs[%u %u]\r\n", service->getUUID().getShortUUID(), service->getStartHandle(), service->getEndHandle());
    } else {
        printf("S UUID-");
        const uint8_t *longUUIDBytes = service->getUUID().getBaseUUID();
        for (unsigned i = 0; i < UUID::LENGTH_OF_LONG_UUID; i++) {
            printf("%02x", longUUIDBytes[i]);
        }
        printf(" attrs[%u %u]\r\n", service->getStartHandle(), service->getEndHandle());
    }
}
//поиск характеристик и вывод инфы
void characteristicDiscoveryCallback(const DiscoveredCharacteristic *characteristicP) {
    printf("C UUID-%x valueAttr[%u] props[%x]\r\n", characteristicP->getUUID().getShortUUID(), characteristicP->getValueHandle(), (uint8_t)characteristicP->getProperties().broadcast());
    if (characteristicP->getUUID().getShortUUID() == 0xa001) { /* !ALERT! Alter this filter to suit your device. */
        ledCharacteristic        = *characteristicP;
        triggerLedCharacteristic = true;
    }
}

void discoveryTerminationCallback(Gap::Handle_t connectionHandle) {
    printf("terminated SD for handle %u\r\n", connectionHandle);
}
//Действие при событии "onConnection"
void connectionCallback(const Gap::ConnectionCallbackParams_t *params) {
    if (params->role == Gap::CENTRAL) {
        BLE::Instance().gattClient().onServiceDiscoveryTermination(discoveryTerminationCallback);
        BLE::Instance().gattClient().launchServiceDiscovery(params->handle, serviceDiscoveryCallback, characteristicDiscoveryCallback, 0xa000, 0xa001);
    }
    printf("connected\r\n");
    connectLED = 1;
}

//Вызывается событием "onDataRead"
void triggerToggledWrite(const GattReadCallbackParams *response) {
    if (response->handle == ledCharacteristic.getValueHandle()) {
//#if DUMP_READ_DATA
        printf("triggerToggledWrite: handle %u, offset %u, len %u\r\n", response->handle, response->offset, response->len);
        for (unsigned index = 0; index < response->len; index++) {
            printf("%c[%02x]", response->data[index], response->data[index]);
        }
        printf("\r\n");
//#endif
//пишем в характеристику 0х00 или 0х01, т.е. моргаем диодом дистанционно
//        uint8_t toggledValue = response->data[0] ^ 0x1;
        uint8_t toggledValue = bVal;
        ledCharacteristic.write(1, &toggledValue);
        printf("onDataRead\r\ntoggledValue: %02x\r\n", toggledValue);
    }
}

//Вызывается событием "onDataWrite"
void triggerRead(const GattWriteCallbackParams *response) {
    if (response->handle == ledCharacteristic.getValueHandle()) {
        ledCharacteristic.read();
        printf("onDataWrite\r\n");
    }
}

//Действие при событии "Disconnect"
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params) {
    printf("disconnected\r\n");
    pc.printf("Rescan\r\n");
    ble.gap().startScan(advertisementCallback);
    connectLED = 0;
}

/**
 * This function is called when the ble initialization process has failed
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }
 
    ble.gap().onConnection(connectionCallback);
    ble.gap().onDisconnection(disconnectionCallback);

    ble.gattClient().onDataRead(triggerToggledWrite);
    ble.gattClient().onDataWrite(triggerRead);

    ble.gap().setScanParams(500, 400);
    ble.gap().startScan(advertisementCallback);
}

int main(void) {
    pc.baud(9600);
    printf("--- Start!!! ---\r\n");

//Set falling and rising edge to apppropriate interrup function
    sw1Press.fall(&triggerfall_1);
    sw1Press.rise(&triggerrise_1);
    sw2Press.fall(&triggerfall_2);
    sw2Press.rise(&triggerrise_2);
    
    ticker.attach(periodicCallback, 1);

    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }

    while (true) {
        if (triggerLedCharacteristic && !ble.gattClient().isServiceDiscoveryActive()) {
            triggerLedCharacteristic = false;
            ledCharacteristic.read(); /* We could have issued this read just as easily from
                                       * characteristicDiscoveryCallback(); but
                                       * invoking it here demonstrates the use
                                       * of isServiceDiscoveryActive() and also
                                       * the fact that it is permitted to
                                       * operate on application-local copies of
                                       * DiscoveredCharacteristic. */
        }
        ble.waitForEvent();
    }
}

//Button1 falling interrupt function
void triggerfall_1()
{
    bVal =! bVal;
}
//Button1 rising interrupt function
void triggerrise_1()
{
}

//Button1 falling interrupt function
void triggerfall_2()
{
}
//Button1 rising interrupt function
void triggerrise_2()
{
}
