/* This is a project for madelectronengineering.com
 This code is used for a hardware monitoring system for the Hybrid Supercapacitor Car Battery to monitor the supercap voltage, the LiFeP04 voltage,
current, and wattage, plus also monitor 3 seperate temperature points.

Required hardware:
    STM32 NUCLEO-L073RZ
    TMP36 Temperature Sensors (x3)
    INA219 Current & Voltage Breakout (modify by removing 0.1Ohm resistor and replacing with 0.01Ohm Resistor to measure up to 32Amps
    ST NUCLEO Protoboard
    30k & 7.5k resistors for voltage divider to measure supercap voltage
    0.1Ohm 25W current limiting power resistor in between supercap array and LiFeP04 battery
    Sharp Memory LCD 400x240
    
The INA219 measures the current and amperage on the battery side of the current limiting resistor to keep an eye on the LiFeP04 battery.
voltage divider connects directly to + of supercap array

Arduino equivelant connections:
A0 - TMP36 #1
A1 - TMP36 #2
A2 - TMP36 #3
A3 - Voltage divider Input
A4&A5 - I2C for INA219

Arduino equivelant connections for LCD Screen:
D13 - SClk
D11 - MOSI
D10 - CD
D9 - Enable
D8 - Extcom

*/
#include "mbed.h"
#include "SharpLCD.h"
#include "INA219.hpp"
#include "Neu44x36.h"
#include "Neu31x26.h"
#include "WakeUp.h"

SharpLCD display(PA_7, NC, PA_5, PB_6, PC_7, PA_9); //mosi, miso(not used), sck, cs, enable, extcom
INA219 ina219(PC_1, PC_0, 0x40, 400000, RES_12BITS);
AnalogIn   ain1(PA_0); // connect A0 to Vout(Temp36)
AnalogIn   ain2(PA_1); // connect A1 to Vout(Temp36)
AnalogIn   ain3(PA_4); // connect A2 to Vout(Temp36)
AnalogIn   ain4(PB_0); // Connect A3 to Voltage Divider

int main()
{        
    display.enableDisplay();  //enable sharp memory lcd
    display.clearImmediate();   //clear the screen buffer and screen
    
    display.set_font(Neu44x36);
       
       //only send graphics and text once to the screen during bootup
       display.locate(17,0);
       display.printf("Hybrid Supercap");
       display.locate(75,35);
       display.printf("Car Battery");
       
       display.rect(0,73,195,239, Black);
       display.rect(205,73,399,239, Black);
       display.fillrect(195,73,10,167, Black);
       display.line(1,174,194,174, Black);
       display.line(1,176,194,176, Black);
       display.line(1,107,194,107, Black);
       display.line(206,107,399,107, Black);
       display.line(1,141,194,141, Black);
       display.line(206,141,399,141, Black);
       display.line(1,175,194,175, Black);
       display.line(206,175,399,175, Black);
       display.line(1,209,194,209, Black);
       display.line(206,209,399,209, Black);
       
       display.set_font(Neu31x26);
       display.locate(15,76);
       display.printf("Supercaps");
       display.locate(240,76);
       display.printf("LiFePo4");
       display.locate(30,179);
       display.printf("Resistor");
       
       display.locate(360,110);
       display.printf("V");
       display.locate(360,143);
       display.printf("A");
       display.locate(360,177);
       display.printf("W");
       display.locate(155,212);
       display.printf("'F");
       display.locate(360,212);
       display.printf("'F");
       display.locate(155,143);
       display.printf("'F");
       display.locate(155,110);
       display.printf("V");
       
  while(1) {    
       
       float volt;
       float current_ma;
       float power;
                  
       volt       = ina219.read_bus_voltage();
       current_ma = ina219.read_current_mA() / 1000;
       power      = volt * current_ma;
       
       float V1 = ain1.read() * 3.3; // connect Vs(Tmp36) to 3.3V        
       float tempC1 = (V1-0.5) * 100; // calculate temperature C
       float tempF1 = (tempC1 * 9 / 5) + 32.0; // calculate temperature F
       
       float V2 = ain2.read() * 3.3; // connect Vs(Tmp36) to 3.3V        
       float tempC2 = (V2-0.5) * 100; // calculate temperature C
       float tempF2 = (tempC2 * 9 / 5) + 32.0; // calculate temperature F
       
       float V3 = ain3.read() * 3.3; // connect Vs(Tmp36) to 3.3V        
       float tempC3 = (V3-0.5) * 100; // calculate temperature C
       float tempF3 = (tempC3 * 9 / 5) + 32.0; // calculate temperature F
 
       float V4 = (ain4.read()) * 3.3; // Voltage divider for supercap voltage
       float voltage = (V4 * 5);   // Voltage divider is 1 to 5 ratio
   
       display.set_font(Neu31x26);
       display.locate(210,110);
       display.printf("      ");
       display.locate(210,110);
       display.printf("%+05.2f", volt);      
       display.locate(210,143);
       display.printf("      "); 
       display.locate(210,143);
       display.printf("%+04.1f", current_ma);      
       display.locate(210,177);
       display.printf("      ");
       display.locate(210,177);
       display.printf("%+05.1f", power);      
       display.locate(20,212);
       display.printf("      ");
       display.locate(20,212);
       display.printf("%+03.0f", tempF1);      
       display.locate(210,212);
       display.printf("      ");
       display.locate(210,212);
       display.printf("%+03.0f", tempF2);      
       display.locate(20,143);
       display.printf("      ");
       display.locate(20,143);
       display.printf("%+03.0f", tempF3);      
       display.locate(20,110);
       display.printf("      ");
       display.locate(20,110);
       display.printf("%+05.2f", voltage);       

       display.update();
 
//Set wakeup time for 1 second
//        WakeUp::set_ms(500);
        
        //Enter deepsleep, the program won't go beyond this point until it is woken up
//        deepsleep();   
}
}