/* MAX1704X Simple Driver Library
 * Copyright (c) 2017 Mac Lobdell
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef MAX1704X_H
#define MAX1704X_H

#include "mbed.h"

/** MAX1704X class.
 *  Used for reading  MAX1704X lipo fuel guage connected via I2C.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "MAX1704X.h"
 *
 * //Create an MAX1704X object at the default address (ADDRESS_0)
 * MAX1704X battery_level(p28, p27);
 *
 * int main()
 * {
 *     //Try to open the MAX1704X
 *     if (battery_level.open()) {
 *         printf("Device detected!\n");
 *
 *         while (1) {
 *             //Print the current battery level
 *             printf("battery = %d\n", battery_level);
 *
 *             //Sleep for 1 seconds
 *             wait(1);
 *         }
 *     } else {
 *         error("Device not detected!\n");
 *     }
 * }
 * @endcode
 */
class MAX1704X
{
public:
    /** Represents the different I2C address possibilities for the MAX1704X
     */
     
    enum Address {
        ADDRESS_0 = (0x36 << 1)    /* Slave Address 011 0110 + r/w bit */
    };

    /** Create an MAX1704X object connected to the specified I2C pins with the specified I2C slave address
     *
     * @param sda The I2C data pin.
     * @param scl The I2C clock pin.
     * @param addr The I2C slave address (defaults to ADDRESS_0).
     * @param hz The I2C bus frequency (defaults to 400kHz).
     */
    MAX1704X(PinName sda, PinName scl, Address addr = ADDRESS_0, int hz = 400000);

    /** Probe for the MAX1704X and indicate if it's present on the bus
     *
     * @returns
     *   'true' if the device exists on the bus,
     *   'false' if the device doesn't exist on the bus.
     */
    bool open();

    /** assert power on reset on MAX1704X
     *
     * @returns void
     */
     void power_on_reset(void);

    /** Get the current ad of the MAX1704X
     *
     * @returns The current ad.
     */
    uint32_t read_ad();

    /** Get the current config of the MAX1704X
     *
     * @returns The current config of the MAX170X.
     */
    uint16_t read_config();

    /** Get the current battery charge percent from MAX1704X
     *
     * @returns The current battery charge percent from MAX1704X.
     */
    uint32_t read_percent();

//#ifdef MBED_OPERATORS
    /** A shorthand for percentage()
     *
     * @returns the current battery percentage measurement.
     */
//    operator uint32_t();
//#endif

private:
    //I2C register addresses
    enum Register {
        REG_VCELL_H = 0x02,
        REG_VCELL_L = 0x03,
        REG_SOC_H   = 0x04,
        REG_SOC_L   = 0x05,        
        REG_MODE_H  = 0x06,        
        REG_MODE_L  = 0x07,                
        REG_VER_H   = 0x08,        
        REG_VER_L   = 0x09,                
        REG_CONFIG_H  = 0x0C,        
        REG_CONFIG_L  = 0x0D,                                
        REG_COMMAND_H  = 0xFE,        
        REG_COMMAND_L  = 0xFF                                       
    };

    //Member variables
    I2C m_I2C;
    const int m_ADDR;
    
    //Internal functions
    char read8(char reg);
    void write8(char reg, char data);
    uint16_t read16(char reg);
    void write16(char reg, uint16_t data);
};

#endif
