/*************************************************** 
  This is a library for our I2C LED Backpacks

  Designed specifically to work with the Adafruit LED Matrix backpacks 
  ----> http://www.adafruit.com/products/
  ----> http://www.adafruit.com/products/

  These displays use I2C to communicate, 2 pins are required to 
  interface. There are multiple selectable I2C addresses. For backpacks
  with 2 Address Select pins: 0x70, 0x71, 0x72 or 0x73. For backpacks
  with 3 Address Select pins: 0x70 thru 0x77

  Adafruit invests time and resources providing this open source code, 
  please support Adafruit and open-source hardware by purchasing 
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.  
  BSD license, all text above must be included in any redistribution
 ****************************************************/
 
 /*
 *  Modified by Luiz Hespanha (http://www.d3.do) 8/16/2013 for use in LPC1768
 */

#include "mbed.h"
#include "Adafruit_LEDBackpack.h"
#include "Adafruit_GFX.h"
#include "glcdfont.h"

//mpl - something with stdio screwing things up.  need to define serial port to use for debug
extern Serial pc;

void Adafruit_LEDBackpack::setBrightness(uint8_t b) {
  if (b > 15) b = 15;
  uint8_t c = 0xE0 | b;
  char foo[1];
  foo[0] = c;
  _i2c->write(i2c_addr, foo, 1); 
}

void Adafruit_LEDBackpack::blinkRate(uint8_t b) {
  if (b > 3) b = 0; // turn off if not sure
  uint8_t c = HT16K33_BLINK_CMD | HT16K33_BLINK_DISPLAYON | (b << 1);
  char foo[1];
  foo[0] = c;
  _i2c->write(i2c_addr, foo, 1);
}

Adafruit_LEDBackpack::Adafruit_LEDBackpack(I2C *i2c): _i2c(i2c) {
}

void Adafruit_LEDBackpack::begin(uint8_t _addr = 0x70) {
  i2c_addr = _addr << 1;

  char foo[1];
  foo[0] = 0x21;

  _i2c->write(i2c_addr, foo, 1);  // turn on oscillator

  blinkRate(HT16K33_BLINK_OFF);
  
  setBrightness(15); // max brightness
}

void Adafruit_LEDBackpack::writeDisplay(void) {
  char foo[17];
  foo[0] = 0x00;
  int j = 0;
  for (uint8_t i=1; i<=16; i+=2) {
    int x = displaybuffer[j] & 0xFF;
    foo[i] = x;
    int x2 = displaybuffer[j] >> 8;
    foo[i+1] = x2;
    j++;
  }
  _i2c->write(i2c_addr, foo, 17);
}

void Adafruit_LEDBackpack::clear(void) {
  for (uint8_t i=0; i<16; i++) {
    displaybuffer[i] = 0;
  }
}

Adafruit_8x8matrix::Adafruit_8x8matrix(I2C *i2c) : Adafruit_LEDBackpack(i2c), Adafruit_GFX(8, 8) {
}

void Adafruit_8x8matrix::drawPixel(int16_t x, int16_t y, uint16_t color) {
  if ((y < 0) || (y >= 8)) return;
  if ((x < 0) || (x >= 8)) return;

 // check rotation, move pixel around if necessary
  switch (getRotation()) {
  case 1:
    swap(x, y);
    x = 8 - x - 1;
    break;
  case 2:
    x = 8 - x - 1;
    y = 8 - y - 1;
    break;
  case 3:
    swap(x, y);
    y = 8 - y - 1;
    break;
  }

  // wrap around the x
  x += 7;
  x %= 8;


  if (color) {
    displaybuffer[y] |= 1 << x;
  } else {
    displaybuffer[y] &= ~(1 << x);
  }
  

}


Adafruit_16x8matrix::Adafruit_16x8matrix(I2C *i2c) : Adafruit_LEDBackpack(i2c), Adafruit_GFX(16, 8) {
}

void Adafruit_16x8matrix::drawPixel(int16_t x, int16_t y, uint16_t color) {
  if ((y < 0) || (y >= 8)) return;
  if ((x < 0) || (x >= 16)) return;
  
 // check rotation, move pixel around if necessary 
  switch (getRotation()) {
  case 1:
    swap(x, y);
    x = 16 - x - 1;
    break;
  case 2:
    x = 16 - x - 1;
    y = 8 - y - 1;
    break;
  case 3:
    swap(x, y);
    y = 8 - y - 1;
    break;
  }

  // wrap around the x
  x += 16;
  x %= 16;


  if (color) {
    displaybuffer[y] |= 1 << x;
  } else {
    displaybuffer[y] &= ~(1 << x);
  }
}

size_t Adafruit_16x8matrix::writeChar(uint8_t c)
{
        
    if (c == '\n')
    {
        cursor_y += textsize*8;
        cursor_x = 0;
    }
    else if (c == '\r')
        cursor_x = 0;
    else
    {
        drawChar(cursor_x, cursor_y, c, textcolor, textbgcolor, textsize);
        cursor_x += textsize*6;
       // if (wrap && (cursor_x > (_width - textsize*6)))
       // {
       //     cursor_y += textsize*8;
       //     cursor_x = 0;
        //}
    }
    return 1;
}

void Adafruit_16x8matrix::scrollChar(uint8_t c)
{
        
    if (c == '\n')
    {
        cursor_y += textsize*8;
        cursor_x = 0;
    }
    else if (c == '\r')
        cursor_x = 0;
    else
    {
        for(int i = 0; i < 16; i++)
        {
            drawChar(cursor_x, cursor_y, c, textcolor, textbgcolor, textsize);
            writeDisplay();
            Thread::wait(100);
            clear();
            writeDisplay();
            cursor_x++;
        }    
    }
}

// draw a character
void Adafruit_16x8matrix::drawChar(int16_t x, int16_t y, unsigned char c, uint16_t color, uint16_t bg, uint8_t size)
{
  
//allow clipping   
//    if(
//        (x >= _width) || // Clip right
//        (y >= _height) || // Clip bottom
//        ((x + 5 * size - 1) < 0) || // Clip left
//        ((y + 8 * size - 1) < 0) // Clip top
//        )
//        return;
    
    for (int8_t i=0; i<6; i++ )
    {
        uint8_t line = 0;

        if (i == 5) 
            line = 0x0;
        else 
            line = font[(c*5)+i];
            
        for (int8_t j = 0; j<8; j++)
        {
            if (line & 0x1)
            {
                if (size == 1) // default size
                    drawPixel(x+i, y+j, color);
#ifdef WANT_ABSTRACTS
                else // big size
                    fillRect(x+(i*size), y+(j*size), size, size, color);
#endif                    
            }
            else if (bg != color)
            {
                if (size == 1) // default size
                    drawPixel(x+i, y+j, bg);
#ifdef WANT_ABSTRACTS
                else // big size
                    fillRect(x+i*size, y+j*size, size, size, bg);
#endif
            }
            line >>= 1;
        }
    }
}