/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "EthernetNetIf.h"
#include "SimpleSocket.h"
#include <stdarg.h>

DatagramSocket::DatagramSocket(int port, bool debug) : bufsize(512), debug(debug) {
    host = Host(IpAddr(), port);
    createDatagramSocket();
}

DatagramSocket::DatagramSocket(Host host, bool debug) : host(host), bufsize(512), debug(debug) {
    createDatagramSocket();
}

DatagramSocket::DatagramSocket(IpAddr ip, int port, bool debug) : host(ip, port), bufsize(512), debug(debug) {
    createDatagramSocket();
}

DatagramSocket::~DatagramSocket() {
    delete[] buf;
}

void DatagramSocket::createDatagramSocket() {
    udpSocket.setOnEvent(this, &DatagramSocket::onUDPSocketEvent);
    UDPSocketErr err = udpSocket.bind(host);
    DBG(DatagramError(err));
    buf = new char[bufsize + 1];
}

int DatagramSocket::write(char *buf, int length) {
    if (length > bufsize) length = bufsize;
    this->length = length;
    memcpy(this->buf, buf, length);
    return length;
}

int DatagramSocket::printf(const char* format, ...) {
    va_list argp;
    va_start(argp, format);
    int len = vsnprintf(buf, bufsize, format, argp);
    va_end(argp);
    if (len > 0)
        length = len;
    return len;
}

void DatagramSocket::send(Host& host) {
    DBG("Host = %d.%d.%d.%d:%d\n", host.getIp()[0], host.getIp()[1], host.getIp()[2], host.getIp()[3], host.getPort());
    Net::poll();
    int err = udpSocket.sendto(buf, length, &host);
    DBG(DatagramError((UDPSocketErr) (err < 0 ? err : 0)));
}

void DatagramSocket::send(IpAddr ip, int port) {
    DBG("IP = %d.%d.%d.%d, Port = %d\n", ip[0], ip[1], ip[2], ip[3], port);
    Host host(ip, port);
    send(host);
}

void DatagramSocket::send(char *name, int port) {
    DBG("Name = %s, Port = %d\n", name, port);
    Resolver resolver;
    IpAddr ip = resolver.resolve(name, debug);
    send(ip, port);
}

int DatagramSocket::read(char *buf, int size) {
    int len = length < size ? length : size;
    if (len > 0)
        memcpy(buf, this->buf, len);
    return len;
}

int DatagramSocket::scanf(const char* format, ...) {
    va_list argp;
    va_start(argp, format);
    buf[length] = '\0';
    int ret = vsscanf(buf, format, argp);
    va_end(argp);
    return ret;
}

int DatagramSocket::receive(Host *host, float timeout) {
    timer.reset();
    timer.start();
    readable = false;
    length = 0;

    do {
        Net::poll();
        if (readable) {
            Host host2;
            length = udpSocket.recvfrom(buf, bufsize, &host2);
            if (length > 0 && host)
                *host = host2;
            break;
        }
    } while (timer.read() < timeout);

    return length;
}

void DatagramSocket::setDebug(bool debug) {
    this->debug = debug;
}

void DatagramSocket::onUDPSocketEvent(UDPSocketEvent e) {
    DBG(DatagramEvent(e));

    switch (e) {
        case UDPSOCKET_READABLE: //The only event for now
            readable = true;
            break;
    }
}