/*******************************************************************************
 *  Functions for NSL01 class to communicate with onboard RF radio module via
 *  UART interface.
 *
 *  For more information about the NSL01 LoRaWAN shield:
 *      http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
 *
 *  @note The SerialDevice files are included in the dependencies directory of
 *        the project and these files are necessary for the NSL01 class!
 *
 *  @author  -
 *  @version 1.0
 *  @date    20-June-2018
*******************************************************************************/

//------------------------------------------------------------------------------
//
//  Include Files
//
//------------------------------------------------------------------------------

#include "./dependencies/SerialDevice.h"

//------------------------------------------------------------------------------
//
//  Function: SerialDevice_Open
//
//  @brief Function to open serial device with given parameters.
//
//  @param baudrate : Baudrate for serial communication
//  @param dataBits : Number of data bits for serial communication
//
//------------------------------------------------------------------------------

void
SerialDevice_Open(UINT32        baudRate,
                  int           dataBits)
{
    //--Initialize UART interface
    comm_LoRa.baud(baudRate);
    comm_LoRa.format(dataBits, Serial::None, 1);
    comm_LoRa.attach(&rxCallback, Serial::RxIrq);
}

//------------------------------------------------------------------------------
//
//  Function: SerialDevice_SendData
//
//  @brief Function to transmit data to the RF device.
//
//  @param txBuffer : Pointer to Tx buffer
//  @param txLength : Length of Tx buffer
//
//  @returns number of Tx bytes on success, -1 on error
//
//------------------------------------------------------------------------------

int
SerialDevice_SendData(UINT8* txBuffer, int txLength)
{              
    //--Internal variables and constants
    UINT32  numTxBytes= 0;
    
    //--Display input Tx buffer for debugging
    #ifdef DEBUG
        comm_pc.printf("\r\n Content Tx buffer: %u\r\n",&txBuffer[0]);
    #endif
        
    //--Transmit txBuffer
    while((numTxBytes <= (UINT32)txLength))
    {
        //--Transmit data array byte by byte
        comm_LoRa.putc(txBuffer[numTxBytes]); 
       
        //--Update counter
        numTxBytes++;        
    }
        
    //--Display SLIP encoded message for debugging
    #ifdef DEBUG
        comm_pc.printf("\r\n Encoded transmit command: ");
        
        for (int i=0; i< txLength; i++)
        {
            comm_pc.printf("%02X ",txBuffer[i]);
        }
        
        comm_pc.printf("\r\n");    
    #endif
        
    //--Check if all bytes were written
    if (numTxBytes == (UINT32)txLength + 1)
    {
        //--OK
        #ifdef DEBUG
            comm_pc.printf("\r\n numTX Bytes: %i\n\r", numTxBytes);
        #endif
        
        return numTxBytes;
    }

    return -1;
}

//------------------------------------------------------------------------------
//
//  Function: SerialDevice_SendByte
//
//  @brief Function to transmit a single (wakeup) byte to the RF device.
//
//  @param txByte : Tx byte
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int
SerialDevice_SendByte(UINT8 txByte)
{    
    //--Transmit a single byte
    comm_LoRa.putc(txByte);
    
    return 1;
}

//------------------------------------------------------------------------------
//
//  Function: SerialDevice_ReadData
//
//  @brief Function to manually read data into Rx buffer.
//
//  @note This function is not necessary at the moment, because of Rx callback
//        function!
//
//  @param rxBuffer     : Pointer to Rx buffer
//  @param rxBufferSize : Length of Rx buffer
//
//  @returns number of Rx bytes on success, -1 on error
//
//------------------------------------------------------------------------------

int
SerialDevice_ReadData(UINT8* rxBuffer, int rxBufferSize)
{  
    //--Internal variables and constants
    UINT8 numRxBytes = 0;
    UINT8 tmp_tx;
    
    //--Check if interface is accessible
    if ((!comm_LoRa.readable()))
        return -1;

    //--Read rxBuffer
    while(numRxBytes < rxBufferSize)
    {
        //--Read data byte by byte
        tmp_tx = (UINT8)comm_LoRa.getc();    
        rxBuffer[numRxBytes] = tmp_tx;   
    }
  
    return (int)numRxBytes;      
}

//------------------------------------------------------------------------------
//
//  Function: rxCallback
//
//  @brief UART Rx callback function for data reception (UART IRQ)
//
//------------------------------------------------------------------------------

void 
rxCallback() 
{   
    //--Init Rx Buffer
    UINT8 tmp_tx[1];
    
    //--Get actual character from device
    tmp_tx[0]= (UINT8)comm_LoRa.getc();
    
    //--Decode chunk of data
    SLIP_DecodeData(tmp_tx, 1);
}

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------