/**
********************************************************************************
* @file     SLIP.h
*
* @brief    Functions for NSL01 class for SLIP encoder/decoder.
* 
* @brief    For more information about the NSL01 LoRaWAN shield:
*           http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
*
* @note     The SLIP files are included in the dependencies directory of the
*           project and these files are necessary for the NSL01 class!
*
* @author   -
* @version  V1.0
* @date     20-June-2018
********************************************************************************
*/

#ifndef SLIP_H
#define SLIP_H

//------------------------------------------------------------------------------
//
// Include Files
//
//------------------------------------------------------------------------------

#include <stdint.h>

//------------------------------------------------------------------------------
//
// Section Defines
//
//------------------------------------------------------------------------------

//--SLIP protocol characters
#define SLIP_END                    0xC0
#define SLIP_ESC                    0xDB
#define SLIP_ESC_END                0xDC
#define SLIP_ESC_ESC                0xDD

//------------------------------------------------------------------------------
//
//  Section Declarations
//
//------------------------------------------------------------------------------

typedef uint8_t     UINT8;

//------------------------------------------------------------------------------
//
//  Function Prototypes
//
//------------------------------------------------------------------------------


/** SLIP message receiver callback defintion.
*
*   @param message  : Pointer to message
*   @param length   : Length of message
*/
typedef UINT8*  (*TSLIP_CbRxMessage)(UINT8* message, int length);


/** Function to init SLIP decoder.
*
*   @param cbRxMessage : Message receiver callback
*/
void
SLIP_Init(TSLIP_CbRxMessage cbRxMessage);


/** Function to init Rx buffer and enable receiver/decoder.
*
*   @param rxBuffer     : Pointer to Rx message buffer
*   @param rxBufferSize : Rx message buffer size
*
*   @returns true on success, false on error
*/
bool
SLIP_SetRxBuffer(UINT8* rxBuffer, int rxBufferSize);


/** Function to encode outgoing data.
*
*   @param dstBuffer    : Pointer to destination buffer
*   @param txBufferSize : Tx message buffer size
*   @param srcData      : Pointer to source data
*   @param srcLength    : Length of source data
*
*   @returns >=0 on success, -1 on error
*/
int
SLIP_EncodeData(UINT8* dstBuffer, int txBufferSize, UINT8* srcData, int srcLength);


/** Function to process/decode received byte stream.
*
*   @param srcData      : Pointer to source data
*   @param srcLength    : Length of source data
*/
void
SLIP_DecodeData(UINT8*  srcData, int srcLength);

#endif  // SLIP_H

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------