/*******************************************************************************
 *  Functions for NSL01 class for SLIP encoder/decoder.
 *
 *  For more information about the NSL01 LoRaWAN shield:
 *      http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
 *
 *  @note The SLIP files are included in the dependencies directory of the
 *        project and these files are necessary for the NSL01 class!
 *
 *  @author  -
 *  @version 1.0
 *  @date    20-June-2018
*******************************************************************************/

//------------------------------------------------------------------------------
//
//  Include Files
//
//------------------------------------------------------------------------------

#include "./dependencies/SLIP.h"

//------------------------------------------------------------------------------
//
//  Protocol definitions
//
//------------------------------------------------------------------------------

//--SLIP receiver/decoder states
#define SLIPDEC_IDLE_STATE          0
#define SLIPDEC_START_STATE         1
#define SLIPDEC_IN_FRAME_STATE      2
#define SLIPDEC_ESC_STATE           3

//------------------------------------------------------------------------------
//
// Declaration of SLIP variables
//
//------------------------------------------------------------------------------

typedef struct
{
    //--Decoder
    int                 RxState;
    int                 RxIndex;
    int                 RxBufferSize;
    UINT8*              RxBuffer;
    TSLIP_CbRxMessage   CbRxMessage;

    //--Encoder
    int                 TxIndex;
    int                 TxBufferSize;
    UINT8*              TxBuffer;
}TSLIP;

//------------------------------------------------------------------------------
//
// Section RAM
//
//------------------------------------------------------------------------------

//--SLIP Instance
static TSLIP   SLIP;

//------------------------------------------------------------------------------
//
//  Function: SLIP_Init
//
//  @brief Function to init SLIP decoder.
//
//  @param cbRxMessage : Message receiver callback
//
//------------------------------------------------------------------------------

void
SLIP_Init(TSLIP_CbRxMessage cbRxMessage)
{
    //--Init decoder to idle state, no Rx-buffer avaliable
    SLIP.RxState         =   SLIPDEC_IDLE_STATE;
    SLIP.RxIndex         =   0;
    SLIP.RxBuffer        =   0;
    SLIP.RxBufferSize    =   0;

    //--Save message receiver callback
    SLIP.CbRxMessage     =   cbRxMessage;

    //--Init encoder
    SLIP.TxIndex         =   0;
    SLIP.TxBuffer        =   0;
    SLIP.TxBufferSize    =   0;
}

//------------------------------------------------------------------------------
//
//  Function: SLIP_StoreTxByte
//
//  @brief Internal function to store a byte into TxBuffer.
//
//  @param txByte : Tx Byte
//
//------------------------------------------------------------------------------

static void
SLIP_StoreTxByte(UINT8 txByte)
{
    if (SLIP.TxIndex < SLIP.TxBufferSize)
        SLIP.TxBuffer[SLIP.TxIndex++] = txByte;
}

//------------------------------------------------------------------------------
//
//  Function: SLIP_EncodeData
//
//  @brief Function to encode outgoing data.
//
//  @param dstBuffer    : Pointer to destination buffer
//  @param dstBufferSize: Tx message buffer size
//  @param srcData      : Pointer to source data
//  @param srcLength    : Length of source data
//
//  @returns >=0 on success, -1 on error
//
//------------------------------------------------------------------------------

int
SLIP_EncodeData(UINT8* dstBuffer, int dstBufferSize, UINT8* srcData, int srcLength)
{
    //--Init TxBuffer
    SLIP.TxBuffer = dstBuffer;

    //--Init TxIndex
    SLIP.TxIndex  = 0;

    //--Init size
    SLIP.TxBufferSize = dstBufferSize;

    //--Send start of SLIP message
    SLIP_StoreTxByte(SLIP_END);

    //--Iterate over all message bytes
    while(srcLength--)
    {
        switch (*srcData)
        {
                case SLIP_END:
                    SLIP_StoreTxByte(SLIP_ESC);
                    SLIP_StoreTxByte(SLIP_ESC_END);
                    break;

                case SLIP_ESC:
                    SLIP_StoreTxByte(SLIP_ESC);
                    SLIP_StoreTxByte(SLIP_ESC_ESC);
                    break;

                default:
                    SLIP_StoreTxByte(*srcData);
                    break;
        }
        
        //--Update pointer: next byte
        srcData++;
    }

    //--Send end of SLIP message
    SLIP_StoreTxByte(SLIP_END);

    //--Check if length is okay
    if (SLIP.TxIndex <= SLIP.TxBufferSize)
        return SLIP.TxIndex;

    //--Return Tx length error
    return -1;
}

//------------------------------------------------------------------------------
//
//  Function: SLIP_SetRxBuffer
//
//  @brief Function to init Rx buffer and enable receiver/decoder.
//
//  @param rxBuffer     : Pointer to Rx message buffer
//  @param rxBufferSize : Rx message buffer size
//
//  @returns true on success, false on error
//
//------------------------------------------------------------------------------

bool
SLIP_SetRxBuffer(UINT8* rxBuffer, int rxBufferSize)
{
    //--Receiver in IDLE state ==> Check if client already registered
    if ((SLIP.RxState == SLIPDEC_IDLE_STATE) && SLIP.CbRxMessage)
    {
        //--Same buffer parameters
        SLIP.RxBuffer        = rxBuffer;
        SLIP.RxBufferSize    = rxBufferSize;

        //--Enable decoder
        SLIP.RxState = SLIPDEC_START_STATE;

        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//
//  Function: SLIP_StoreRxByte
//
//  @brief Internal function to store SLIP decoded Rx byte.
//
//  @param rxBuffer     : Rx Byte
//
//------------------------------------------------------------------------------

static void
SLIP_StoreRxByte(UINT8 rxByte)
{
    if (SLIP.RxIndex < SLIP.RxBufferSize)
        SLIP.RxBuffer[SLIP.RxIndex++] = rxByte;
}

//------------------------------------------------------------------------------
//
//  Function: SLIP_DecodeData
//
//  @brief Function to process/decode received byte stream.
//
//  @param srcData      : Pointer to source data
//  @param srcLength    : Length of source data
//
//------------------------------------------------------------------------------

void
SLIP_DecodeData(UINT8* srcData, int srcLength)
{
    //--Iterate over all received bytes
    while(srcLength--)
    {
        //--Get Rx byte
        UINT8 rxByte = *srcData++;

        //--Decode according to current state
        switch(SLIP.RxState)
        {
            case    SLIPDEC_START_STATE:                                       
                    //--Start of SLIP frame?
                    if(rxByte == SLIP_END)
                    {                        
                        //--Init read index
                        SLIP.RxIndex = 0;

                        //--Next state
                        SLIP.RxState = SLIPDEC_IN_FRAME_STATE;
                    }
                    break;

            case    SLIPDEC_IN_FRAME_STATE:
                    switch(rxByte)
                    {
                        case    SLIP_END:
                                //--Data received?
                                if(SLIP.RxIndex > 0)
                                {
                                    //--Yes, receiver registered?
                                    if (SLIP.CbRxMessage)
                                    {
                                        //--Yes, call message receive
                                        SLIP.RxBuffer = (*SLIP.CbRxMessage)(SLIP.RxBuffer, SLIP.RxIndex);

                                        //--New buffer available?
                                        if (!SLIP.RxBuffer)
                                        {
                                            SLIP.RxState = SLIPDEC_IDLE_STATE;
                                        }
                                        else
                                        {
                                            SLIP.RxState = SLIPDEC_START_STATE;
                                        }
                                    }
                                    else
                                    {
                                        //--Disable decoder, temp. no buffer avaliable
                                        SLIP.RxState = SLIPDEC_IDLE_STATE;
                                    }
                                }
                                //--Init read index
                                SLIP.RxIndex = 0;
                                break;

                        case  SLIP_ESC:
                                //--Enter escape sequence state
                                SLIP.RxState = SLIPDEC_ESC_STATE;
                                break;

                        default:
                                //--Store byte
                                SLIP_StoreRxByte(rxByte);
                                break;
                    }
                    break;

            case    SLIPDEC_ESC_STATE:
                    switch(rxByte)
                    {
                        case    SLIP_ESC_END:
                                SLIP_StoreRxByte(SLIP_END);
                                //--Quit escape sequence state
                                SLIP.RxState = SLIPDEC_IN_FRAME_STATE;
                                break;

                        case    SLIP_ESC_ESC:
                                SLIP_StoreRxByte(SLIP_ESC);
                                //--Quit escape sequence state
                                SLIP.RxState = SLIPDEC_IN_FRAME_STATE;
                                break;

                        default:
                                //--Abort frame receiption
                                SLIP.RxState = SLIPDEC_START_STATE;
                                break;
                    }
                    break;

            default:
                    break;
        }
    }
}

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------