/*******************************************************************************
 *  Functions for NSL01 class to create a Human Control Interface (HCI) for
 *  controlling the onboard RF radio module.
 *
 *  For more information about the NSL01 LoRaWAN shield:
 *      http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
 *
 *  @note The NSL01_HCI_Layer files are included in the dependencies directory
 *        of the project and these files are necessary for the NSL01 class!
 *
 *  @author  -
 *  @version 1.0
 *  @date    20-June-2018
*******************************************************************************/

//------------------------------------------------------------------------------
//
// Include Files
//
//------------------------------------------------------------------------------

#include "./dependencies/NSL01_HCI_Layer.h"
#include <string.h>

//------------------------------------------------------------------------------
//
//  Forward Declaration
//
//------------------------------------------------------------------------------

//--SLIP Message Receiver Callback
static UINT8*   WiMOD_HCI_ProcessRxMessage(UINT8* rxData, int rxLength);

//------------------------------------------------------------------------------
//
// Declare Layer Instance
//
//------------------------------------------------------------------------------

typedef struct
{
    //--CRC Error counter
    UINT32                  CRCErrors;

    //--RxMessage
    TWiMOD_HCI_Message*     RxMessage;

    //--Receiver callback
    TWiMOD_HCI_CbRxMessage  CbRxMessage;

}TWiMOD_HCI_MsgLayer;

//------------------------------------------------------------------------------
//
//  Section RAM
//
//------------------------------------------------------------------------------

//--Reserve HCI Instance
static TWiMOD_HCI_MsgLayer  HCI;

//--Reserve one TxBuffer
static UINT8                TxBuffer[sizeof( TWiMOD_HCI_Message ) * 2 + 2];

//------------------------------------------------------------------------------
//
//  Function: WiMOD_HCI_Init
//
//  @brief Function to initialize Human Control Interface (HCI) message layer.
//
//  @param cbRxMessage : Function pointer
//  @param rxMessage   : Pointer to HCI Rx message struct
//
//------------------------------------------------------------------------------

void
WiMOD_HCI_Init(TWiMOD_HCI_CbRxMessage   cbRxMessage,    //-HCI msg recv. callback
               TWiMOD_HCI_Message*      rxMessage)      //-Initial rxMessage
{
    //--Init error counter
    HCI.CRCErrors   =   0;

    //--Save receiver callback
    HCI.CbRxMessage =   cbRxMessage;

    //--Save RxMessage
    HCI.RxMessage   =   rxMessage;

    //--Init SLIP
    SLIP_Init(WiMOD_HCI_ProcessRxMessage);

    //--Init first RxBuffer to SAP_ID of HCI message, size without 16-Bit length field
    SLIP_SetRxBuffer(&rxMessage->SapID, sizeof(TWiMOD_HCI_Message) - sizeof(UINT16));

    //--Init serial (UART) device
    SerialDevice_Open(Baudrate_115200, DataBits_8);
}

//------------------------------------------------------------------------------
//
//  Function: WiMOD_HCI_SendMessage
//
//  @brief Function to transmit a message via specified Human Control Interface
//         (HCI) with or without payload.
//
//  @param txMessage : Pointer to HCI Tx message struct
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int 
WiMOD_HCI_SendMessage(TWiMOD_HCI_Message* txMessage)
{
    //-------------------------------------------------------------
    // 1. Check parameter
    //-------------------------------------------------------------

    // 1.1 Check ptr
    if (!txMessage)
    {
        //--Error
        return -1;
    }

    //-------------------------------------------------------------
    // 2. Calculate CRC16 over header and optional payload data
    //-------------------------------------------------------------

    UINT16 crc16 = CRC16_Calc(&txMessage->SapID,
                              txMessage->Length + WIMOD_HCI_MSG_HEADER_SIZE,
                              CRC16_INIT_VALUE);

    // 2.1 Get first complement
    crc16 = ~crc16;

    // 2.2 Attach CRC16 and correct length, LSB first
    txMessage->Payload[txMessage->Length]     = LOBYTE(crc16);
    txMessage->Payload[txMessage->Length + 1] = HIBYTE(crc16);

    //-------------------------------------------------------------
    // 3. Perform SLIP encoding:
    //    - start transmission with SAP_ID
    //    - correct length by header size
    //-------------------------------------------------------------

    int txLength = SLIP_EncodeData(TxBuffer,
                                   sizeof(TxBuffer),
                                   &txMessage->SapID,
                                   txMessage->Length + WIMOD_HCI_MSG_HEADER_SIZE + WIMOD_HCI_MSG_FCS_SIZE);

    //--Initialize return value
    int return_value;

    //--Check if message is okay
    if (txLength > 0)
    {
        //--Transmit wakeup chars
        for(int i= 0; i < 40; i++)
            SerialDevice_SendByte(SLIP_END);

        //-------------------------------------------------------------
        // 4. Transmit octet sequence over serial device
        //-------------------------------------------------------------
        return_value = SerialDevice_SendData(TxBuffer, txLength);

        #ifdef DEBUG
            comm_pc.printf("\r\n Return value SerialDevice_SendData: %i\n\r", return_value);
        #endif

        //--Evaluate result
        if ( return_value > 0 )
        {
            //--Return ok
            return 1;
        }
    }

    // Error - SLIP layer couldn't encode message - buffer to small ?
    return -1;
}

//------------------------------------------------------------------------------
//
//  Function: WiMOD_HCI_Process
//
//  @brief Function to read/process incoming serial data, which is a user
//         triggered function without using interrupts.
//
//  @note This function is not necessary at the moment, because the incoming
//        serial data is processed via an Interrupt Service Routine (ISR).
//
//------------------------------------------------------------------------------

void
WiMOD_HCI_Process()
{
    UINT8   rxBuf[20];

    //--Read small chunk of data
    int rxLength = SerialDevice_ReadData(rxBuf, sizeof(rxBuf));

    //--Check if data is available
    if (rxLength > 0)
    {
        //--Yes, forward to SLIP decoder, decoded SLIP message will be passed to
        //--function "WiMOD_HCI_ProcessRxMessage"
        SLIP_DecodeData(rxBuf, rxLength);
    }
}

//------------------------------------------------------------------------------
//
//  Function: WiMOD_HCI_ProcessRxMessage
//
//  @brief Internal function to process received SLIP message and return new
//         Rx buffer.
//
//  @param rxData   : Pointer to Rx message
//  @param rxLength : Length of Rx message
//
//  @returns pointer to first byte of Rx message
//
//------------------------------------------------------------------------------

static UINT8*
WiMOD_HCI_ProcessRxMessage(UINT8* rxData, int rxLength)
{
    //--Check min length
    if (rxLength >= (WIMOD_HCI_MSG_HEADER_SIZE + WIMOD_HCI_MSG_FCS_SIZE))
    {
        if (CRC16_Check(rxData, rxLength, CRC16_INIT_VALUE))
        {
            //--Check if receiver is registered
            if (HCI.CbRxMessage)
            {
                //--Yes, complete message info
                HCI.RxMessage->Length = rxLength - (WIMOD_HCI_MSG_HEADER_SIZE + WIMOD_HCI_MSG_FCS_SIZE);

                //--Call upper layer receiver and save new RxMessage
                HCI.RxMessage = (*HCI.CbRxMessage)(HCI.RxMessage);                
            }
        }
        else
        {
            HCI.CRCErrors++;
        }
    }

    //--Check if free HCI message is available
    if (HCI.RxMessage)
    {
        //--Yes, return pointer to first byte
        return &HCI.RxMessage->SapID;
    }

    //--Error, disable SLIP decoder
    return 0;
}

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------