/*
********************************************************************************
* @file     NSL01.h
*
* @brief    Class for controlling the NSL01 LoRaWAN shield from
*           mCloud Systems GmbH.
* 
* @brief    For more information about the NSL01 LoRaWAN shield:
*           http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
*
* @author   mCloud Systems Team 
*  
* @version  V1.0
* @date     20-June-2018
********************************************************************************
* @attention  
*
*  Copyright (c) 2018 mCloud Systems GmbH, MIT License
*
*  Permission is hereby granted, free of charge, to any person obtaining a copy 
*  of this software and associated documentation files (the "Software"), to deal
*  in the Software without restriction, including without limitation the rights 
*  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
*  copies of the Software, and to permit persons to whom the Software is
*  furnished to do so, subject to the following conditions:
*
*  The above copyright notice and this permission notice shall be included in
*  all copies or substantial portions of the Software.
*
*  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
*  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
*  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
*  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
*  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
*  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
*  SOFTWARE.
*/

#ifndef NSL01_h
#define NSL01_h

//#define DEBUG
 
//------------------------------------------------------------------------------
//
//  Include Files
//
//------------------------------------------------------------------------------ 

#include "mbed.h"
#include "./dependencies/NSL01_HCI_Layer.h"
#include <string.h>
#include <stdint.h>

//------------------------------------------------------------------------------
//
//  Section Defines
//
//------------------------------------------------------------------------------

#define DEVICE_ADDR_LEN      4
#define KEY_LEN             16
#define EUI_LEN              8

//------------------------------------------------------------------------------
//
//  General Declarations
//
//------------------------------------------------------------------------------

//--Global variables from main.h
extern Serial comm_pc;
extern Serial comm_LoRa;

//--Define shortcuts for datatypes
typedef int8_t      INT8;
typedef uint8_t     UINT8;
typedef uint16_t    UINT16;
typedef uint32_t    UINT32;

//--Struct for Real Time Clock (RTC) information
typedef struct
{
    UINT16  year;
    UINT8   month;
    UINT8   day;
    UINT8   hour;
    UINT8   minute;
    UINT8   second;      
}RTC_struct;

//--Struct for Real Time Clock (RTC) alarm
typedef struct
{
    bool    periodic;
    UINT8   hour;
    UINT8   minute;
    UINT8   second;           
}RTC_alarm_struct;

//--Struct for radio stack configuration
typedef struct
{
    UINT8   data_rate_index;
    UINT8   Tx_power;
    UINT8   adaptive_data_rate;
    UINT8   duty_cycle_control;
    UINT8   class_C_support;
    UINT8   RF_packet_format;
    UINT8   forwarding;
    UINT8   power_saving;
    UINT8   number_retransmissions;
    UINT8   band_index;
    UINT8   header_MAC_cmd_capacity;       
}Radio_config_struct;

//--Helper struct for ID -> string conversion
typedef struct
{
    int         ID;
    const char* String;
}TIDString;

//------------------------------------------------------------------------------
//
//  Endpoint (SAP) Identifier
//
//------------------------------------------------------------------------------

#define DEVMGMT_SAP_ID                              0x01
#define LORAWAN_SAP_ID                              0x10

//------------------------------------------------------------------------------
//
//  Device Management SAP Message Identifier
//
//------------------------------------------------------------------------------

//--Status Identifier
#define DEVMGMT_STATUS_OK                           0x00
#define DEVMGMT_STATUS_ERROR                        0x01
#define DEVMGMT_STATUS_CMD_NOT_SUPPORTED            0x02
#define DEVMGMT_STATUS_WRONG_PARAMETER              0x03
#define DEVMGMT_STATUS_WRONG_DEVICE_MODE            0x04

//--Message Identifier
#define DEVMGMT_MSG_PING_REQ                        0x01    //-Identifier for ping LoRa module
#define DEVMGMT_MSG_PING_RSP                        0x02

#define DEVMGMT_MSG_GET_DEVICE_INFO_REQ             0x03    //-Identifier for getting device infos
#define DEVMGMT_MSG_GET_DEVICE_INFO_RSP             0x04

#define DEVMGMT_MSG_GET_FW_VERSION_REQ              0x05    //-Identifier for getting firmware infos
#define DEVMGMT_MSG_GET_FW_VERSION_RSP              0x06

#define DEVMGMT_MSG_RESET_REQ                       0x07    //--Software reset
#define DEVMGMT_MSG_RESET_RSP                       0x08

#define DEVMGMT_MSG_SET_OPMODE_REQ                  0x09    //-Identifier for system operating mode
#define DEVMGMT_MSG_SET_OPMODE_RSP                  0x0A
#define DEVMGMT_MSG_GET_OPMODE_REQ                  0x0B
#define DEVMGMT_MSG_GET_OPMODE_RSP                  0x0C

#define DEVMGMT_MSG_SET_RTC_REQ                     0x0D    //-Identifier for RTC
#define DEVMGMT_MSG_SET_RTC_RSP                     0x0E
#define DEVMGMT_MSG_GET_RTC_REQ                     0x0F
#define DEVMGMT_MSG_GET_RTC_RSP                     0x10

#define DEVMGMT_MSG_GET_DEVICE_STATUS_REQ           0x17    //-Identifier for getting device status
#define DEVMGMT_MSG_GET_DEVICE_STATUS_RSP           0x18

#define DEVMGMT_MSG_SET_RTC_ALARM_REQ               0x31    //--Identifier for RTC alarm
#define DEVMGMT_MSG_SET_RTC_ALARM_RSP               0x32
#define DEVMGMT_MSG_CLEAR_RTC_ALARM_REQ             0x33
#define DEVMGMT_MSG_CLEAR_RTC_ALARM_RSP             0x34
#define DEVMGMT_MSG_GET_RTC_ALARM_REQ               0x35
#define DEVMGMT_MSG_GET_RTC_ALARM_RSP               0x36
#define DEVMGMT_MSG_RTC_ALARM_IND                   0x38

//------------------------------------------------------------------------------
//
//  LoRaWAN SAP Message Identifier
//
//------------------------------------------------------------------------------

//--Status Identifier
#define LORAWAN_STATUS_OK                           0x00
#define LORAWAN_STATUS_ERROR                        0x01
#define LORAWAN_STATUS_CMD_NOT_SUPPORTED            0x02
#define LORAWAN_STATUS_WRONG_PARAMETER              0x03
#define LORAWAN_STATUS_WRONG_DEVICE_MODE            0x04
#define LORAWAN_STATUS_NOT_ACTIVATED                0x05
#define LORAWAN_STATUS_BUSY                         0x06
#define LORAWAN_STATUS_QUEUE_FULL                   0x07
#define LORAWAN_STATUS_LENGTH_ERROR                 0x08
#define LORAWAN_STATUS_NO_FACTORY_SETTINGS          0x09
#define LORAWAN_STATUS_CHANNEL_BLOCKED_BY_DC        0x0A
#define LORAWAN_STATUS_CHANNEL_NOT_AVAILABLE        0x0B

//--Message Identifier
#define LORAWAN_MSG_ACTIVATE_DEVICE_REQ             0x01    //--Identifier for ABP activation/reactivation and deactivation
#define LORAWAN_MSG_ACTIVATE_DEVICE_RSP             0x02
#define LORAWAN_MSG_REACTIVATE_DEVICE_REQ           0x1D
#define LORAWAN_MSG_REACTIVATE_DEVICE_RSP           0x1E
#define LORAWAN_MSG_DEACTIVATE_DEVICE_REQ           0x21
#define LORAWAN_MSG_DEACTIVATE_DEVICE_RSP           0x22

#define LORAWAN_MSG_SET_JOIN_PARAM_REQ              0x05    //--Identifier for joining in OTA mode
#define LORAWAN_MSG_SET_JOIN_PARAM_RSP              0x06
#define LORAWAN_MSG_JOIN_NETWORK_REQ                0x09
#define LORAWAN_MSG_JOIN_NETWORK_RSP                0x0A
#define LORAWAN_MSG_JOIN_TRANSMIT_IND               0x0B
#define LORAWAN_MSG_JOIN_NETWORK_IND                0x0C

#define LORAWAN_MSG_SEND_UDATA_REQ                  0x0D    //--Identifier for U-data
#define LORAWAN_MSG_SEND_UDATA_RSP                  0x0E
#define LORAWAN_MSG_SEND_UDATA_TX_IND               0x0F
#define LORAWAN_MSG_RECV_UDATA_RX_IND               0x10

#define LORAWAN_MSG_SEND_CDATA_REQ                  0x11    //--Identifier for C-data
#define LORAWAN_MSG_SEND_CDATA_RSP                  0x12
#define LORAWAN_MSG_SEND_CDATA_TX_IND               0x13
#define LORAWAN_MSG_RECV_CDATA_RX_IND               0x14

#define LORAWAN_MSG_RECV_ACK_IND                    0x15    //--Identifier for ACK receiption
#define LORAWAN_MSG_RECV_NODATA_IND                 0x16

#define LORAWAN_MSG_SET_RSTACK_CONFIG_REQ           0x19    //--Identifier for radio stack configuration
#define LORAWAN_MSG_SET_RSTACK_CONFIG_RSP           0x1A
#define LORAWAN_MSG_GET_RSTACK_CONFIG_REQ           0x1B
#define LORAWAN_MSG_GET_RSTACK_CONFIG_RSP           0x1C

#define LORAWAN_MSG_FACTORY_RESET_REQ               0x23    //--Apply factory reset
#define LORAWAN_MSG_FACTORY_RESET_RSP               0x24

#define LORAWAN_MSG_SET_DEVICE_EUI_REQ              0x25    //--Identifier for EUI
#define LORAWAN_MSG_SET_DEVICE_EUI_RSP              0x26
#define LORAWAN_MSG_GET_DEVICE_EUI_REQ              0x27
#define LORAWAN_MSG_GET_DEVICE_EUI_RSP              0x28

#define LORAWAN_MSG_GET_NWK_STATUS_REQ              0x29    //--Identifier for network status
#define LORAWAN_MSG_GET_NWK_STATUS_RSP              0x2A

#define LORAWAN_MSG_SET_CUSTOM_CFG_REQ              0x31    //--Identifier for custom configuration
#define LORAWAN_MSG_SET_CUSTOM_CFG_RSP              0x32
#define LORAWAN_MSG_GET_CUSTOM_CFG_REQ              0x33
#define LORAWAN_MSG_GET_CUSTOM_CFG_RSP              0x34

#define LORAWAN_MSG_GET_SUPPORTED_BANDS_REQ         0x35    //--Identifier for supported frequency bands
#define LORAWAN_MSG_GET_SUPPORTED_BANDS_RSP         0x36

#define LORAWAN_MSG_SET_TXPOWER_LIMIT_CONFIG_REQ    0x37    //--Identifier for Tx power limit configuration
#define LORAWAN_MSG_SET_TXPOWER_LIMIT_CONFIG_RSP    0x38
#define LORAWAN_MSG_GET_TXPOWER_LIMIT_CONFIG_REQ    0x39
#define LORAWAN_MSG_GET_TXPOWER_LIMIT_CONFIG_RSP    0x3A

//------------------------------------------------------------------------------
//
//  LoRaWAN Event Message Identifier
//
//------------------------------------------------------------------------------

#define LORAWAN_EVENT_MSG_ALARM                     0x01    //--RTC alarm indication event
#define LORAWAN_EVENT_MSG_JOIN_TIMEOUT              0x02    //--Join network indication timeout
#define LORAWAN_EVENT_MSG_CDATA_MAX_RETRANSMISSION  0x03    //--Maximum number of retransmissions reached
#define LORAWAN_EVENT_MSG_CDATA_PAYLOAD_EXCEEDED    0x04    //--Maximum payload size exceeded for current data rate


/** NSL01 class: A library which controls the NSL01 LoRaWAN shield from 
 *               mCloud Systems GmbH.
 *
 *  @brief The NSL01 is a professional plug & play LoRaWAN shield for a wide
 *         range of STM32 Nucleo-64&tm; boards with Arduino&tm; Uno Rev3
 *         connectivity.
 *
 *  @brief  For more information about the NSL01 LoRaWAN shield:
*             http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
 *
 *  @note Two serial device objects have to be specified in main:
 *      - Serial comm_pc(SERIAL_TX, SERIAL_RX)
 *      - Serial comm_LoRa(PB_6,PA_10) => TX and RX pins depending on STM32
 *        Nucleo-64 board! => Please check datasheet of NSL01!
 *
 *  @note The dependencies directory contains several C-driver functions which
 *        have to be included in the project! The NSL01 class is based on these
 *        drivers!
 *
 *  @author     mCloud Systems Team
 *  @version    V1.0
 *  @date       20-June-2018
 *
 *  Copyright (c) 2018 mCloud Systems GmbH, MIT License
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy 
 *  of this software and associated documentation files (the "Software"), to
 *  deal in the Software without restriction, including without limitation the
 *  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 *  sell copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 *
 *   Example:
 *   @code
 *   #include "mbed.h"
 *   #include "./NSL01/NSL01.h"
 *
 *   //--UART interface for onboard LoRaWAN radio module
 *   //--(Tx and Rx pin depending on STM32 Nucleo-64 board)
 *   PinName Pin_UART_Tx = PB_6;
 *   PinName Pin_UART_Rx = PA_10;
 *   Serial comm_LoRa(Pin_UART_Tx, Pin_UART_Rx);
 *
 *   //--Standard serial interface (PC) for (internal) debugging
 *   Serial comm_pc(SERIAL_TX, SERIAL_RX);
 * 
 *   int main() 
 *   {
 *      //--Create NSL01 object and initialize device
 *      NSL01 LoRa;
 *      
 *      //--Apply methods, e.g. ping NSL01 device
 *      LoRa.ping();
 *
 *      //--Insert your specific code here
 *
 *   }
 *   @endcode
 */

class NSL01
{
public:     


    /** Consructor of NSL01: Create a NSL01 instance with the serial device
    *   objects specified in main.
    */  
    NSL01(void);


    /** Destructor of NSL01: Destroy instance.
    *
    */
    ~NSL01();


    /** Member variable for Real Time Clock (RTC) information based on global
    *   defined RTC_struct which contains:
    *   - UINT16    year
    *   - UINT8     month
    *   - UINT8     day
    *   - UINT8     hour
    *   - UINT8     minute
    *   - UINT8     second
    */
    RTC_struct RTC_data; 


    /** Member variable for Real Time Clock (RTC) alarm based on global
    *   defined RTC_alarm_struct which contains:
    *   - bool      periodic
    *   - UINT8     hour
    *   - UINT8     minute
    *   - UINT8     second
    */    
    RTC_alarm_struct RTC_alarm;


    /** Member variable for radio stack configuration based on global defined
    *   Radio_config_struct which contains:
    *   - UINT8   data_rate_index;
    *   - UINT8   Tx_power;
    *   - UINT8   adaptive_data_rate;
    *   - UINT8   duty_cycle_control;
    *   - UINT8   class_C_support;
    *   - UINT8   RF_packet_format;
    *   - UINT8   forwarding;
    *   - UINT8   power_saving;
    *   - UINT8   number_retransmissions;
    *   - UINT8   band_index;
    *   - UINT8   header_MAC_cmd_capacity;  
    */    
    Radio_config_struct Radio_config;


    /** Method to ping NSL01.
    *
    *   @returns 1 on success, -1 on error
    */    
    int ping(void);


    /** Method to reset NSL01 (software reset).
    *
    *   @note After executing this command the module will be restarted after
    *   approximately 200ms!
    *
    *   @returns 1 on success, -1 on error
    */    
    int reset(void);


    /** Method to identify the local connected NSL01 device. As a result the
    *   device sends a response message which contains all necessary device
    *   information.
    *
    *   @returns 1 on success, -1 on error
    */    
    int get_device_info(void);


    /** Method to identify the onboard radio firmware of NSL01. As a result the 
    *   device sends a response message which contains all firmware related 
    *   data.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_firmware_info(void);


    /** Method to get the current device status of NSL01. As a result the device
    *   sends a response message which contains all device status information.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_device_status(void);


    /** Method to read the current network / activation status of NSL01.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_network_status(void);


    /** Method to read the current Real Time Clock (RTC) value of NSL01.
    *
    *   @returns 1 on success, -1 on error
    */    
    int get_RTC_info(void);


    /** Method to set the Real Time Clock (RTC) value of NSL01.
    *
    *   @param RTC_data : RTC data defined in RTC_struct
    *
    *   @returns 1 on success, -1 on error
    */
    int set_RTC(RTC_struct RTC_data);


    /** Method to set the Real Time Clock (RTC) alarm of NSL01.
    *
    *   @param RTC_alarm_data : RTC alarm defined in RTC_alarm_struct
    *
    *   @returns 1 on success, -1 on error
    */    
    int set_RTC_alarm(RTC_alarm_struct RTC_alarm_data);


    /** Method to get the Real Time Clock (RTC) alarm info of NSL01.
    *
    *   @returns 1 on success, -1 on error
    */    
    int get_RTC_alarm(void);


    /** Method to clear a pending Real Time Clock (RTC) alarm of NSL01.
    *
    *   @returns 1 on success, -1 on error
    */    
    int clear_RTC_alarm(void);


    /** Method to get the current system operation mode of NSL01.
    *
    *   @note Two system operation modes are supported:
    *   - Standard / Application Mode
    *   - Customer Mode
    *
    *   @note Some device functionality is only supported in customer mode!
    *
    *   @returns 1 on success, -1 on error
    */    
    int get_system_operation_mode(void);


    /** Method to set the system operation mode of NSL01.
    *
    *   @note After executing this command the module will be restarted after
    *   approximately 200ms!
    *
    *   @param system_mode 0 : Standard/Application mode
    *   @param system_mode 1 : Reserved/Not used
    *   @param system_mode 2 : Reserved/Not used
    *   @param system_mode 3 : Customer mode
    *
    *   @returns 1 on success, -1 on error
    */
    int set_system_operation_mode(UINT8 system_mode);


    /** Method to activate NSL01 in Activation by Personalization (ABP) mode
    *   and save parameters in non-volatile memory.
    *
    *   @note A device must be activated prior to any further data exchange with
    *   a network server!
    *
    *   @note After a successful activation the NSL01 will automatically send an
    *   empty unconfirmed uplink message ("alive message") over the air!
    *
    *   @param deviceAddress  : unique 32-Bit Device Address
    *   @param *nwkSessionKey : device specific 128-Bit Network Session Key
    *   @param *appSessionKey : device specific 128-Bit Application Session Key
    *
    *   @returns 1 on success, -1 on error
    */
    int ABP_device_activation(UINT32 deviceAddress,
                              UINT8* nwkSessionKey,
                              UINT8* appSessionKey);


    /** Method to reactivate NSL01 in Activation by Personalization (ABP) mode
    *   using the parameters previously stored in non-volatile memory.
    *
    *   @note A device must be activated prior to any further data exchange with
    *   a network server!
    *
    *   @note After a successful activation the NSL01 will automatically send an
    *   empty unconfirmed uplink message ("alive message") over the air!
    *
    *   @returns 1 on success, -1 on error
    */
    int ABP_device_reactivation(void);


    /** Method to deactivate the NSL01 LoRaWAN end-device => further data
    *   exchange over the air will be disabled.
    *
    *   @returns 1 on success, -1 on error
    */
    int deactivate(void);


    /** Method to configure the Over-the-Air Activation (OTTA) parameters of
    *   NSL01 which are used during the join network procedure 'join_network'.
    *
    *   @param *appEUI : globally unique 64-Bit Application EUI
    *   @param *appKey : device specific 128-Bit AES Application Key
    *
    *   @returns 1 on success, -1 on error
    */
    int set_join_parameter(UINT8* appEUI, UINT8* appKey); 


    /** Method to start the join network procedure of NSL01 in Over-the-Air
    *   Activation (OTTA) mode with previously defined parameters in 
    *   'set_join_parameter' method.
    *
    *   @note Once the 'join_network' method is triggered an internal event
    *   method is invoked periodically to indicate the joining procedure.
    *
    *   @note An internal join network indication event method is invoked after
    *   successful reception of a server join response packet or after the
    *   expiration of a complete join process without success (the join request
    *   will be retransmitted changing the spreading factor from SF7 till SF12,
    *   reusing each spreading factor twice).
    *
    *   @returns 1 on success, -1 on error
    */        
    int join_network(void);


    /** Method to transmit data in an unreliable way to the network server <=>
    *   no acknowledgement will be sent from the network server and no
    *   retransmission method is available on the end-device (NSL01) side
    *
    *   @note An internal 'unreliable data transmit indication' event method is 
    *   invoked after the radio packet has been sent, containing information
    *   about the data transfer.
    *
    *   @param port      : LoRaWAN Port number (> 0)
    *   @param *srcData  : Application Payload (data which have to be transmitted)
    *   @param srcLength : Lenght of Application Payload data
    *
    *   @returns 1 on success, -1 on error
    */
    int send_Udata(UINT8 port, UINT8* srcData, int srcLength);


    /** Method to transmit data in a reliable way to the network server <=>
    *   confirmed data transmission.
    *
    *   @note An internal 'reliable data transmit indication' event method is 
    *   invoked after the radio packet has been sent, containing information
    *   about the data transfer.
    *
    *   @param port      : LoRaWAN Port number (> 0)
    *   @param *srcData  : Application Payload (data which have to be transmitted)
    *   @param srcLength : Lenght of Application Payload data
    *
    *   @returns 1 on success, -1 on error
    */
    int send_Cdata(UINT8 port, UINT8* srcData, int srcLength);


    /** Method to set/configure the internal radio stack configuration of NSL01
    *   <=> main parameters for data transmission.
    *
    *   @param radio_stack : Radio stack config data defined in
    *   Radio_config_struct
    *
    *   @returns 1 on success, -1 on error
    */ 
    int set_radio_stack_config(Radio_config_struct radio_stack);


    /** Method to read the current radio stack configuration of NSL01 <=> main
    *   parameters for data transmission.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_radio_stack_config(void);


    /** Method to get information related to the supported frequency bands used
    *   by the firmware. Moreover, the maximum supported EIRP for each band is
    *   provided.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_freqband_info(void);


    /** Method to get/read the unique 64-bit device EUI.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_EUI(void);


    /** Method to set/write the unique 64-bit device EUI.
    *
    *   @note The device EUI can only be set in "customer mode"!
    *
    *   @param *device_EUI : Unique 64-bit device EUI (UINT8 array)
    *
    *   @returns 1 on success, -1 on error
    */
    int set_EUI(UINT8 *device_EUI);


    /** Method to configure the custom parameters (RF gain offset to compensate
    *   possible transmission losses/gains including circuit matching,
    *   antennas, ...). The RF gain offset should be rated in units of dBd
    *   (decibels relative to a half-wavelength dipole antenna, where 
    *   0 dBd = 2.15 dBi).
    *
    *   @note This parameter can only be set in "customer mode"!
    *
    *   @param RF_gain_offset : RF gain offset in dBd
    *
    *   @returns 1 on success, -1 on error
    */
    int set_custom_configuration(int RF_gain_offset);


    /** Method to get/read the custom configuration parameters (RF gain offset)
    *   of NSL01.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_custom_configuration(void);


    /** Method to restore the initial firmware settings, stored during
    *   production time (factory reset).
    *
    *   @returns 1 on success, -1 on error
    */
    int factory_reset(void);


    /** Method to read the transmit power limit configuration parameters of
    *   NSL01 for different frequency sub-bands.
    *
    *   @returns 1 on success, -1 on error
    */
    int get_PowerLimit_configuration(void);


    /** Method to set/configure the transmit power limit configuration
    *   parameters of NSL01 for a dedicated frequency sub-band.
    *
    *   @note These parameters can only be set in "customer mode"!
    *
    *   @param sub_band : Sub-band index [0...5] according to specification
    *   @param power_limit_flag : Tx power limit flag for sub-band [enabled/disabled]
    *   @param power_limit : Tx power limit for sub-band in dBm [0...16]
    *
    *   @returns 1 on success, -1 on error
    */
    int set_PowerLimit_configuration(UINT8 sub_band, UINT8 power_limit_flag, UINT8 power_limit);


private:


    /** Internal method to initialize NSL01.
    *
    *   @note This method is automatically invoked in constructor.
    *
    */
    void initialize(void);


    /** Internal callback method for event handling of NSL01.
    *
    *   @note This method is invoked automatically whenever an event occurs.
    *
    *   @param Event_ID : Event message identifier defined in header of NSL01.h
    */    
    static void 
    LoRa_event(int Event_ID);


    /** Internal processing of received HCI messages (callback method).
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    *
    *   @returns TWiMOD_HCI_Message* struct
    */    
    static TWiMOD_HCI_Message* 
    Process_RxMessage(TWiMOD_HCI_Message*  rxMessage);


    /** Internal processing of received device management messages.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */        
    static void 
    Process_DevMgmt_Message(TWiMOD_HCI_Message*  rxMessage);    


    /** Internal processing of received LoRaWAN messages.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    Process_LoRaWAN_Message(TWiMOD_HCI_Message*  rxMessage);


    /** Internal method to show device information of NSL01.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */    
    static void
    DevMgmt_DeviceInfo_Rsp(TWiMOD_HCI_Message*  rxMessage);


    /** Internal method to show firmware information of NSL01.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    DevMgmt_FirmwareVersion_Rsp(TWiMOD_HCI_Message*  rxMessage);


    /** Internal method to show device status of NSL01.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */    
    static void
    DevMgmt_DeviceStatus_Rsp(TWiMOD_HCI_Message*  rxMessage);


    /** Internal method to show RTC information of NSL01.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    DevMgmt_RTC_Rsp(TWiMOD_HCI_Message*  rxMessage);


    /** Internal method to show RTC alarm information of NSL01.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    DevMgmt_Get_Alarm_Rsp(TWiMOD_HCI_Message*  rxMessage);


    /** Internal method to get the system operation mode of NSL01.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    DevMgmt_OperationMode_Rsp(TWiMOD_HCI_Message*  rxMessage);


    /** Internal method to get the status of NSL01 after reactivation in 
    *   Activation by Personalization (ABP) mode with parameters in non-
    *   volatile memory.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Reactivate_Device_Rsp(TWiMOD_HCI_Message*  rxMessage);


    /** Internal event method for join network transmit indication which is
    *   automatically invoked whenever the join radio message has been sent to
    *   the host.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */    
    static void
    LoRaWAN_Process_JoinTxIndication(TWiMOD_HCI_Message* rxMessage);  


    /** Internal event method to indicate the join network procedure which is
    *   automatically invoked after successful reception of a server join
    *   response or after expiration of a complete join process without success.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Process_JoinNetworkIndication(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'send_Udata' method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_send_Udata_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal event method which is triggered by the host after a radio
    *   packet has been sent.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void 
    LoRaWAN_Udata_Tx_Indication(TWiMOD_HCI_Message* rxMessage);


    /** Internal event method which is sent to the host after reception of an 
    *   unreliable radio packet containing application payload.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Udata_Rx_Indication(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'send_Cdata' method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_send_Cdata_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal event method which is invoked by the host after a radio
    *   packet has been sent or if the retransmission procedure finishes 
    *   without success.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void 
    LoRaWAN_Cdata_Tx_Indication(TWiMOD_HCI_Message* rxMessage);


    /** Internal event message which is sent to the host after reception of a 
    *   reliable radio packet containing application payload. The device will
    *   acknowledge the reception with a set Ack-Bit in the next reliable/
    *   unreliable uplink radio packet to the network server.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void 
    LoRaWAN_Cdata_Rx_Indication(TWiMOD_HCI_Message* rxMessage);


    /** Internal event method which is sent to the host in case no expected
    *   confirmation or data has been received as a result of prior reliable
    *   uplink radio packet.
    *
    *   @note Method is not used at the moment!
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Rx_NoData_Indication(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'set_radio_stack_config'
    *   method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Set_RadioStack_Config_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'get_radio_stack_config'
    *   method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Get_RadioStack_Config_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'get_freqband_info'
    *   method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Get_FreqBand_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'get_EUI' method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Get_DeviceEUI_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'get_custom_configuration'
    *   method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Get_Custom_Config_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'get_network_status'
    *   method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Network_Status_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'get_PowerLimit_configuration'
    *   method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Get_TxPower_Limit_Config_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to evaluate response from 'set_PowerLimit_configuration'
    *   method.
    *
    *   @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
    */
    static void
    LoRaWAN_Set_TxPower_Limit_Config_Rsp(TWiMOD_HCI_Message* rxMessage);


    /** Internal method to show status response as human readable string.
    *
    *   @param message_string : Message string to display
    *   @param statusTable : Device management status strings
    *   @param statusID : Status identifier
    */    
    static void
    LoRaWAN_ShowResponse(const char* message_string, const TIDString* statusTable, UINT8 statusID);


    /** Internal help method to convert RTC data.
    *
    *   @param RTC_data : RTC data in UINT32 format
    *   @param *RTC_string : RTC data as string
    */
    static void
    convert_RTC_message(UINT32 RTC_data, char *RTC_string);    


    /** Internal help method for data conversion (UINT8 array in UINT16 value).
    *
    *   @param *Payload : UINT8 array
    */    
    static UINT16
    Convert_Payload_Data_UINT16(UINT8 *Payload);


    /** Internal help method for data conversion (UINT8 array in UINT32 value).
    *
    *   @param *Payload : UINT8 array
    */
    static UINT32 
    Convert_Payload_Data_UINT32(UINT8 *Payload);


    /** Internal method to wait until response from LoRa module.
    *
    */  
    void LoRaWAN_wait(void);
};

#endif  // NSL01_h

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------