/**
********************************************************************************
* @file     SerialDevice.h
*
* @brief    Functions for NSL01 class to communicate with onboard RF radio 
*           module via UART interface.
* 
* @brief    For more information about the NSL01 LoRaWAN shield:
*           http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
*
* @note     The SerialDevice files are included in the dependencies directory of
*           the project and these files are necessary for the NSL01 class!
*
* @author   mCloud Systems Team
* @version  V1.0
* @date     20-June-2018
********************************************************************************
*/

#ifndef SERIALDEVICE_H
#define SERIALDEVICE_H

//------------------------------------------------------------------------------
//
//  Include Files
//
//------------------------------------------------------------------------------

#include "mbed.h"
#include "./dependencies/SLIP.h"
#include <stdint.h>

//------------------------------------------------------------------------------
//
//  Section Defines
//
//------------------------------------------------------------------------------

#define Baudrate_9600       9600
#define Baudrate_115200     115200
#define DataBits_7          7
#define DataBits_8          8
#define Parity_Even         EVENPARITY
#define Parity_None         NOPARITY

//#define DEBUG

//------------------------------------------------------------------------------
//
//  Section Declarations
//
//------------------------------------------------------------------------------

//--Global variables from main
extern Serial comm_pc;
extern Serial comm_LoRa;

//--Define shortcuts for datatypes
typedef uint8_t             UINT8;
typedef uint32_t            UINT32;

//------------------------------------------------------------------------------
//
//  Function Prototypes
//
//------------------------------------------------------------------------------


/** Function to open serial device with given parameters.
*
*   @note The serial object have to be defined in main.cpp file!
*
*   @param baudrate : Baudrate for serial communication
*   @param dataBits : Number of data bits for serial communication
*/
void
SerialDevice_Open(UINT32        baudRate,
                  int           dataBits);


/** UART Rx callback function for data reception.
*
*/
void rxCallback(void);


/** Function to transmit a single (wakeup) byte to the RF device.
*
*   @param txByte : Tx byte
*
*   @returns 1 on success, -1 on error
*/
int
SerialDevice_SendByte(UINT8 txByte);


/** Function to transmit data to the RF device.
*
*   @param txBuffer : Pointer to Tx buffer
*   @param txLength : Length of Tx buffer
*
*   @returns number of Tx bytes on success, -1 on error
*/
int
SerialDevice_SendData(UINT8*    txBuffer,
                      int       txLength);


/** Function to manually read data into Rx buffer.
*
*   @note This function is not necessary at the moment, because of internal Rx
*         callback function!
*
*   @param rxBuffer     : Pointer to Rx buffer
*   @param rxBufferSize : Length of Rx buffer
*
*   @returns number of Rx bytes on success, -1 on error
*/
int
SerialDevice_ReadData(UINT8*    rxBuffer,
                      int       rxBufferSize);

#endif  // SERIALDEVICE_H

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------