/**
********************************************************************************
* @file     NSL01_HCI_Layer.h
*
* @brief    Functions for NSL01 class to create a Human Control Interface (HCI)
*           for controlling the onboard RF radio module.
* 
* @brief    For more information about the NSL01 LoRaWAN shield:
*           http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
*
* @note     The NSL01_HCI_Layer files are included in the dependencies directory
*           of the project and these files are necessary for the NSL01 class!
*
* @author   -
* @version  V1.0
* @date     20-June-2018
********************************************************************************
*/

#ifndef NSL01_HCI_LAYER_H
#define NSL01_HCI_LAYER_H

//------------------------------------------------------------------------------
//
//  Include Files
//
//------------------------------------------------------------------------------

#include "mbed.h"
#include "./dependencies/CRC16.h"
#include "./dependencies/SLIP.h"
#include "./dependencies/SerialDevice.h"
#include <stdint.h>

//------------------------------------------------------------------------------
//
//  Section Defines
//
//------------------------------------------------------------------------------

//--Define HCI message parameter
#define WIMOD_HCI_MSG_HEADER_SIZE       2
#define WIMOD_HCI_MSG_PAYLOAD_SIZE      300
#define WIMOD_HCI_MSG_FCS_SIZE          2

#define LOBYTE(x)                       (x)
#define HIBYTE(x)                       ((UINT16)(x) >> 8)

//#define DEBUG

//------------------------------------------------------------------------------
//
//  Section Declarations
//
//------------------------------------------------------------------------------

//--Global variables from main
extern Serial comm_pc;
extern Serial comm_LoRa;

//--Define shortcuts for datatypes
typedef unsigned char                   UINT8;
typedef uint16_t                        UINT16;
typedef uint32_t                        UINT32;

//------------------------------------------------------------------------------
//
//  HCI Message Structure (internal software usage)
//
//------------------------------------------------------------------------------

typedef struct
{
    //--Payload length information
    UINT16  Length;

    //--Service Access Point Identifier
    UINT8   SapID;

    //--Message Identifier
    UINT8   MsgID;

    //--Payload Field
    UINT8   Payload[WIMOD_HCI_MSG_PAYLOAD_SIZE];

    //--Frame Check Sequence Field
    UINT8   CRC16[WIMOD_HCI_MSG_FCS_SIZE];

}TWiMOD_HCI_Message;

//------------------------------------------------------------------------------
//
//  Function Prototypes
//
//------------------------------------------------------------------------------


/** Human Control Interface (HCI) message receiver callback defintion.
*
*   @param rxMessage : Pointer to HCI Rx message struct
*/
typedef TWiMOD_HCI_Message* (*TWiMOD_HCI_CbRxMessage)(TWiMOD_HCI_Message* rxMessage);


/** Function to initialize Human Control Interface (HCI) message layer.
*
*   @param cbRxMessage : Function pointer
*   @param rxMessage   : Pointer to HCI Rx message struct
*/
void
WiMOD_HCI_Init(TWiMOD_HCI_CbRxMessage   cbRxMessage,
               TWiMOD_HCI_Message*      rxMessage);


/** Function to transmit a message via specified Human Control Interface (HCI)
*   with or without payload.
*
*   @param txMessage : Pointer to HCI Tx message struct
*
*   @returns 1 on success, -1 on error
*/
int
WiMOD_HCI_SendMessage(TWiMOD_HCI_Message* txMessage);


/** Function to read/process incoming serial data, which is a user triggered
*   function without using interrupts.
*
*   @note This function is not necessary at the moment, because the incoming
*         serial data is processed via an Interrupt Service Routine (ISR).
*
*/
void
WiMOD_HCI_Process();

#endif  // NSL01_HCI_LAYER_H

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------