/**
********************************************************************************
* @file     CRC16.h
*
* @brief    Functions for NSL01 class to calculate and check CRC16.
* 
* @brief    For more information about the NSL01 LoRaWAN shield:
*           http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
*
* @note     The CRC16 files are included in the dependencies directory of the
*           project and these files are necessary for the NSL01 class!
*
* @author   -
* @version  V1.0
* @date     20-June-2018
********************************************************************************
*/

#ifndef    __CRC16_H__
#define    __CRC16_H__

//------------------------------------------------------------------------------
//
//  Include Files
//
//------------------------------------------------------------------------------ 

#include <stdint.h>

//------------------------------------------------------------------------------
//
//  Section Defines
//
//------------------------------------------------------------------------------

#define CRC16_INIT_VALUE    0xFFFF    //-Initial value for CRC algorithm
#define CRC16_GOOD_VALUE    0x0F47    //-Constant compare value for check
#define CRC16_POLYNOM       0x8408    //-16-BIT CRC CCITT POLYNOM

//------------------------------------------------------------------------------
//
//  Section Declarations
//
//------------------------------------------------------------------------------

typedef uint8_t     UINT8;
typedef uint16_t    UINT16;

//------------------------------------------------------------------------------
//
//  Function Prototypes
//
//------------------------------------------------------------------------------


/** Function to calculate CRC16.
*
*   @note This function calculates the one's complement of the standard 16-BIT
*   CRC CCITT polynomial G(x) = 1 + x^5 + x^12 + x^16.
*
*   @param data     : Input sequency
*   @param length   : Length of input sequence
*   @param initVal  : Inital value
*
*   @returns calculated checksum.
*/
UINT16
CRC16_Calc  (UINT8*     data,
             UINT16     length,
             UINT16     initVal);


/** Function to check CRC16.
*
*   @note This function checks a data block with attached CRC16.
*
*   @param data     : Input sequency
*   @param length   : Length of input sequence
*   @param initVal  : Inital value
*
*   @returns true on success, false on error
*/
bool
CRC16_Check (UINT8*     data,
             UINT16     length,
             UINT16     initVal);


#endif  // __CRC16_H__

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------