/*******************************************************************************
//  Class for controlling NSL01 LoRaWAN shield from mCloud Systems GmbH
//
//  For more information about the NSL01 LoRaWAN shield:
//      http://www.mcloud-systems.com/nsl01-lorawan-nucleo-arduino-shield
//
//  @author  mCloud Systems Team
//  @version 1.0
//  @date    20-June-2018
//
//  @attention  
//
//  Copyright (c) 2018 mCloud Systems GmbH, MIT License
//
//  Permission is hereby granted, free of charge, to any person obtaining a copy 
//  of this software and associated documentation files (the "Software"), to deal
//  in the Software without restriction, including without limitation the rights 
//  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//  copies of the Software, and to permit persons to whom the Software is
//  furnished to do so, subject to the following conditions:
//
//  The above copyright notice and this permission notice shall be included in
//  all copies or substantial portions of the Software.
//
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
//  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//  SOFTWARE.
*******************************************************************************/

//------------------------------------------------------------------------------
//
// Include Files
//
//------------------------------------------------------------------------------

#include "mbed.h"
#include "NSL01.h"
#include <string.h>
#include <stdio.h>

#define MAKEWORD(lo,hi) ((lo)|((hi) << 8))
#define MAKELONG(lo,hi) ((lo)|((hi) << 16))

//------------------------------------------------------------------------------
//
// Initialize internal functions and variables
//
//------------------------------------------------------------------------------

//--Flag to control response from LoRa module
bool flag_LoRaWAN_response = false;

//------------------------------------------------------------------------------
//
//  Section RAM
//
//------------------------------------------------------------------------------

//--Reserve one Tx-Message
TWiMOD_HCI_Message TxMessage;

//--Reserve one Rx-Message
TWiMOD_HCI_Message RxMessage;

//------------------------------------------------------------------------------
//
//  Section Const
//
//------------------------------------------------------------------------------

//--Status Codes for DeviceMgmt Response Messages
static const TIDString DeviceMgmt_StatusStrings[] =
{
    { DEVMGMT_STATUS_OK,                   "ok" },
    { DEVMGMT_STATUS_ERROR,                "error" } ,
    { DEVMGMT_STATUS_CMD_NOT_SUPPORTED,    "command not supported" },
    { DEVMGMT_STATUS_WRONG_PARAMETER,      "wrong parameter" },
    { DEVMGMT_STATUS_WRONG_DEVICE_MODE,    "wrong device mode" },

    //--End of table
    { 0, 0 }
};

//--Status Codes for LoRaWAN Response Messages
static const TIDString LoRaWAN_StatusStrings[] =
{
    { LORAWAN_STATUS_OK,                    "ok" },
    { LORAWAN_STATUS_ERROR,                 "error" } ,
    { LORAWAN_STATUS_CMD_NOT_SUPPORTED,     "command not supported" },
    { LORAWAN_STATUS_WRONG_PARAMETER,       "wrong parameter" },
    { LORAWAN_STATUS_WRONG_DEVICE_MODE,     "wrong device mode" },
    { LORAWAN_STATUS_NOT_ACTIVATED,         "device not activated" },
    { LORAWAN_STATUS_BUSY,                  "device busy - command rejected" },
    { LORAWAN_STATUS_QUEUE_FULL,            "message queue full - command rejected" },
    { LORAWAN_STATUS_LENGTH_ERROR,          "HCI message length error" },
    { LORAWAN_STATUS_NO_FACTORY_SETTINGS,   "no factory settings available" },
    { LORAWAN_STATUS_CHANNEL_BLOCKED_BY_DC, "channel blocked by duty cycle" },
    { LORAWAN_STATUS_CHANNEL_NOT_AVAILABLE, "channel not available" },

    // end of table
    { 0, 0 }
};

//------------------------------------------------------------------------------
//
//  Constructor of NSL01:
//
//  @brief  Create a NSL01 instance with the serial device objects specified in
//          main.
//
//------------------------------------------------------------------------------

NSL01::NSL01(void)
{  
    //--Wait for power-up LoRa module
    wait(2.0);
    
    //--Initialize NSL01 and open serial (UART) device
    initialize();
}

//------------------------------------------------------------------------------
//
//  Destructor of NSL01:
//
//  @brief  Destroy instance
//
//------------------------------------------------------------------------------

NSL01::~NSL01() {}

//------------------------------------------------------------------------------
//
//  Public method for device management: ping
//
//  @brief  Ping NSL01 device.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::ping(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_PING_REQ;
    TxMessage.Length    = 0;
    
    comm_pc.printf("\r\nPing request: ");
    
    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();
    
    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: reset
//
//  @brief  Reset NSL01 device (Software reset).
//
//  @note  After executing this command the module will be restarted after
//  approximately 200ms!
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::reset(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_RESET_REQ;
    TxMessage.Length    = 0;
    
    comm_pc.printf("\r\nSoftware reset request: ");
    
    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();
    
    return return_value;    
}

//------------------------------------------------------------------------------
//
//  Public method for device management: get_device_info
//
//  @brief  Method to identify the local connected NSL01 device. As a result the
//          device sends a response message which contains all necessary device 
//          information.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_device_info(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_GET_DEVICE_INFO_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet device information: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();
    
    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: get_firmware_info
//
//  @brief  Method to identify the onboard radio firmware of NSL01. As a result
//          the device sends a response message which contains all firmware 
//          related data.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_firmware_info(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_GET_FW_VERSION_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet firmware information: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: get_device_status
//
//  @brief  Method to get the current device status of NSL01. As a result the
//          device sends a response message which contains all device status
//          information.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_device_status(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_GET_DEVICE_STATUS_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet device status: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: get_network_status
//
//  @brief  Method to read the current network / activation status of NSL01.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_network_status(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_GET_NWK_STATUS_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet network activation status: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: get_RTC_info
//
//  @brief  Method to read the current Real Time Clock (RTC) value of NSL01.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_RTC_info(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_GET_RTC_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet RTC info: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: set_RTC
//
//  @brief  Method to set the Real Time Clock (RTC) value of NSL01.
//
//  @param RTC_data defined in RTC_struct
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_RTC(RTC_struct RTC_data)
{
    //--Internal variables and constants
    UINT32 result_UINT32 = 0;
    
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_SET_RTC_REQ;
    TxMessage.Length    = 4;

    //--Prepare RTC info
    result_UINT32 = (UINT32)(((RTC_data.year - 2000) & 0x3f) << 26 | (RTC_data.day & 0x1f) << 21 | 
                              (RTC_data.hour & 0x1f) << 16 | (RTC_data.month & 0xf) << 12 | 
                              (RTC_data.minute & 0x3f) << 6 | (RTC_data.second & 0x1f));

    //--Convert RTC info (LSB first)     
    unsigned char *s = (unsigned char *)&result_UINT32;
    UINT32 result_conversion = (UINT32)(s[0] << 24 | s[1] << 16 | s[2] << 8 | s[3]);    
    TxMessage.Payload[0]= s[0];
    TxMessage.Payload[1]= s[1];
    TxMessage.Payload[2]= s[2];
    TxMessage.Payload[3]= s[3];    
    
    comm_pc.printf("\r\nSet RTC: ");
    
    // 2. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;    
}

//------------------------------------------------------------------------------
//
//  Public method for device management: set_RTC_alarm
//
//  @brief  Method to set the Real Time Clock (RTC) alarm of NSL01.
//
//  @param RTC_alarm_data defined in RTC_alarm_struct
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_RTC_alarm(RTC_alarm_struct RTC_alarm_data)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_SET_RTC_ALARM_REQ;
    TxMessage.Length    = 4;
    
    #ifdef DEBUG
        comm_pc.printf("\r\nAlarm period: %d\r\n",RTC_alarm_data.period);
        comm_pc.printf("Alarm hour: %u\r\n",RTC_alarm_data.hour);
        comm_pc.printf("Alarm minutes: %u\r\n",RTC_alarm_data.minute);
        comm_pc.printf("Alarm seconds: %u\r\n",RTC_alarm_data.second);
    #endif
    
    comm_pc.printf("\r\nSet RTC alarm: "); 
    
    //2. Check input parameters
    if (!((RTC_alarm_data.periodic == true) || (RTC_alarm_data.periodic == false)))
    {
        comm_pc.printf("Error - No valid data for alarm period\r\n");
        return -1;
    }
    
    if (((int)RTC_alarm_data.hour < 0) || (RTC_alarm_data.hour > 23))
    {
        comm_pc.printf("Error - No valid data for alarm hour\r\n");
        return -1;
    }
    
    if (((int)RTC_alarm_data.minute < 0) || (RTC_alarm_data.minute > 59))
    {
        comm_pc.printf("Error - No valid data for alarm minute\r\n");
        return -1;
    }
    
    if (((int)RTC_alarm_data.second < 0) || (RTC_alarm_data.second > 59))
    {
        comm_pc.printf("Error - No valid data for alarm second\r\n");
        return -1;
    }
    
    // 3. Prepare payload
    TxMessage.Payload[0] = RTC_alarm_data.periodic;
    TxMessage.Payload[1] = RTC_alarm_data.hour;
    TxMessage.Payload[2] = RTC_alarm_data.minute;
    TxMessage.Payload[3] = RTC_alarm_data.second;
    
    // 4. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 5. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: get_RTC_alarm
//
//  @brief  Method to get the Real Time Clock (RTC) alarm info of NSL01.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_RTC_alarm(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_GET_RTC_ALARM_REQ;
    TxMessage.Length    = 0;
    
    comm_pc.printf("\r\nGet RTC alarm: ");
        
    // 2. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: clear_RTC_alarm
//
//  @brief  Method to clear a pending Real Time Clock (RTC) alarm of NSL01.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::clear_RTC_alarm(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_CLEAR_RTC_ALARM_REQ;
    TxMessage.Length    = 0;
    
    comm_pc.printf("\r\nClear RTC alarm: ");
    
    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: get_system_operation_mode
//
//  @brief  Method to get the current system operation mode of NSL01.
//
//  @note  Two system operation modes are supported:
//  - Standard / Application Mode
//  - Customer Mode
//
//  @note  Some device functionality is only supported in customer mode!
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_system_operation_mode(void)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_GET_OPMODE_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet system operation mode: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for device management: set_system_operation_mode
//
//  @brief  Method to set the system operation mode of NSL01.
//
//  @note  After executing this command the module will be restarted after
//  approximately 200ms!
//
//  @param system_mode 0 : Standard/Application mode
//  @param system_mode 1 : Reserved/Not used
//  @param system_mode 2 : Reserved/Not used
//  @param system_mode 3 : Customer mode
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_system_operation_mode(UINT8 system_mode)
{
    // 1. Init header
    TxMessage.SapID     = DEVMGMT_SAP_ID;
    TxMessage.MsgID     = DEVMGMT_MSG_SET_OPMODE_REQ;
    TxMessage.Length    = 1;    
    
    comm_pc.printf("\r\nSet system operation mode: ");
            
    // 2. Check system mode
    if (!(system_mode == 0 || system_mode == 3))
    {
        comm_pc.printf("Error - Mode not supported!\r\n");
        return -1;
    }
        
    // 3. Prepare payload
    TxMessage.Payload[0] = system_mode;
    
    // 4. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);
    
    // 5. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: ABP_device_activation
//
//  @brief  Method to activate NSL01 in Activation by Personalization (ABP) mode
//          and save parameters in non volatile memory.
//
//  @note  A device must be activated prior to any further data exchange with
//         a network server!
//
//  @note  After a successful activation the NSL01 will automatically send an
//         empty unconfirmed uplink message ("alive message") over the air!
//
//  @param deviceAddress  : unique 32-Bit Device Address
//  @param *nwkSessionKey : device specific 128-Bit Network Session Key
//  @param *appSessionKey : device specific 128-Bit Application Session Key
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::ABP_device_activation(UINT32 deviceAddress,    //-Device address         
                                 UINT8* nwkSessionKey,    //-network session key
                                 UINT8* appSessionKey)    //-application session key
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_ACTIVATE_DEVICE_REQ;
    TxMessage.Length    = 36;    

    // 2. Payload
    // 2.1 Device address: Payload[0...3]: LSB first            

    //--Convert device address (LSB first)     
    unsigned char *s = (unsigned char *)&deviceAddress;
    UINT32 result_conversion = (UINT32)(s[0] << 24 | s[1] << 16 | s[2] << 8 | s[3]);    
    TxMessage.Payload[0]= s[0];
    TxMessage.Payload[1]= s[1];
    TxMessage.Payload[2]= s[2];
    TxMessage.Payload[3]= s[3];

    //--Show result of conversion
    #ifdef DEBUG
        comm_pc.printf("\r\n ABP activation: - device address Payload 01: %02x, Payload 02: %02x, Payload 03: %02x, Payload 04: %02x\r\n", s[0], s[1], s[2], s[3]);
    #endif

    // 2.2 Network session key: Payload[4...19]: MSB first
    if(nwkSessionKey)
    {
        UINT8*  dstPtr  = TxMessage.Payload + DEVICE_ADDR_LEN;
        int     n       = KEY_LEN;

        // copy bytes
        while(n--)
            *dstPtr++ = *nwkSessionKey++;
    }

    // 2.3 Application session key: Payload[20...35]: MSB first
    if(appSessionKey)
    {
        UINT8*  dstPtr  = TxMessage.Payload + DEVICE_ADDR_LEN + KEY_LEN;
        int     n       = KEY_LEN;

        // copy bytes
        while(n--)
            *dstPtr++ = *appSessionKey++;
    }    

    comm_pc.printf("\r\nActivate device in ABP mode: ");

    // 3. send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 4. Wait until any response from LoRa module
    LoRaWAN_wait();
    
    // 5. Wait an additional moment
    wait(2);

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: ABP_device_reactivation
//
//  @brief  Method to reactivate NSL01 in Activation by Personalization (ABP)
//          mode using the parameters previously stored in non-volatile memory.
//
//  @note  A device must be activated prior to any further data exchange with
//         a network server!
//
//  @note  After a successful activation the NSL01 will automatically send an
//         empty unconfirmed uplink message ("alive message") over the air!
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::ABP_device_reactivation(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_REACTIVATE_DEVICE_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nReactivate device in ABP mode: ");    

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();
    
    // 4. Wait an additional moment
    wait(2);

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: deactivate NSL01
//
//  @brief  Method to deactivate the NSL01 LoRaWAN end-device, i.e. further data
//          exchange over the air will be disabled.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::deactivate(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_DEACTIVATE_DEVICE_REQ;
    TxMessage.Length    = 0;            

    comm_pc.printf("\r\nDeactivate device: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: set_join_parameter
//
//  @brief  Method to configure the Over-the-Air Activation (OTTA) parameters of
//          NSL01 which are used during the join network procedure 
//          'join_network'.
//
//  @param *appEUI : globally unique 64-Bit Application EUI
//  @param *appKey : device specific 128-Bit AES Application Key
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_join_parameter(UINT8* appEUI, UINT8* appKey)
{
   // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_SET_JOIN_PARAM_REQ;
    TxMessage.Length    = 24;

    // 2. Prepare payload
    // 2.1 64-Bit Application EUI: Payload[0...7]: MSB first            
    if (appEUI)
    {
        UINT8*  dstPtr  = TxMessage.Payload;
        int     n       = EUI_LEN;

        // copy bytes
        while(n--)
            *dstPtr++ = *appEUI++;
    }    

    // 2.2 128-Bit Application Key: Payload[8...23]: MSB first
    if(appKey)
    {
        UINT8*  dstPtr  = TxMessage.Payload + EUI_LEN;
        int     n       = KEY_LEN;

        // copy bytes
        while(n--)
            *dstPtr++ = *appKey++;
    }    

    comm_pc.printf("\r\nSet join parameter: ");

    // 3. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 4. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: join_network
//
//  @brief  Method to start the join network procedure of NSL01 in Over-the-Air
//          Activation (OTTA) mode with previously defined parameters in 
//          'set_join_parameter' method.
//
//  @note  Once the 'join_network' method is triggered an internal event
//         method is invoked periodically to indicate the joining procedure.
//
//  @note  An internal join network indication event method is invoked after
//         successful reception of a server join response packet or after the
//         expiration of a complete join process without success (the join 
//         request will be retransmitted changing the spreading factor from SF7
//         till SF12, reusing each spreading factor twice).
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::join_network(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_JOIN_NETWORK_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nJoin network request: ");

    // 2. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: send_Udata
//
//  @brief  Method to transmit data in an unreliable way to the network server,
//          i.e. no acknowledgement will be sent from the network server and no
//          retransmission method is available on the end-device (NSL01) side.
//
//  @note  An internal 'unreliable data transmit indication' event method is 
//         invoked after the radio packet has been sent, containing information
//         about the data transfer.
//
//  @param port      : LoRaWAN Port number (> 0)
//  @param *srcData  : Application Payload (data which have to be transmitted)
//  @param srcLength : Lenght of Application Payload data
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::send_Udata(UINT8 port, UINT8* srcData, int srcLength)
{
    comm_pc.printf("\r\nTransmit U-Data: ");

    // 1. Check length of application payload
    if (srcLength > (WIMOD_HCI_MSG_PAYLOAD_SIZE - 1))
    {
        //--Error
        comm_pc.printf("Error - transmit radio packet: Payload size exceeded!\r\n");
        return -1;
    }

    // 2. Check LoRAWAN port number range
    if (port <= 0)
    {
        //--Error
        comm_pc.printf("Error - transmit radio packet: Invalid port number!\r\n");
        return -1;        
    }

    // 3. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_SEND_UDATA_REQ;
    TxMessage.Length    = 1 + srcLength;

    // 4.  Init payload
    // 4.1 Init port
    TxMessage.Payload[0] = port;

    // 4.2 Init radio message payload
    memcpy(&TxMessage.Payload[1], srcData, srcLength);

    // 5. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 6. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: send_Cdata
//
//  @brief  Method to transmit data in a reliable way to the network server,
//          i.e. confirmed data transmission.
//
//  @note  An internal 'reliable data transmit indication' event method is 
//         invoked after the radio packet has been sent, containing information
//         about the data transfer.
//
//  @param port      : LoRaWAN Port number (> 0)
//  @param *srcData  : Application Payload (data which have to be transmitted)
//  @param srcLength : Lenght of Application Payload data
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::send_Cdata(UINT8 port, UINT8* srcData, int srcLength)
{
    comm_pc.printf("\r\nTransmit C-Data: ");

    // 1. Check length of application payload
    if (srcLength > (WIMOD_HCI_MSG_PAYLOAD_SIZE - 1))
    {
        //--Error
        comm_pc.printf("Error - transmit radio packet: Payload size exceeded!\r\n");
        return -1;
    }

    // 2. Check LoRAWAN port number range
    if (port <= 0)
    {
        //--Error
        comm_pc.printf("Error - transmit radio packet: Invalid port number!\r\n");
        return -1;        
    }

    // 3. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_SEND_CDATA_REQ;
    TxMessage.Length    = 1 + srcLength;

    // 4.  Init payload
    // 4.1 Init port
    TxMessage.Payload[0] = port;

    // 4.2 Init radio message payload
    memcpy(&TxMessage.Payload[1], srcData, srcLength);

    // 5. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 6. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: set_radio_stack_config
//
//  @brief  Method to set/configure the internal radio stack configuration
//          of NSL01 <=> main parameters for data transmission.
//
//  @param radio_stack : Radio stack config data defined in Radio_config_struct
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_radio_stack_config(Radio_config_struct radio_stack)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_SET_RSTACK_CONFIG_REQ;
    TxMessage.Length    = 7;

    comm_pc.printf("\r\nSet radio stack config: ");

    //---------------------------
    // 2. Check input parameters
    //---------------------------

    // 2.1 Data rate index [0...6]
    if ((int)radio_stack.data_rate_index < 0 || radio_stack.data_rate_index > 6)
    {
        comm_pc.printf("Error - Data rate index is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.2 Tx power level (EIRP) [0...16]
    if ((int)radio_stack.Tx_power < 0 || radio_stack.Tx_power > 16)
    {
        comm_pc.printf("Error - Tx power level is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.3 Adaptive data rate [enabled/disabled]
    if (!(radio_stack.adaptive_data_rate == 0 || radio_stack.adaptive_data_rate == 1))
    {
        comm_pc.printf("Error - Adaptive data rate is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.4 Duty Cycle control [enabled/disabled]
    if (!(radio_stack.duty_cycle_control == 0 || radio_stack.duty_cycle_control == 1))
    {
        comm_pc.printf("Error - Duty Cycle mode is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.5 Class C support [enabled/disabled]
    if (!(radio_stack.class_C_support == 0 || radio_stack.class_C_support == 1))
    {
        comm_pc.printf("Error - Class A/C mode is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.6 RF packet output format [standard/extended]
    if (!(radio_stack.RF_packet_format == 0 || radio_stack.RF_packet_format == 1))
    {
        comm_pc.printf("Error - RF packet output format is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.7 Rx MAC command forwarding [enabled/disabled]
    if (!(radio_stack.forwarding == 0 || radio_stack.forwarding == 1))
    {
        comm_pc.printf("Error - Rx MAC command forwarding is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.8 Power saving mode [enabled/disabled]
    if (!(radio_stack.power_saving == 0 || radio_stack.power_saving == 1))
    {
        comm_pc.printf("Error - Power saving mode is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.9 Maximum number of retransmissions [0...254]
    if ((int)radio_stack.number_retransmissions < 0 || radio_stack.number_retransmissions > 254)
    {
        comm_pc.printf("Error - Maximum number of retransmissions is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.10 Radio band selection [1...3]
    if (radio_stack.band_index < 1 || radio_stack.band_index > 3)
    {
        comm_pc.printf("Error - Radio band selection is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    // 2.11 Header MAC Cmd capacity [0...15]
    if ((int)radio_stack.header_MAC_cmd_capacity < 0 || radio_stack.header_MAC_cmd_capacity > 15)
    {
        comm_pc.printf("Error - Header MAC Cmd capacity is out of range!\r\n");
        comm_pc.printf("  Adapt parameter and try again!\r\n");
        return -1;
    }

    //---------------------------
    // 3. Prepare payload
    //---------------------------

    // 3.1 Data rate index
    TxMessage.Payload[0] = radio_stack.data_rate_index;

    // 3.2 TX power level (EIRP) in dBm in 1dB steps
    TxMessage.Payload[1] = radio_stack.Tx_power;

    // 3.3 Options: Bit 0: adaptive data rate          (0:disabled, 1:enabled)
    //              Bit 1: duty cycle control          (0:disabled, 1:enabled)
    //              Bit 2: Class C support             (0:Class A, 1: Class C)
    //              Bit 6: RF packet output format     (0:standard format, 1:extended format)
    //              Bit 7: Rx MAC command forwarding   (0:disabled, 1:enabled)
    TxMessage.Payload[2] = (radio_stack.adaptive_data_rate & 0x01) | (radio_stack.duty_cycle_control << 1 & 0x02) |
                           (radio_stack.class_C_support << 2 & 0x04) | (radio_stack.RF_packet_format << 6 & 0x40) |
                           (radio_stack.forwarding << 7 & 0x80);

    // 3.4 Power saving mode (0:off, 1:automatic)
    TxMessage.Payload[3] = radio_stack.power_saving;

    // 3.5 Maximum number of retransmissions (0...254)
    TxMessage.Payload[4] = radio_stack.number_retransmissions;

    // 3.6 Radio band selection (1...3)
    TxMessage.Payload[5] = radio_stack.band_index;

    // 3.7 Header MAC Cmd capacity (0...15)
    TxMessage.Payload[6] = radio_stack.header_MAC_cmd_capacity;

    //--Show configuration for debugging
    #ifdef DEBUG
        comm_pc.printf("\r\n Radio Stack: 0x%08x\r\n",(radio_stack.adaptive_data_rate & 0x01));
        comm_pc.printf(" Duty Cycle: 0x%08x\r\n",(radio_stack.duty_cycle_control << 1 & 0x02));
        comm_pc.printf(" Class C: 0x%08x\r\n",(radio_stack.class_C_support << 2 & 0x04));
        comm_pc.printf(" RF Packet format: 0x%08x\r\n",(radio_stack.RF_packet_format << 6 & 0x40));
        comm_pc.printf(" Forwarding: 0x%08x\r\n",(radio_stack.forwarding << 7 & 0x80));
    #endif

    // 4. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 5. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: get_radio_stack_config
//
//  @brief  Method to read the current radio stack configuration of NSL01
//          <=> main parameters for data transmission
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_radio_stack_config(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_GET_RSTACK_CONFIG_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet radio stack config: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;    
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: get_freqband_info
//
//  @brief  Method to get information related to the supported frequency bands 
//          used by the firmware. Moreover, the maximum supported EIRP for each
//          band is provided.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_freqband_info(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_GET_SUPPORTED_BANDS_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet frequency band information: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: get_EUI
//
//  @brief  Method to get/read the unique 64-bit device EUI.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_EUI(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_GET_DEVICE_EUI_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nDevice EUI request: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: set_EUI
//
//  @brief  Method to set/write the unique 64-bit device EUI.
//
//  @note  The device EUI can only be set in "customer mode"!
//
//  @param *device_EUI : Unique 64-bit device EUI (UINT8 array)
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_EUI(UINT8 *device_EUI)
{
    //--Internal variables and constants
    UINT8   index;

    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_SET_DEVICE_EUI_REQ;
    TxMessage.Length    = EUI_LEN;

    // 2. Prepare payload
    index = 0;

    //--Copy bytes to array
    while(index < EUI_LEN)
    {
        TxMessage.Payload[index] = *device_EUI;

        //--Update pointer and indices
        device_EUI++;
        index++;
    }

    comm_pc.printf("\r\nSet device EUI: ");

    // 3. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 4. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: set_custom_configuration
//
//  @brief  Method to configure the custom parameters (RF gain offset to
//          compensate possible transmission losses/gains including circuit
//          matching, antennas, ...). This parameter should be rated in units of
//          dBd (decibels relative to a half-wavelength dipole antenna, where
//          0dBd = 2.15 dBi).
//
//  @note  This parameter can only be set in "customer mode"!
//
//  @param RF_gain_offset : RF gain offset in dBd
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_custom_configuration(int RF_gain_offset)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_SET_CUSTOM_CFG_REQ;
    TxMessage.Length    = 1;

    comm_pc.printf("\r\nSet custom configuration: ");

    // 2. Check Tx power offset
    if (RF_gain_offset < -128 || RF_gain_offset > 127)
    {
        comm_pc.printf("Error - Tx power offset not supported!\r\n");
        return -1;
    }

    // 3. Prepare payload
    TxMessage.Payload[0] = (UINT8)RF_gain_offset;

    //--Display result for debugging
    #ifdef DEBUG
        UINT8 tmp_Tx_power_offset = (UINT8)RF_gain_offset;
        INT8  converted_Tx_power_offset = (INT8)tmp_Tx_power_offset;
        comm_pc.printf("RF gain offset: %d dBd\r\n", converted_Tx_power_offset);
    #endif

    // 4. Send HCI message with payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 5. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: get_custom_configuration
//
//  @brief  Method to get/read the custom configuration parameters
//          (RF gain offset) of NSL01.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_custom_configuration(void)
{
   // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_GET_CUSTOM_CFG_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet custom configuration: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: factory_reset
//
//  @brief  Method to restore the initial firmware settings, stored during
//          production time (factory reset).
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::factory_reset(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_FACTORY_RESET_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nApply factory reset: ");

    // 2. send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: get_PowerLimit_configuration
//
//  @brief  Method to read the transmit power limit configuration parameters of
//          NSL01 for different frequency sub-bands.
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::get_PowerLimit_configuration(void)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_GET_TXPOWER_LIMIT_CONFIG_REQ;
    TxMessage.Length    = 0;

    comm_pc.printf("\r\nGet Tx power limit configuration: ");

    // 2. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 3. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Public method for LoRaWAN service: set_PowerLimit_configuration
//
//  @brief  Method to set/configure the transmit power limit configuration
//          parameters of NSL01 for a dedicated frequency sub-band.
//
//  @note  These parameters can only be written in "customer mode"!
//
//  @param sub_band : Sub-band index [0...5] according to specification
//  @param power_limit_flag : Tx power limit flag for sub-band [enabled/disabled]
//  @param power_limit : Tx power limit for sub-band in dBm [0...16]
//
//  @returns 1 on success, -1 on error
//
//------------------------------------------------------------------------------

int NSL01::set_PowerLimit_configuration(UINT8 sub_band, UINT8 power_limit_flag, UINT8 power_limit)
{
    // 1. Init header
    TxMessage.SapID     = LORAWAN_SAP_ID;
    TxMessage.MsgID     = LORAWAN_MSG_SET_TXPOWER_LIMIT_CONFIG_REQ;
    TxMessage.Length    = 3;

    comm_pc.printf("\r\nTx power limit configuration: ");

    // 2. check input parameters
    if (((int)sub_band < 0) || (sub_band > 5))
    {
        comm_pc.printf("Error - Sub-band index is out of range\r\n");
        return -1;
    }

    if (((int)power_limit_flag < 0) || (power_limit_flag > 1))
    {
        comm_pc.printf("Error - Power limit flag is out of range\r\n");
        return -1;
    }

    if (((int)power_limit < 0) || (power_limit > 16))
    {
        comm_pc.printf("Error - Power limit is out of range\r\n");
        return -1;
    }

    // 3. Parameters correct => prepare payload
    TxMessage.Payload[0] = sub_band;
    TxMessage.Payload[1] = power_limit_flag;
    TxMessage.Payload[2] = power_limit;

    // 4. Send HCI message without payload
    int return_value = WiMOD_HCI_SendMessage(&TxMessage);

    // 5. Wait until any response from LoRa module
    LoRaWAN_wait();

    return return_value;
}

//------------------------------------------------------------------------------
//
//  Private method: initialize
//
//  @brief  Internal method to initialize NSL01.
//
//  @note  This function is automatically invoked in constructor.
//
//------------------------------------------------------------------------------

void NSL01::initialize()
{    
    //--Init HCI layer
    WiMOD_HCI_Init(Process_RxMessage,    //-Receiver callback
                   &RxMessage);          //-Rx message
}

//------------------------------------------------------------------------------
//
//  Private method: LoRa_event
//
//  @brief  Internal callback method for event handling of NSL01.
//
//  @note  This method is invoked automatically whenever an event occurs.
//
//  @param Event_ID : Event message identifier defined in header of NSL01.h
//
//------------------------------------------------------------------------------

void NSL01::LoRa_event(int Event_ID)
{
    //--Event-handling
    switch(Event_ID)
    {
        case LORAWAN_EVENT_MSG_ALARM:
            comm_pc.printf("  RTC alarm event!\r\n");
            break;

        case LORAWAN_EVENT_MSG_JOIN_TIMEOUT:
            comm_pc.printf("  Join network timeout -> Please try again!\r\n");
            break;

        case LORAWAN_EVENT_MSG_CDATA_MAX_RETRANSMISSION:
            comm_pc.printf("  Maximum number of C-Data retransmissions reached!\r\n");
            break;

        case LORAWAN_EVENT_MSG_CDATA_PAYLOAD_EXCEEDED:
            comm_pc.printf("  Maximum C-Data payload size exceeded for current data rate!\r\n");
            break;

        default:
            comm_pc.printf("  Unspecified event callback received!\r\n");
            break;                   
    }
}

//------------------------------------------------------------------------------
//
//  Private method (internal callback): Process_RxMessage
//
//  @brief  Internal processing of received HCI messages of NSL01.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//  @returns TWiMOD_HCI_Message* struct
//
//------------------------------------------------------------------------------

TWiMOD_HCI_Message* NSL01::Process_RxMessage(TWiMOD_HCI_Message*  rxMessage)
{
    switch(rxMessage->SapID)
    {
        //--Device management message
        case DEVMGMT_SAP_ID:
            Process_DevMgmt_Message(rxMessage);
            break;

        //--LoRaWAN message
        case LORAWAN_SAP_ID:
            Process_LoRaWAN_Message(rxMessage);
            break;
    }
    return &RxMessage;
}

//------------------------------------------------------------------------------
//
//  Private method: Process_DevMgmt_Message
//
//  @brief  Internal processing of received device management messages.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::Process_DevMgmt_Message(TWiMOD_HCI_Message*  rxMessage)
{
    switch(rxMessage->MsgID)
    {
        case    DEVMGMT_MSG_PING_RSP:
                LoRaWAN_ShowResponse("ping response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    DEVMGMT_MSG_RESET_RSP:
                LoRaWAN_ShowResponse("reset response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);
                comm_pc.printf("  LoRaWAN module will be restarted in approx. 200ms!\r\n");
                flag_LoRaWAN_response = true;
                wait(0.5);
                break;

        case    DEVMGMT_MSG_GET_DEVICE_INFO_RSP:
                DevMgmt_DeviceInfo_Rsp(rxMessage);
                break;                             

        case    DEVMGMT_MSG_GET_FW_VERSION_RSP:
                DevMgmt_FirmwareVersion_Rsp(rxMessage);
                break;

        case    DEVMGMT_MSG_GET_DEVICE_STATUS_RSP:
                DevMgmt_DeviceStatus_Rsp(rxMessage);
                break;

        case    DEVMGMT_MSG_GET_RTC_RSP:
                DevMgmt_RTC_Rsp(rxMessage);
                break;

        case    DEVMGMT_MSG_SET_RTC_RSP:
                LoRaWAN_ShowResponse("RTC set response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    DEVMGMT_MSG_SET_RTC_ALARM_RSP:                                                             
                LoRaWAN_ShowResponse("RTC alarm set response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    DEVMGMT_MSG_RTC_ALARM_IND:
                LoRaWAN_ShowResponse("\r\nRTC alarm event", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);
                LoRa_event(LORAWAN_EVENT_MSG_ALARM);
                break;

        case    DEVMGMT_MSG_GET_RTC_ALARM_RSP:
                DevMgmt_Get_Alarm_Rsp(rxMessage);
                break;                  

        case    DEVMGMT_MSG_CLEAR_RTC_ALARM_RSP:
                LoRaWAN_ShowResponse("clear RTC alarm response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    DEVMGMT_MSG_GET_OPMODE_RSP:
                DevMgmt_OperationMode_Rsp(rxMessage);
                break;                                            

        case    DEVMGMT_MSG_SET_OPMODE_RSP:
                LoRaWAN_ShowResponse("Operation mode response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);
                comm_pc.printf("  LoRaWAN module will be restarted in approx. 200ms!\r\n");
                flag_LoRaWAN_response = true;
                wait(0.5);
                break;

        default:
                comm_pc.printf("unhandled DeviceMgmt message received - MsgID : 0x%02X\n\r", (UINT8)rxMessage->MsgID);
                break;
    }
}

//------------------------------------------------------------------------------
//
//  Private method: Process_LoRaWAN_Message
//
//  @brief  Internal processing of received LoRaWAN messages.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::Process_LoRaWAN_Message(TWiMOD_HCI_Message*  rxMessage)
{    
    switch(rxMessage->MsgID)
    {
        case    LORAWAN_MSG_ACTIVATE_DEVICE_RSP:
                LoRaWAN_ShowResponse("device activation response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);
                break;        

        case    LORAWAN_MSG_REACTIVATE_DEVICE_RSP:
                LoRaWAN_Reactivate_Device_Rsp(rxMessage);
                break;  

        case    LORAWAN_MSG_DEACTIVATE_DEVICE_RSP:
                LoRaWAN_ShowResponse("device deactivation response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    LORAWAN_MSG_SET_JOIN_PARAM_RSP:
                LoRaWAN_ShowResponse("set join parameter response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;                        

        case    LORAWAN_MSG_JOIN_NETWORK_RSP:
                LoRaWAN_ShowResponse("join network response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);
                break;

        case    LORAWAN_MSG_SEND_UDATA_RSP:
                LoRaWAN_send_Udata_Rsp(rxMessage);                
                break;
                
        case    LORAWAN_MSG_SEND_UDATA_TX_IND:
                LoRaWAN_Udata_Tx_Indication(rxMessage);
                break;                

        case    LORAWAN_MSG_RECV_UDATA_RX_IND:
                LoRaWAN_Udata_Rx_Indication(rxMessage);
                break; 

        case    LORAWAN_MSG_SEND_CDATA_RSP:
                LoRaWAN_send_Cdata_Rsp(rxMessage);                
                break;

        case    LORAWAN_MSG_SEND_CDATA_TX_IND:
                LoRaWAN_Cdata_Tx_Indication(rxMessage);
                break;

        case    LORAWAN_MSG_RECV_CDATA_RX_IND:
                LoRaWAN_Cdata_Rx_Indication(rxMessage);
                break;                

        case    LORAWAN_MSG_RECV_NODATA_IND:
                //--Method not used/necessary at the moment
                //LoRaWAN_Rx_NoData_Indication(rxMessage);
                break;
                
        case    LORAWAN_MSG_SET_RSTACK_CONFIG_RSP:
                LoRaWAN_Set_RadioStack_Config_Rsp(rxMessage);
                break;

        case    LORAWAN_MSG_GET_RSTACK_CONFIG_RSP:
                LoRaWAN_Get_RadioStack_Config_Rsp(rxMessage);
                break;

        case    LORAWAN_MSG_GET_SUPPORTED_BANDS_RSP:
                LoRaWAN_Get_FreqBand_Rsp(rxMessage);
                break;

        case    LORAWAN_MSG_GET_DEVICE_EUI_RSP:
                LoRaWAN_Get_DeviceEUI_Rsp(rxMessage);
                break;

        case    LORAWAN_MSG_SET_DEVICE_EUI_RSP:
                LoRaWAN_ShowResponse("device EUI response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    LORAWAN_MSG_SET_CUSTOM_CFG_RSP:
                LoRaWAN_ShowResponse("set custom configuration response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    LORAWAN_MSG_GET_CUSTOM_CFG_RSP:
                LoRaWAN_Get_Custom_Config_Rsp(rxMessage);
                break;

        case    LORAWAN_MSG_FACTORY_RESET_RSP:
                LoRaWAN_ShowResponse("factory reset response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);
                flag_LoRaWAN_response = true;
                break;

        case    LORAWAN_MSG_GET_NWK_STATUS_RSP:
                LoRaWAN_Network_Status_Rsp(rxMessage);
                break;

        case    LORAWAN_MSG_GET_TXPOWER_LIMIT_CONFIG_RSP:
                LoRaWAN_Get_TxPower_Limit_Config_Rsp(rxMessage);
                break; 

        case    LORAWAN_MSG_SET_TXPOWER_LIMIT_CONFIG_RSP:
                LoRaWAN_Set_TxPower_Limit_Config_Rsp(rxMessage);
                break;

        case    LORAWAN_MSG_JOIN_TRANSMIT_IND:
                LoRaWAN_Process_JoinTxIndication(rxMessage);
                break;

        case    LORAWAN_MSG_JOIN_NETWORK_IND:
                LoRaWAN_Process_JoinNetworkIndication(rxMessage);
                break;

        default:
                comm_pc.printf("unspecified LoRaWAN message received - MsgID : 0x%02X\r\n", (UINT8)rxMessage->MsgID);
                break;
    }
}

//------------------------------------------------------------------------------
//
//  Private method: DevMgmt_DeviceInfo_Rsp
//
//  @brief  Internal method to show device information of NSL01
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::DevMgmt_DeviceInfo_Rsp(TWiMOD_HCI_Message*  rxMessage)
{   
    //--Internal variables and constants
    UINT32 result_UINT32;    

    LoRaWAN_ShowResponse("device information response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    if (rxMessage->Payload[0] == DEVMGMT_STATUS_OK)
    {               
        //*******************************
        //--Get radio module identifier
        //*******************************
        UINT8 module_type = rxMessage->Payload[1];

        //--Show information
        switch (module_type)
        {
            case 144:
                comm_pc.printf("  Module Type   : iM880A\r\n");
                break;

            case 146:
                comm_pc.printf("  Module Type   : iM880A-L\r\n");
                break;

            case 147:
                comm_pc.printf("  Module Type   : iU880A\r\n");
                break;

            case 152:
                comm_pc.printf("  Module Type   : iM880B-L\r\n");
                break;

            case 153:
                comm_pc.printf("  Module Type   : iU880B\r\n");
                break;

            case 154:
                comm_pc.printf("  Module Type   : iM980A\r\n");
                break;                           

            case 160:
                comm_pc.printf("  Module Type   : iM881A\r\n");
                break;

            default:
                comm_pc.printf("  Module Type   : not specivied\r\n");
                break;                                        
        }

        //*******************************
        //--Get device address
        //*******************************

        //--Convert device address (ntohl-function)
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[2]));        
        comm_pc.printf("  Device Address: 0x%08x\r\n",result_UINT32);

        //*******************************
        //--Get device ID
        //*******************************

        //--Convert device address (ntohl-function)
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[6]));        
        comm_pc.printf("  Device ID     : 0x%08x\r\n",result_UINT32);
     }

     flag_LoRaWAN_response = true;         
}

//------------------------------------------------------------------------------
//
//  Private method: DevMgmt_FirmwareVersion_Rsp
//
//  @brief  Internal method to show firmware version of NSL01.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::DevMgmt_FirmwareVersion_Rsp(TWiMOD_HCI_Message*  rxMessage)
{
    //--Variables for string splitting
    char tmp[80];
    char info_1[30];
    char info_2[30];
    char info_3[30];
    char info_4[30];
    char cnt= 0;
    char delimiter[] = ",;";
    char *ptr;

    LoRaWAN_ShowResponse("firmware information response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    if (rxMessage->Payload[0] == DEVMGMT_STATUS_OK)
    {
        comm_pc.printf("  Firmware version: V%d.%d\n\r", (int)rxMessage->Payload[2], (int)rxMessage->Payload[1]);
        comm_pc.printf("  Build Count     : %d\n\r", (int)MAKEWORD(rxMessage->Payload[3], rxMessage->Payload[4]));

        memcpy(tmp, &rxMessage->Payload[5], 10);
        tmp[10] = 0;
        comm_pc.printf("  Build Date      : %s\n\r", tmp);

        //--More information attached?
        if (rxMessage->Length > 15)
        {
            //--Add string termination
            rxMessage->Payload[rxMessage->Length] = 0;

            //--Copy attached information to temporary string
            memcpy(tmp, &rxMessage->Payload[15], 40);

            //--Find first deliminator in string
            ptr = strtok(tmp, delimiter);

            //--Split information
            while(ptr != NULL) 
            {
                //--Update counter
                cnt++;

                //--Split information
                if (cnt == 1)
                {
                    memcpy(info_1, ptr, 20);
                }
                else if (cnt == 2)
                {
                    memcpy(info_2, ptr, 20);
                }
                else if (cnt == 3)
                {
                    memcpy(info_3, ptr, 20);
                }
                else if (cnt == 4)
                {
                    memcpy(info_4, ptr, 20);
                }

                //--Find next delimiter
                ptr = strtok(NULL, delimiter);
            }

            //--Show information
            switch (cnt)
            {
                case 0: 
                    return;

                case 1:
                    comm_pc.printf("  Firmware Image  : %s\r\n", &info_1[0]);
                    break;

                case 2:                    
                    comm_pc.printf("  Firmware Image  : %s\r\n", &info_1[0]);
                    comm_pc.printf("  Radio Stack     : %s\r\n", &info_2[0]);
                    break;

                case 3:                    
                    comm_pc.printf("  Firmware Image  : %s\r\n", &info_1[0]);
                    comm_pc.printf("  Radio Stack     : %s\r\n", &info_2[0]);
                    comm_pc.printf("  RF Band         : %s\r\n", &info_3[0]);                
                    break;

                case 4:                    
                    comm_pc.printf("  Firmware Image  : %s\r\n", &info_1[0]);
                    comm_pc.printf("  Radio Stack     : %s\r\n", &info_2[0]);
                    comm_pc.printf("  RF Band         : %s\r\n", &info_3[0]);
                    comm_pc.printf("  Application     : %s\r\n", &info_4[0]);
                    break;

                default:
                    return;                                                                        
            }

        }
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: DevMgmt_DeviceStatus_Rsp
//
//  @brief  Internal method to show device status of NSL01.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::DevMgmt_DeviceStatus_Rsp(TWiMOD_HCI_Message*  rxMessage)
{
    //--Internal variables and constants
    char RTC_string[20];
    UINT16 result_UINT16;
    UINT32 result_UINT32;

    LoRaWAN_ShowResponse("device status response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    if (rxMessage->Payload[0] == DEVMGMT_STATUS_OK)
    {                       
        //****************************************
        //--Get System Tick resolution
        //****************************************
        UINT8 system_tick_res = rxMessage->Payload[1];
        comm_pc.printf("  System Tick Res.: %u ms\r\n",system_tick_res);

        //****************************************
        //--Get System Ticks since last 
        //--start-up/reset
        //****************************************

        //--Convert system ticks (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[2]));        
        comm_pc.printf("  System Ticks    : %lu\r\n",result_UINT32);        

        //****************************************
        //--Get Target Time 
        //****************************************

        //--Convert target time (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[6]));                
        convert_RTC_message(result_UINT32,RTC_string);
        comm_pc.printf("  Target Time     : %s\r\n",RTC_string); 

        //****************************************
        //--Get NVM Status
        //****************************************

        //--Convert NVM status (ntohl-function)        
        result_UINT16 = Convert_Payload_Data_UINT16(&(rxMessage->Payload[10]));
        comm_pc.printf("  NVM Status      : 0x%04X\r\n",result_UINT16);

        //****************************************
        //--Get battery level
        //****************************************

        //--Convert battery level (ntohl-function)                        
        result_UINT16 = Convert_Payload_Data_UINT16(&(rxMessage->Payload[12]));                        
        comm_pc.printf("  Battery level   : %u mV\r\n",result_UINT16);

        //****************************************
        //--Get Extra Status (Reserved)
        //****************************************

        //--Convert extra status (ntohl-function)                        
        result_UINT16 = Convert_Payload_Data_UINT16(&(rxMessage->Payload[14]));                        
        comm_pc.printf("  Extra Status    : 0x%04x\r\n",result_UINT16);

        //****************************************
        //--Tx U-Data (# unreliable data
        //--packets transmitted)
        //****************************************

        //--Convert Tx U-data (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[16]));        
        comm_pc.printf("  Tx U-Data       : %lu\r\n", result_UINT32);                 

        //****************************************
        //--Tx C-Data (# reliable data
        //--packets transmitted)
        //****************************************

        //--Convert Tx C-data (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[20]));        
        comm_pc.printf("  Tx C-Data       : %lu\r\n", result_UINT32);                         

        //****************************************
        //--Tx Error (# of packets not
        //--transmitted due to an error)
        //****************************************

        //--Convert Tx error (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[24]));        
        comm_pc.printf("  Tx ERROR        : %lu\r\n", result_UINT32);

        //****************************************
        //--Rx1 U-Data (# of unreliable
        //--packets received in 1st window)
        //****************************************

        //--Convert Rx1 U-Data (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[28]));        
        comm_pc.printf("  Rx1 U-Data      : %lu\r\n", result_UINT32);

        //****************************************
        //--Rx1 C-Data (# of reliable
        //--packets received in 1st window)
        //****************************************

        //--Convert Rx1 C-Data (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[32]));        
        comm_pc.printf("  Rx1 C-Data      : %lu\r\n", result_UINT32);  

        //****************************************
        //--Rx1 MIC-Error (# of packets
        //--received with MIC error)
        //****************************************

        //--Convert Rx1 MIC-Error (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[36]));        
        comm_pc.printf("  Rx1 MIC-Error   : %lu\r\n", result_UINT32);  

        //****************************************
        //--Rx2 U-Data (# of unreliable
        //--packets received in 2nd window)
        //****************************************

        //--Convert Rx2 U-Data (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[40]));        
        comm_pc.printf("  Rx2 U-Data      : %lu\r\n", result_UINT32);

        //****************************************
        //--Rx2 C-Data (# of reliable
        //--packets received in 2nd window)
        //****************************************

        //--Convert Rx2 C-Data (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[44]));        
        comm_pc.printf("  Rx2 C-Data      : %lu\r\n", result_UINT32);

        //****************************************
        //--Rx2 MIC-Error (# of packets
        //--received in 2nd window with MIC error)
        //****************************************

        //--Convert Rx2 MIC-Error (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[48]));        
        comm_pc.printf("  Rx2 MIC-Error   : %lu\r\n", result_UINT32);                                      

        //****************************************
        //--Tx Join (# of join request
        //--radio packets transmitted)
        //****************************************

        //--Convert Tx Join (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[52]));        
        comm_pc.printf("  Tx Join         : %lu\r\n", result_UINT32);                                      

        //****************************************
        //--Rx Accept (# of join accept
        //--radio packets received)
        //****************************************

        //--Convert Rx Accept (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[56]));        
        comm_pc.printf("  Rx Accept       : %lu\r\n", result_UINT32);  
    }

    flag_LoRaWAN_response = true;  
}

//------------------------------------------------------------------------------
//
//  Private method: DevMgmt_RTC_Rsp
//
//  @brief  Internal method to show RTC information of NSL01
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::DevMgmt_RTC_Rsp(TWiMOD_HCI_Message*  rxMessage)
{
    //--Internal variables and constants
    char RTC_string[20];
    UINT32 result_UINT32;

    LoRaWAN_ShowResponse("RTC get response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    if (rxMessage->Payload[0] == DEVMGMT_STATUS_OK)
    {                   
        //****************************************
        //--Get RTC info
        //****************************************

        //--Convert RTC time (ntohl-function)        
        result_UINT32 = Convert_Payload_Data_UINT32(&(rxMessage->Payload[1]));                
        convert_RTC_message(result_UINT32,RTC_string);
        comm_pc.printf("  RTC Time : %s\r\n",RTC_string);
    }

    flag_LoRaWAN_response = true;         
}

//------------------------------------------------------------------------------
//
//  Private method: DevMgmt_Get_Alarm_Rsp
//
//  @brief  Internal method to show RTC alarm information of NSL01.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::DevMgmt_Get_Alarm_Rsp(TWiMOD_HCI_Message*  rxMessage)
{
    //--Internal variables and constants
    UINT8   alarm_indicator;
    UINT8   periodical_alarm;
    UINT8   hour;
    UINT8   minutes;
    UINT8   seconds;  

    LoRaWAN_ShowResponse("RTC alarm get response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    if (rxMessage->Payload[0] == DEVMGMT_STATUS_OK)
    {                   
        //****************************************
        //--Get alarm data
        //****************************************

        //--Alarm set or not set
        alarm_indicator =  rxMessage->Payload[1];

        //--Periodical alarm
        periodical_alarm = rxMessage->Payload[2];

        //--Hour
        hour = rxMessage->Payload[3];

        //--Minutes
        minutes = rxMessage->Payload[4]; 

        //--Seconds
        seconds = rxMessage->Payload[5];

        //****************************************
        //--Show alarm data 
        //****************************************        
        if (alarm_indicator == 0x00)
        {
            comm_pc.printf("  Alarm status : No alarm set\r\n");
        }
        else if (alarm_indicator == 0x01)
        {
            comm_pc.printf("  Alarm status : Alarm set\r\n");       
        }
        else
        {
            comm_pc.printf("  Alarm status : No information\r\n");
        }
        
        if (periodical_alarm == 0x00)
        {
            comm_pc.printf("  Alarm option : Single alarm\r\n");
        }
        else if (periodical_alarm == 0x01)
        {
            comm_pc.printf("  Alarm option : Daily alarm\r\n");       
        }
        else
        {
            comm_pc.printf("  Alarm option : No information\r\n");
        }
        
        if (((int)hour >= 0) && (hour <= 23))
        {
            comm_pc.printf("  Hour         : %u\r\n",hour);
        }
        else
        {
            comm_pc.printf("  Hour         : Out of range!\r\n");
        }
        
        if (((int)minutes >= 0) && (minutes <= 59))
        {
            comm_pc.printf("  Minutes      : %u\r\n",minutes);
        }
        else
        {
            comm_pc.printf("  Minutes      : Out of range!\r\n");
        }        
                
        if (((int)seconds >= 0) && (seconds <= 59))
        {
            comm_pc.printf("  Seconds      : %u\r\n",seconds);
        }
        else
        {
            comm_pc.printf("  Seconds      : Out of range!\r\n");
        }               
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: DevMgmt_OperationMode_Rsp
//
//  @brief  Internal method to get the system operation mode of NSL01.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::DevMgmt_OperationMode_Rsp(TWiMOD_HCI_Message*  rxMessage)
{
   //--Internal variables and constants
    UINT8 op_mode;

    LoRaWAN_ShowResponse("Operation mode response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //*******************************
    //--Operation system mode
    //*******************************    
    op_mode = rxMessage->Payload[1];

    comm_pc.printf("  System operation mode: ");

    switch (op_mode)
    {
        case 0:
            comm_pc.printf("Standard application mode / default mode\r\n");
            break;

        case 1:                            
            comm_pc.printf("Reserved\r\n");
            break;        

        case 2:                            
            comm_pc.printf("Reserved\r\n");
            break;               

        case 3:                            
            comm_pc.printf("Customer mode\r\n");
            break;                       

        default:
            comm_pc.printf("Not available\r\n");                      
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Reactivate_Device_Rsp
//
//  @brief  Internal method to get the status of NSL01 after reactivation in
//          Activation by Personalization (ABP) mode with parameters in non- 
//          volatile memory.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Reactivate_Device_Rsp(TWiMOD_HCI_Message*  rxMessage)
{
    //****************************************
    //--Status byte
    //****************************************    
    LoRaWAN_ShowResponse("device reactivation response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);

    //****************************************
    //--Get device address if status ok
    //****************************************    
    if (rxMessage->Payload[0] == 0)
    {
        UINT32 address = MAKELONG(MAKEWORD(rxMessage->Payload[1],rxMessage->Payload[2]),
                                  MAKEWORD(rxMessage->Payload[3],rxMessage->Payload[4]));

        comm_pc.printf("  DeviceAddress: 0x%08X\r\n", address);
    }
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Process_JoinTxIndication
//
//  @brief  Internal event method for join network transmit indication which is
//          automatically invoked whenever the join radio message has been sent
//          to the host.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Process_JoinTxIndication(TWiMOD_HCI_Message* rxMessage)
{
    //--Check if channel info is attached
    if (rxMessage->Payload[0] == 0)
    {
        //--No channel info attached
        comm_pc.printf("\r\nJoin Tx event - Status : ok\r\n");
    }
    else if(rxMessage->Payload[0] == 1)
    {
        //--Channel info attached
        comm_pc.printf("\r\nJoin Tx event: %d with parameters:\r\n", (int)rxMessage->Payload[3]);
        comm_pc.printf("  Channel Index     : %d\r\n", (int)rxMessage->Payload[1]);
        comm_pc.printf("  Data Rate Index   : %d\r\n", (int)rxMessage->Payload[2]);
        comm_pc.printf("  TRX Power Level   : %d [dBm]\r\n", (int)rxMessage->Payload[4]);

        UINT8* rxInfo = &rxMessage->Payload[5];  
        UINT32 airtime = (UINT32)(rxInfo[3] << 24 | rxInfo[2] << 16 | rxInfo[1] << 8 | rxInfo[0]);

        comm_pc.printf("  RF message airtime: %lu [ms]\r\n", airtime);
    }
    else
    {
        //--Error: No joining Tx event triggered
        comm_pc.printf("\r\nJoin Tx event - Status : error\r\n");
    }  
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Process_JoinNetworkIndication
//
//  @brief  Internal event method to indicate the join network procedure which
//          is automatically invoked after successful reception of a server
//          join response or after expiration of a complete join process without
//          success.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Process_JoinNetworkIndication(TWiMOD_HCI_Message* rxMessage)
{
    //--Check if channel info is attached
    if (rxMessage->Payload[0] == 0)
    {
        //--Only device address attached
        UINT32 address = MAKELONG(MAKEWORD(rxMessage->Payload[1],rxMessage->Payload[2]),
                                  MAKEWORD(rxMessage->Payload[3],rxMessage->Payload[4]));

        comm_pc.printf("\r\nDevice activated with parameters:\r\n");
        comm_pc.printf("  New device address: 0x%08X\r\n", address);
    }
    else if (rxMessage->Payload[0] == 1)
    {
        //--Channel info attached
        UINT32 address = MAKELONG(MAKEWORD(rxMessage->Payload[1],rxMessage->Payload[2]),
                                  MAKEWORD(rxMessage->Payload[3],rxMessage->Payload[4]));

        comm_pc.printf("\r\nDevice activated with parameters:\r\n");
        comm_pc.printf("  New device address: 0x%08X\r\n", address);
        comm_pc.printf("  Channel Index     : %d\r\n", (int)rxMessage->Payload[5]);
        comm_pc.printf("  Data Rate Index   : %d\r\n", (int)rxMessage->Payload[6]);
        comm_pc.printf("  RSSI              : %d [dBm]\r\n", (int)rxMessage->Payload[7]);
        comm_pc.printf("  SNR               : %d [dB]\r\n", (int)rxMessage->Payload[8]);
        comm_pc.printf("  Rx Slot           : %d\r\n", (int)rxMessage->Payload[9]);
    }
    else
    {
        //--Error: Joining failed (Timeout)
        comm_pc.printf("\r\nTimout network joining:\n\r");
    }

    flag_LoRaWAN_response = true;

    //--Trigger event function
    LoRa_event(LORAWAN_EVENT_MSG_JOIN_TIMEOUT);
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_send_Udata_Rsp
//
//  @brief  Internal method to evaluate response from 'send_Udata' method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_send_Udata_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Evaluate status byte
    LoRaWAN_ShowResponse("transmit U-Data response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);

    //--Check if additional payload data exist
    if (rxMessage->Length > 1)
    {
        //--Channel is blocked by Duty Cycle: Get time [ms] remaining till 
        //--channel is available
        UINT8* rxInfo = &rxMessage->Payload[1];
        UINT32 time = (UINT32)(rxInfo[3] << 24 | rxInfo[2] << 16 | rxInfo[1] << 8 | rxInfo[0]);

        comm_pc.printf("  Channel available in: %lu [ms]\r\n", time);
    }
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Udata_Tx_Indication
//
//  @brief  Internal event method which is triggered by the host after a radio
//          packet has been sent
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Udata_Tx_Indication(TWiMOD_HCI_Message* rxMessage)
{
    //--Evaluation status and payload format
    if (rxMessage->Payload[0] == 0x00)
    {
        comm_pc.printf("\r\nU-Radio packet has been sent successfully\r\n");
        flag_LoRaWAN_response = true;
        return;
    }
    else if (rxMessage->Payload[0] == 0x01)
    {
        comm_pc.printf("\r\nU-Radio packet has been sent successfully with attached Tx channel info:\r\n");                    
    }
    else
    {
        comm_pc.printf("\r\nU-Radio packet has not been sent! Please try again\r\n");
        flag_LoRaWAN_response = true;
        return;
    }

    //--Evaluate channel info
    if (rxMessage->Length > 1)
    {
        UINT8* rxInfo = &rxMessage->Payload[1];

        comm_pc.printf("  Channel Index       : %d\r\n",rxInfo[0]);
        comm_pc.printf("  Data Rate Index     : %d\r\n",rxInfo[1]);
        comm_pc.printf("  Number of Tx packets: %d\r\n",rxInfo[2]);
        comm_pc.printf("  TRX Power Level     : %d [dBm]\r\n",rxInfo[3]);

        UINT32 airtime = (UINT32)(rxInfo[7] << 24 | rxInfo[6] << 16 | rxInfo[5] << 8 | rxInfo[4]); 
        comm_pc.printf("  RF message airtime  : %lu [ms]\r\n",airtime);                
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Udata_Rx_Indication
//
//  @brief  Internal event method which is sent to the host after reception of
//          an unreliable radio packet containing application payload.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Udata_Rx_Indication(TWiMOD_HCI_Message* rxMessage)
{
    int payloadSize = rxMessage->Length - 1;

    //--Determine payload size
    if (rxMessage->Payload[0] & 0x01)
        payloadSize -= 5;

    //--Show payload data and LoRaWAN port number
    if (payloadSize >= 1)
    {
        comm_pc.printf("\r\nU-Radio packet received at port 0x%02X with payload: ", rxMessage->Payload[1]);

        for(int i = 1; i < payloadSize; i++)
            comm_pc.printf("%02X ", rxMessage->Payload[1+i]);

        comm_pc.printf("\r\n");
    }

    //--Check if ACK for uplink packet was received
    if (rxMessage->Payload[0] & 0x02)
    {
        comm_pc.printf("  ACK for last uplink packet received: YES\r\n");
    }
    else
    {
        comm_pc.printf("  ACK for last uplink packet received: NO\r\n");
    }

    //--Check if downlink frame is pending
    if (rxMessage->Payload[0] & 0x04)
    {
        comm_pc.printf("  Downlink frame pending             : YES\r\n");
    }
    else
    {
        comm_pc.printf("  Downlink frame pending             : NO\r\n");
    }

    //--Check if Rx channel info is attached
    comm_pc.printf("  Received channel info:\r\n");

    if (rxMessage->Payload[0] & 0x01)
    {
        UINT8* rxInfo = &rxMessage->Payload[1 + payloadSize];

        comm_pc.printf("    Channel Index                    : %d\r\n",rxInfo[0]);
        comm_pc.printf("    Data Rate Index                  : %d\r\n",rxInfo[1]);
        comm_pc.printf("    RSSI                             : %d [dBm]\r\n",rxInfo[2]);
        comm_pc.printf("    SNR                              : %d [dB]\r\n",rxInfo[3]);
        comm_pc.printf("    Rx Slot                          : %d\r\n",rxInfo[4]);
    }
    else
    {
        comm_pc.printf("    No data available\r\n");
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_send_Cdata_Rsp
//
//  @brief  Internal method to evaluate response from 'send_Cdata' method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_send_Cdata_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Evaluate status byte
    LoRaWAN_ShowResponse("transmit C-Data response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);

    //--Check if additional payload data exist
    if (rxMessage->Length > 1)
    {
        //--Channel is blocked by Duty Cycle: Get time [ms] remaining till 
        //--channel is available
        UINT8* rxInfo = &rxMessage->Payload[1];
        UINT32 time = (UINT32)(rxInfo[3] << 24 | rxInfo[2] << 16 | rxInfo[1] << 8 | rxInfo[0]);

        comm_pc.printf("  Channel available in: %lu [ms]\r\n", time);
    }
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Cdata_Tx_Indication
//
//  @brief  Internal event method which is invoked by the host after a radio
//          packet has been sent or if the retransmission procedure finishes
//          without success.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Cdata_Tx_Indication(TWiMOD_HCI_Message* rxMessage)
{
   //--Evaluation status and payload format
    if (rxMessage->Payload[0] == 0x00)
    {
        comm_pc.printf("\r\nC-Radio packet has been sent successfully\r\n");
        return;
    }
    else if (rxMessage->Payload[0] == 0x01)
    {
        comm_pc.printf("\r\nC-Radio packet has been sent successfully with attached Tx channel info:\r\n");
    }
    else if (rxMessage->Payload[0] == 0x02)
    {
        comm_pc.printf("\r\nError: No ACK received and maximum number of retransmissions reached:\r\n");
        flag_LoRaWAN_response = true;
        
        //--Trigger event function
        LoRa_event(LORAWAN_EVENT_MSG_CDATA_MAX_RETRANSMISSION);
        return;
    }
    else if (rxMessage->Payload[0] == 0x04)
    {
        comm_pc.printf("\r\nError: Maximum payload size exceeded for current data rate\r\n");
        flag_LoRaWAN_response = true;

        //--Trigger callback function
        LoRa_event(LORAWAN_EVENT_MSG_CDATA_PAYLOAD_EXCEEDED);        
        return;
    }
    else
    {
        comm_pc.printf("\r\nC-Radio packet has not been sent! Please try again\r\n");
        flag_LoRaWAN_response = true;
        return;
    }        

    //--Evaluate channel info
    if (rxMessage->Length > 1)
    {
        UINT8* rxInfo = &rxMessage->Payload[1];

        comm_pc.printf("  Channel Index       : %d\r\n",rxInfo[0]);
        comm_pc.printf("  Data Rate Index     : %d\r\n",rxInfo[1]);
        comm_pc.printf("  Number of Tx packets: %d\r\n",rxInfo[2]);
        comm_pc.printf("  TRX Power Level     : %d [dBm]\r\n",rxInfo[3]);

        UINT32 airtime = (UINT32)(rxInfo[7] << 24 | rxInfo[6] << 16 | rxInfo[5] << 8 | rxInfo[4]); 
        comm_pc.printf("  RF message airtime  : %lu [ms]\r\n",airtime);                       
    }
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Cdata_Rx_Indication
//
//  @brief  Internal event message which is sent to the host after reception of
//          a reliable radio packet containing application payload. The device
//          will acknowledge the reception with a set Ack-Bit in the next
//          reliable/unreliable uplink radio packet to the network server.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Cdata_Rx_Indication(TWiMOD_HCI_Message* rxMessage)
{
    int payloadSize = rxMessage->Length - 1;

    //--Determine payload size
    if (rxMessage->Payload[0] & 0x01)
        payloadSize -= 5;

    //--Show payload data and LoRaWAN port number
    if (payloadSize >= 1)
    {
        comm_pc.printf("C-Radio packet received at port: 0x%02X with payload: ", rxMessage->Payload[1]);
        
        for(int i = 1; i < payloadSize; i++)
            comm_pc.printf("%02X ", rxMessage->Payload[1+i]);
        
        comm_pc.printf("\r\n");
    }

    //--Check if ACK for uplink packet was received
    if (rxMessage->Payload[0] & 0x02)
    {
        comm_pc.printf("  ACK for last uplink packet received: YES\r\n");
    }
    else
    {
        comm_pc.printf("  ACK for last uplink packet received: NO\r\n");
    }

    //--Check if downlink frame is pending
    if (rxMessage->Payload[0] & 0x04)
    {
        comm_pc.printf("  Downlink frame pending             : YES\r\n");
    }
    else
    {
        comm_pc.printf("  Downlink frame pending             : NO\r\n");
    }

    //--Check if Rx channel info is attached
    comm_pc.printf("  Received channel info:\r\n");

    if (rxMessage->Payload[0] & 0x01)
    {
        UINT8* rxInfo = &rxMessage->Payload[1 + payloadSize];

        comm_pc.printf("    Channel Index                    : %d\r\n",rxInfo[0]);
        comm_pc.printf("    Data Rate Index                  : %d\r\n",rxInfo[1]);
        comm_pc.printf("    RSSI                             : %d [dBm]\r\n",rxInfo[2]);
        comm_pc.printf("    SNR                              : %d [dB]\r\n",rxInfo[3]);
        comm_pc.printf("    Rx Slot                          : %d\r\n",rxInfo[4]);
    }
    else
    {
        comm_pc.printf("    No data available\r\n");
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Rx_NoData_Indication (not used at the moment)
//
//  @brief  Internal event method which is sent to the host in case no expected
//          confirmation or data has been received as a result of prior reliable
//          uplink radio packet.
//
//  @note  Method is not used at the moment!
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Rx_NoData_Indication(TWiMOD_HCI_Message* rxMessage)
{
    //--Check if error code is attached
    if (rxMessage->Payload[0] & 0x02)
    {
        comm_pc.printf("No Rx data indication event with error code:\r\n");
    }
    else
    {
        comm_pc.printf("No Rx data indication event without error code!\r\n");
        
        flag_LoRaWAN_response = true;
        return;
    }

    //--Show received error code
    if (rxMessage->Payload[1] & 0x01)
    {
        comm_pc.printf("  Wrong MType received\r\n");
    }

    if (rxMessage->Payload[1] & 0x02)
    {
        comm_pc.printf("  Wrong Device Address received\r\n");
    }

    if (rxMessage->Payload[1] & 0x04)
    {
        comm_pc.printf("  Wrong MIC received\r\n");
    }

    if (rxMessage->Payload[1] & 0x08)
    {
        comm_pc.printf("  Unexpected FCnt received\r\n");
    }

    if (rxMessage->Payload[1] & 0x10)
    {
        comm_pc.printf("  Wrong MAC commands received\r\n");
    }

    if (rxMessage->Payload[1] & 0x20)
    {
        comm_pc.printf("  Wrong downlink received\r\n");
    }

    if (rxMessage->Payload[1] & 0x40)
    {
        comm_pc.printf("  Expected ACK missing\r\n");
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Set_RadioStack_Config_Rsp
//
//  @brief  Internal method to evaluate response from 'set_radio_stack_config'
//          method
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Set_RadioStack_Config_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Internal variables and constants
    UINT8   error_code;
    UINT8   flag_data_rate;
    UINT8   flag_Tx_power;
    UINT8   flag_band_index;

    //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("radio stack config response", LoRaWAN_StatusStrings, rxMessage->Payload[0]);

    //****************************************
    //--Optional Payload (Error Codes)
    //****************************************
    if (rxMessage->Length > 1)
    {
        //--Get complete error code
        error_code = rxMessage->Payload[1];

        //--Extract flag of data rate
        flag_data_rate = (error_code & 0x01);

        //--Flag correct TX power level
        flag_Tx_power = (error_code & 0x02);

        //--Flag correct band index
        flag_band_index = (error_code & 0x20);

        //--Show error codes
        comm_pc.printf("  Error code for current radio stack configuration:\r\n");

        if (flag_data_rate)
            comm_pc.printf("    Data Rate     : Wrong\r\n");

        if (flag_Tx_power)
            comm_pc.printf("    TX Power Level: Wrong\r\n");

        if (flag_band_index)
            comm_pc.printf("    Band Index    : Wrong\r\n");
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Get_RadioStack_Config_Rsp
//
//  @brief  Internal method to evaluate response from 'get_radio_stack_config'
//          method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Get_RadioStack_Config_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Internal variables and constants
    UINT8   data_rate_index;
    UINT8   Tx_power;
    UINT8   options;
    UINT8   adaptive_data_rate;
    UINT8   duty_cycle_control;
    UINT8   class_C_support;
    UINT8   RF_packet_format;
    UINT8   forwarding;
    UINT8   power_saving;
    UINT8   number_retransmissions;
    UINT8   band_index;
    UINT8   header_MAC_cmd_capacity;

    //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("radio stack config response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //****************************************
    //--Get data rate index
    //****************************************
    comm_pc.printf("  Modulation parameters      : ");

    data_rate_index = rxMessage->Payload[1];

    switch (data_rate_index)
    {
        case 0:
            comm_pc.printf("SF12 / 125 kHz / 250 bps\r\n");
            break;

        case 1:
            comm_pc.printf("SF11 / 125 kHz / 440 bps\r\n");
            break;

        case 2:
            comm_pc.printf("SF10 / 125 kHz / 980 bps\r\n");
            break;

        case 3:
            comm_pc.printf("SF9 / 125 kHz / 1760 bps\r\n");
            break;

        case 4:
            comm_pc.printf("SF8 / 125 kHz / 3125 bps\r\n");
            break;

        case 5:
            comm_pc.printf("SF7 / 125 kHz / 5470 bps\r\n");
            break;

        case 6:
            comm_pc.printf("SF7 / 250 kHz / 11000 bps\r\n");
            break;

        default:
            comm_pc.printf("Not specified!\r\n");
            break;     
    }

    //****************************************
    //--Get Tx power level (EIRP)
    //****************************************
    Tx_power = rxMessage->Payload[2];
    comm_pc.printf("  TX power (EIRP)            : %u dBm\r\n",Tx_power);

    //****************************************
    //--Options
    //****************************************
    options =  rxMessage->Payload[3];

    //--Adaptive data rate
    adaptive_data_rate = (options & 0x01);

    //--Duty cycle control
    duty_cycle_control = (options & 0x02);

    //--Class C support
    class_C_support = (options & 0x04);

    //--RF packet output format
    RF_packet_format = (options & 0x40);

    //--Rx MAC command forwarding
    forwarding = (options & 0x80);

    comm_pc.printf("  Options: \r\n");

    if (adaptive_data_rate)
        comm_pc.printf("    Adaptive Data Rate       : Enabled\r\n");
    else
        comm_pc.printf("    Adaptive Data Rate       : Disabled\r\n");

    if (duty_cycle_control)
        comm_pc.printf("    Duty Cycle Control       : Enabled\r\n");
    else
        comm_pc.printf("    Duty Cycle Control       : Disabled\r\n");

    if (class_C_support)
        comm_pc.printf("    Class C-Support          : Enabled\r\n");
    else
        comm_pc.printf("    Class C-Support          : Disabled\r\n");

    if (forwarding)
        comm_pc.printf("    Rx MAC command forwarding: Enabled\r\n");
    else
        comm_pc.printf("    Rx MAC command forwarding: Disabled\r\n");

    if (RF_packet_format)
        comm_pc.printf("    RF packet output format  : Extended\r\n");
    else
        comm_pc.printf("    RF packet output format  : Standard\r\n");

    //****************************************
    //--Power saving mode
    //****************************************
    power_saving =  rxMessage->Payload[4];

    if (power_saving == 0x00)
        comm_pc.printf("  Power saving mode          : OFF\r\n");
    else if (power_saving == 0x01)
        comm_pc.printf("  Power saving mode          : ON (Automatic)\r\n");
    else
        comm_pc.printf("  Power saving mode          : Not specified\r\n");

    //****************************************
    //--Number of retransmissions
    //****************************************
    number_retransmissions =  rxMessage->Payload[5];
    comm_pc.printf("  Number of retransmissions  : %u\r\n",number_retransmissions);

    //****************************************
    //--Band Index
    //****************************************
    band_index =  rxMessage->Payload[6];

    comm_pc.printf("  Band Selection             : ");

    switch (band_index)
    {
        case 1:
            comm_pc.printf("EU 868 MHz\r\n");
            break;

        case 2:
            comm_pc.printf("US 915 MHz\r\n");
            break;

        case 3:
            comm_pc.printf("IN 865 MHz\r\n");
            break;

        case 129:
            comm_pc.printf("EU 868 MHz (RX2: SF9)\r\n");
            break;

        case 131:
            comm_pc.printf("IN 865 MHz (RX2: SF8)\r\n");
            break;

        default:
            comm_pc.printf("Not supported\r\n");
    }

    //****************************************
    //--Header MAC command capacity
    //****************************************
    header_MAC_cmd_capacity =  rxMessage->Payload[7];
    comm_pc.printf("  Header MAC cmd capacity    : %u\r\n",header_MAC_cmd_capacity);

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Get_FreqBand_Rsp
//
//  @brief  Internal method to evaluate response from 'get_freqband_info'
//          method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Get_FreqBand_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Internal variables and constants
    int i = 0;
    UINT8   number_of_digits;
    UINT8   number_of_bands;
    UINT8   band_index;
    UINT8   EIRP_band_index;

   //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("frequency band response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //--Check if additional payload data is available
    if (rxMessage->Length < 4)
    {
        flag_LoRaWAN_response = true;
        return;
    }

    //--Determine number of frequency bands
    number_of_bands = (rxMessage->Length - 3) / 2;

    //--Show frequency band and corresponding max. EIRP
    for (i= 0; i < number_of_bands+1; i++)
    {
        //****************************************
        //--Band Index n
        //****************************************
        band_index = rxMessage->Payload[1+2*i];
        number_of_digits = 0;

        //--Adapt string output length
        if (band_index < 10)
        {
            comm_pc.printf("  Band Index %u              : ",band_index);
            number_of_digits = 1;
        }
        else if (band_index >= 10 && band_index < 100)
        {
            comm_pc.printf("  Band Index %u             : ",band_index);
            number_of_digits = 2;
        }
        else if (band_index >= 100)
        {
            comm_pc.printf("  Band Index %u            : ",band_index);
            number_of_digits = 3;
        }

        switch (band_index)
        {
            case 1:
                comm_pc.printf("EU 868 MHz\r\n");
                break;

            case 2:
                comm_pc.printf("US 915 MHz\r\n");
                break;

            case 3:
                comm_pc.printf("IN 865 MHz\r\n");
                break;

            case 129:
                comm_pc.printf("EU 868 MHz (RX2: SF9)\r\n");
                break;

            case 131:
                comm_pc.printf("IN 865 MHz (RX2: SF8)\r\n");
                break;

            default:
                comm_pc.printf("Not supported\r\n");
        }

        //****************************************
        //--Max. EIRP Band Index n
        //****************************************
        EIRP_band_index = rxMessage->Payload[2+2*i];

        //--Adapt string output length
        if ((EIRP_band_index < 10) && (number_of_digits == 1))
            comm_pc.printf("  Max. EIRP Band Index %u     : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index < 10) && (number_of_digits == 2))
            comm_pc.printf("  Max. EIRP Band Index %u    : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index < 10) && (number_of_digits == 3))
            comm_pc.printf("  Max. EIRP Band Index %u   : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index >= 10 && EIRP_band_index < 100) && (number_of_digits == 1))
            comm_pc.printf("  Max. EIRP Band Index %u    : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index >= 10 && EIRP_band_index < 100) && (number_of_digits == 2))
            comm_pc.printf("  Max. EIRP Band Index %u   : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index >= 10 && EIRP_band_index < 100) && (number_of_digits == 3))
            comm_pc.printf("  Max. EIRP Band Index %u  : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index >= 100) && (number_of_digits == 1))
            comm_pc.printf("  Max. EIRP Band Index %u   : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index >= 100) && (number_of_digits == 2))
            comm_pc.printf("  Max. EIRP Band Index %u  : %u\r\n",band_index,EIRP_band_index);
        else if ((EIRP_band_index >= 100) && (number_of_digits == 3))
            comm_pc.printf("  Max. EIRP Band Index %u : %u\r\n",band_index,EIRP_band_index);
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Get_DeviceEUI_Rsp
//
//  @brief  Internal method to evaluate response from 'get_EUI' method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Get_DeviceEUI_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Internal variables and constants
    UINT8*  dstPtr;
    UINT8   index;
    UINT8   device_EUI[EUI_LEN];

    //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("device EUI response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //****************************************
    //--Device EUI
    //****************************************
    dstPtr  = &rxMessage->Payload[1];
    index   = 0;

    comm_pc.printf("  Device EUI: ");

    //--Copy bytes to array and show device EUI
    while(index < EUI_LEN)
    {
        device_EUI[index]= *dstPtr;

        if (index < EUI_LEN - 1)
            comm_pc.printf("%02X-",device_EUI[index]);
        else
            comm_pc.printf("%02X",device_EUI[index]);

        //--Update pointer and indices
        dstPtr++;
        index++;
    }

    comm_pc.printf("\r\n");

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Get_Custom_Config_Rsp
//
//  @brief  Internal method to evaluate response from 'get_custom_configuration'
//          method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Get_Custom_Config_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Internal variables and constants
    UINT8  RF_Gain;

    //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("get custom configuration response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //****************************************
    //--Tx power offset
    //****************************************
    RF_Gain = rxMessage->Payload[1];

    //--Display result
    comm_pc.printf("  RF Gain: %d dBd\r\n", (INT8)RF_Gain);

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Network_Status_Rsp
//
//  @brief  Internal method to evaluate response from 'get_network_status'
//          method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Network_Status_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Internal variables and constants
    UINT8   network_status;
    UINT32  device_address;
    UINT8   data_rate_index;
    UINT8   Tx_power;
    UINT8   max_payload_size;

    //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("network status response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //****************************************
    //--Network status
    //****************************************
    network_status = rxMessage->Payload[1];

    comm_pc.printf("  Network status      : ");

    switch (network_status)
    {
        case 0x00:
            comm_pc.printf("device inactive\r\n");
            break;

        case 0x01:
            comm_pc.printf("device active (ABP)\r\n");
            break;

        case 0x02:
            comm_pc.printf("device active (OTAA)\r\n");
            break;

        case 0x03:
            comm_pc.printf("device joining (OTAA)\r\n");
            break;

        default:
            comm_pc.printf("not specified\r\n");
            break;
    }

    //--Check if additional data is available
    if (rxMessage->Length < 3)
    {
        flag_LoRaWAN_response = true;
        return;
    }

    //****************************************
    //--Device Address
    //****************************************

    //--Convert device address (ntohl-function)
    device_address = Convert_Payload_Data_UINT32(&(rxMessage->Payload[2]));
    comm_pc.printf("  Device Address      : 0x%08x\r\n",device_address);

    //****************************************
    //--Data Rate Index
    //****************************************
    comm_pc.printf("  Data Rate Index     : ");

    data_rate_index = rxMessage->Payload[6];

    switch (data_rate_index)
    {
        case 0:
            comm_pc.printf("SF12 / 125 kHz / 250 bps\r\n");
            break;

        case 1:
            comm_pc.printf("SF11 / 125 kHz / 440 bps\r\n");
            break;

        case 2:
            comm_pc.printf("SF10 / 125 kHz / 980 bps\r\n");
            break;

        case 3:
            comm_pc.printf("SF9 / 125 kHz / 1760 bps\r\n");
            break;

        case 4:
            comm_pc.printf("SF8 / 125 kHz / 3125 bps\r\n");
            break;

        case 5:
            comm_pc.printf("SF7 / 125 kHz / 5740 bps\r\n");
            break;

        case 6:
            comm_pc.printf("SF7 / 250 kHz / 11000 bps\r\n");
            break;

        default:
            comm_pc.printf("Not specified!\r\n");
            break;
    }

    //****************************************
    //--Tx power level (EIRP)
    //****************************************
    Tx_power = rxMessage->Payload[7];
    comm_pc.printf("  Power level (EIRP)  : %u dBm\r\n",Tx_power);

    //****************************************
    //--Maximum payload size
    //****************************************
    max_payload_size = rxMessage->Payload[8];
    comm_pc.printf("  Maximum payload size: %u bytes\r\n",max_payload_size);

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Get_TxPower_Limit_Config_Rsp
//
//  @brief  Internal method to evaluate response from
//          'get_PowerLimit_configuration' method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Get_TxPower_Limit_Config_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //--Internal variables and constants
    int i = 0;
    UINT8   number_of_bands;
    UINT8   sub_band_index;
    UINT8   TxPower_limit_flag;
    UINT8   TxPower_limit;

    //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("Tx power limit configuration response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //--Check if additional payload data is available
    if (rxMessage->Length < 2)
    {
        flag_LoRaWAN_response = true;
        return;
    }

    //--Determine number of frequency bands
    number_of_bands = (rxMessage->Length - 1) / 3;

   //--Show frequency sub-band and corresponding Tx power limit
    for (i= 0; i < number_of_bands; i++)
    {
        //****************************************
        //--Sub-Band Index n
        //****************************************
        sub_band_index = rxMessage->Payload[1+3*i];

        switch (sub_band_index)
        {
            case 0:
                comm_pc.printf("  Sub-Band Index %u:\r\n",sub_band_index);
                comm_pc.printf("    Frequency Range                    : 863 MHz to 865 MHz\r\n");
                comm_pc.printf("    Duty Cycle                         : 0.1%%\r\n");
                break;

            case 1:
                comm_pc.printf("  Sub-Band Index %u:\r\n",sub_band_index);
                comm_pc.printf("    Frequency Range                    : 865 MHz to 868 MHz\r\n");
                comm_pc.printf("    Duty Cycle                         : 1%%\r\n");
                break;

            case 2:
                comm_pc.printf("  Sub-Band Index %u:\r\n",sub_band_index);
                comm_pc.printf("    Frequency Range                    : 868 MHz to 868.6 MHz\r\n");
                comm_pc.printf("    Duty Cycle                         : 1%%\r\n");
                break;

            case 3:
                comm_pc.printf("  Sub-Band Index %u:\r\n",sub_band_index);
                comm_pc.printf("    Frequency Range                    : 868.7 MHz to 869.2 MHz\r\n");
                comm_pc.printf("    Duty Cycle                         : 0.1%%\r\n");
                break;

            case 4:
                comm_pc.printf("  Sub-Band Index %u:\r\n",sub_band_index);
                comm_pc.printf("    Frequency Range                    : 869.4 MHz to 869.65 MHz\r\n");
                comm_pc.printf("    Duty Cycle                         : 10%%\r\n");
                break;

            case 5:
                comm_pc.printf("  Sub-Band Index %u:\r\n",sub_band_index);
                comm_pc.printf("    Frequency Range                    : 869.7 MHz to 870 MHz\r\n");
                comm_pc.printf("    Duty Cycle                         : 1%%\r\n");
                break;
        }

        //****************************************
        //--Tx power limit flag for Sub-Band n
        //****************************************
        TxPower_limit_flag = rxMessage->Payload[2+3*i];

        if (TxPower_limit_flag == 0)
            comm_pc.printf("    Tx Power limit for sub-band %u      : deactivated\r\n",sub_band_index);
        else
            comm_pc.printf("    Tx Power limit for sub-band %u      : activated\r\n",sub_band_index);

        //****************************************
        //--Tx power limit for Sub-Band n
        //****************************************
        TxPower_limit = rxMessage->Payload[3+3*i];

        comm_pc.printf("    Tx Power limit value for sub-band %u: %u dBm\r\n",sub_band_index,TxPower_limit);
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_Set_TxPower_Limit_Config_Rsp
//
//  @brief  Internal method to evaluate response from
//          'set_PowerLimit_configuration' method.
//
//  @param TWiMOD_HCI_Message* : Rx message struct in WiMOD_HCI_Layer.h
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_Set_TxPower_Limit_Config_Rsp(TWiMOD_HCI_Message* rxMessage)
{
    //****************************************
    //--Status byte
    //****************************************
    LoRaWAN_ShowResponse("Tx power limit configuration response", DeviceMgmt_StatusStrings, rxMessage->Payload[0]);

    //--Check if additional payload data is available
    if (rxMessage->Length < 2)
    {
        flag_LoRaWAN_response = true;
        return;
    }

    //****************************************
    //--Optional error code
    //****************************************

    //--Check if sub-band index was correct
    if (rxMessage->Payload[1] & 0x01)
    {
        comm_pc.printf("  Sub-Band index: wrong\r\n");
    }
    else
    {
        comm_pc.printf("  Sub-Band index: correct\r\n");
    }

    //--Check if Tx power limit was correct
    if (rxMessage->Payload[1] & 0x04)
    {
        comm_pc.printf("  Tx power value: wrong\r\n");
    }
    else
    {
        comm_pc.printf("  Tx power value: correct\r\n");
    }

    flag_LoRaWAN_response = true;
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_ShowResponse
//
//  @brief  Internal method to show status response as human readable string.
//
//  @param message_string : Message string to display
//  @param statusTable : Device management status strings
//  @param statusID : Status identifier
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_ShowResponse(const char* message_string, const TIDString* statusTable, UINT8 statusID)
{
    while(statusTable->String)
    {
        if (statusTable->ID == statusID)
        {
            //--Display status information
            comm_pc.printf(message_string);
            comm_pc.printf(" - Status(0x%02X) : ", statusID);
            comm_pc.printf(statusTable->String);
            comm_pc.printf("\n\r");
            
            //--Check status information
            if (statusID != DEVMGMT_STATUS_OK)
            {
                //--Stop waiting process due to response from LoRa module
                flag_LoRaWAN_response = true;
            }
            
            return;
        }

        statusTable++;
    }
}

//------------------------------------------------------------------------------
//
//  Private method: convert_RTC_message
//
//  @brief  Internal help method to convert RTC data.
//
//  @param RTC_data : RTC data in UINT32 format
//  @param *RTC_string : RTC data as string
//
//------------------------------------------------------------------------------

void NSL01::convert_RTC_message(UINT32 RTC_data, char *RTC_string)
{
    //--Internal variables and constants
    char seconds_str[3];
    char minutes_str[3];
    char hours_str[3];
    char days_str[3];
    char months_str[3];
    UINT8 seconds, minutes, months, hours, days;
    UINT16 year;
    
    //--Get RTC data
    seconds = (RTC_data & 0x3f);                //0b00000000000000000000000000111111;
    minutes = (RTC_data & 0xfc0)      >> 6;     //0b00000000000000000000111111000000;        
    months  = (RTC_data & 0xf000)     >> 12;    //0b00000000000000001111000000000000;
    hours   = (RTC_data & 0x1f0000)   >> 16;    //0b00000000000111110000000000000000;
    days    = (RTC_data & 0x3e00000)  >> 21;    //0b00000011111000000000000000000000;
    year    = (RTC_data & 0xfc000000) >> 26;    //0b11111100000000000000000000000000; 
    
    //--Add 2000 to year according to specification
    year = year + 2000;
            
    //--Display information for debugging
    #ifdef DEBUG
        comm_pc.printf("\r\n Year: %u\r\n",year);    
        comm_pc.printf(" Month: %u\r\n",months);
        comm_pc.printf(" Day: %u\r\n",days);
        comm_pc.printf(" Hours: %u\r\n",hours);
        comm_pc.printf(" Minutes: %u\r\n",minutes);
        comm_pc.printf(" Seconds: %lu\r\n",seconds);
    #endif
    
    //--Convert seconds: UINT8 -> char
    if (seconds < 10)
    {
        //--Add a zero as first character
        sprintf(seconds_str, "0%u",seconds);
    }
    else
    {
        sprintf(seconds_str, "%u",seconds);
    }
    
    //--Convert minutes: UINT8 -> char
    if (minutes < 10)
    {
        //--Add a zero as first character
        sprintf(minutes_str, "0%u",minutes);
    }
    else
    {
        sprintf(minutes_str, "%u",minutes);
    }
    
    //--Convert hours: UINT8 -> char
    if (hours < 10)
    {
        //--Add a zero as first character
        sprintf(hours_str, "0%u",hours);
    }
    else
    {
        sprintf(hours_str, "%u",hours);
    }
    
    //--Convert days: UINT8 -> char
    if (days < 10)
    {
        //--Add a zero as first character
        sprintf(days_str, "0%u",days);
    }
    else
    {
        sprintf(days_str, "%u",days);
    }
    
    //--Convert months: UINT8 -> char
    if (months < 10)
    {
        //--Add a zero as first character
        sprintf(months_str, "0%u",months);
    }
    else
    {
        sprintf(months_str, "%u",months);
    }                 
    
    //--Convert data to a final string
    sprintf(RTC_string, "%u-%s-%s %s:%s:%s\0", year,months_str,days_str,hours_str,minutes_str,seconds_str);
    
    //--Show final string
    #ifdef DEBUG
        comm_pc.printf("RTC: %s\r\n",&RTC_string[0]); 
    #endif
}

//------------------------------------------------------------------------------
//
//  Private method: Convert_Payload_Data_UINT16
//
//  @brief  Internal help method for data conversion (UINT8 array in 
//          UINT16 value).
//
//  @param *Payload : UINT8 array
//
//------------------------------------------------------------------------------

UINT16 NSL01::Convert_Payload_Data_UINT16(UINT8 *Payload)
{
    //--Temporary pointer
    UINT8 *dstPtr = Payload;
        
    //--Get first byte value from UINT8 array
    UINT8 tmp_data_0 = *dstPtr;
    
    //--Get second byte value from UINT8 array
    dstPtr++;
    UINT8 tmp_data_1 = *dstPtr;
    
    //--Convert data (LSB first)
    UINT16 result_conversion = (UINT16)(tmp_data_1 << 8 | tmp_data_0); 
                    
    return  result_conversion;    
}

//------------------------------------------------------------------------------
//
//  Private method: Convert_Payload_Data_UINT32
//
//  @brief  Internal help method for data conversion (UINT8 array in 
//          UINT32 value).
//
//  @param *Payload : UINT8 array
//
//------------------------------------------------------------------------------

UINT32 NSL01::Convert_Payload_Data_UINT32(UINT8 *Payload)
{
    //--Temporary pointer
    UINT8 *dstPtr = Payload;
        
    //--Get first byte value from UINT8 array
    UINT8 tmp_data_0 = *dstPtr;
    
    //--Get second byte value from UINT8 array
    dstPtr++;
    UINT8 tmp_data_1 = *dstPtr;
    
    //--Get third byte value from UINT8 array
    dstPtr++;
    UINT8 tmp_data_2 = *dstPtr;    
    
    //--Get fourth byte value from UINT8 array
    dstPtr++;
    UINT8 tmp_data_3 = *dstPtr;    
    
    //--Convert data (LSB first)
    UINT32 result_conversion = (UINT32)(tmp_data_3 << 24 | tmp_data_2 << 16 | tmp_data_1 << 8 | tmp_data_0);      
                    
    return  result_conversion;    
}

//------------------------------------------------------------------------------
//
//  Private method: LoRaWAN_wait
//
//  @brief  Internal method to wait until response from LoRa module.
//
//------------------------------------------------------------------------------

void NSL01::LoRaWAN_wait(void)
{    
    while(1)
    {
        if (flag_LoRaWAN_response)
        {
            break;           
        }
        
        wait_us(100);
    }
    
    flag_LoRaWAN_response = false;
}

//------------------------------------------------------------------------------
// end of file
//------------------------------------------------------------------------------