#include "mbed.h"
#include "main.h"
#include "sx1276-hal.h"
#include "WakeUp.h"
//#include "debug.h"

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1

/* Set this flag to '1' to use the LoRa modulation or to '0' to use FSK modulation */
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA

#define RF_FREQUENCY                               471900000 // Hz
#define TX_OUTPUT_POWER                            20        // 14 dBm

#if USE_MODEM_LORA == 1

#define LORA_BANDWIDTH                              1         // [0: 125 kHz,
//  1: 250 kHz,
//  2: 500 kHz,
//  3: Reserved]
#define LORA_SPREADING_FACTOR                      12         // [SF7..SF12]
#define LORA_CODINGRATE                             1         // [1: 4/5,
//  2: 4/6,
//  3: 4/7,
//  4: 4/8]
#define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
#define LORA_SYMBOL_TIMEOUT                         5         // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false
#define LORA_NB_SYMB_HOP                            4
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

#elif USE_MODEM_FSK == 1

#define FSK_FDEV                                    25000     // HzY
#define FSK_DATARATE                                19200     // bps
#define FSK_BANDWIDTH                               50000     // Hz
#define FSK_AFC_BANDWIDTH                           83333     // Hz
#define FSK_PREAMBLE_LENGTH                         5         // Same for Tx and Rx
#define FSK_FIX_LENGTH_PAYLOAD_ON                   false
#define FSK_CRC_ENABLED                             true

#else
#error "Please define a modem in the compiler options."
#endif

#define RX_TIMEOUT_VALUE                                3000000   // in us
#define BUFFER_SIZE                                     32        // Define the payload size here

#if( defined ( TARGET_KL25Z ) || defined ( TARGET_LPC11U6X ) )
DigitalOut led(LED2);
#else
DigitalOut led(LED1);
#endif
DigitalOut TXIO(PA_1);
/*
 *  Global variables declarations
 */
typedef enum {
    LOWPOWER = 0,
    IDLE,

    RX,
    RX_TIMEOUT,
    RX_ERROR,

    TX,
    TX_TIMEOUT,

    CAD,
    CAD_DONE
    
} AppStates_t;

volatile AppStates_t State = LOWPOWER;

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1276MB1xAS Radio( NULL );

const uint8_t PingMsg[] = "PING";
const uint8_t PongMsg[] = "PONG";

uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];
int16_t RssiValue = 0.0;
int8_t SnrValue = 0.0;
Serial pc(PB_6, PB_7);
void srprintf(uint8_t *load,uint8_t size)
{
    pc.printf(":");
    for(int i=0; i<size; i++) {
        pc.printf("%x ",load[i]);
    }
  //  pc.printf("size:%drssi:%dsnr:%d\r\n",size,rssi,snr);
}
int main()
{
    uint8_t i;
    bool isMaster = true;
    pc.printf("Hello World !\n");
    //debug( "\n\n\r     SX1276 Ping Pong Demo Application \n\n\r" );
    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );

    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  ) {
        // debug( "Radio could not be detected!\n\r", NULL );
        pc.printf(".");
        wait( 1 );
    }
    TXIO = 1;
    //  debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1276MB1LAS ) ) , "\n\r > Board Type: SX1276MB1LAS < \n\r" );
    // debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1276MB1MAS ) ) , "\n\r > Board Type: SX1276MB1MAS < \n\r" );

    Radio.SetChannel( RF_FREQUENCY );
    pc.printf("VERSION:%x\r\n",Radio.Read( REG_FRFMSB ));
    pc.printf("VERSION:%x\r\n",Radio.Read( REG_FRFMID ));
#if USE_MODEM_LORA == 1

//    debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
//   debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                       LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                       LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, 2000000 );

    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                       LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                       LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, true );

#elif USE_MODEM_FSK == 1

//   debug("\n\n\r              > FSK Mode < \n\n\r");
    Radio.SetTxConfig( MODEM_FSK, TX_OUTPUT_POWER, FSK_FDEV, 0,
                       FSK_DATARATE, 0,
                       FSK_PREAMBLE_LENGTH, FSK_FIX_LENGTH_PAYLOAD_ON,
                       FSK_CRC_ENABLED, 0, 0, 0, 2000000 );

    Radio.SetRxConfig( MODEM_FSK, FSK_BANDWIDTH, FSK_DATARATE,
                       0, FSK_AFC_BANDWIDTH, FSK_PREAMBLE_LENGTH,
                       0, FSK_FIX_LENGTH_PAYLOAD_ON, 0, FSK_CRC_ENABLED,
                       0, 0, false, false );

#else

#error "Please define a modem in the compiler options."

#endif

    pc.printf("VERSION:%x\r\n",Radio.Read( REG_LR_MODEMCONFIG1 ));
    pc.printf("VERSION:%x\r\n",Radio.Read( REG_LR_MODEMCONFIG2 ));
    //  debug_if( DEBUG_MESSAGE, "Starting Ping-Pong loop\r\n" );

    led = 0;
 //   WakeUp::calibrate();
    Radio.Rx( RX_TIMEOUT_VALUE );
    while( 0 ) {
 //       WakeUp::set_ms(6000);
 //       deepsleep();
        switch( State ) {
            case RX:
                //     debugs((char* )Buffer,sizeof(Buffer));
                //      debug( "rssi:%d dbm\r\n",Radio.GetRssi( MODEM_LORA ));
                //      debug( "\r\n" );
                if( isMaster == true ) {
                    if( BufferSize > 0 ) {
                        if( strncmp( ( const char* )Buffer, ( const char* )PongMsg, 4 ) == 0 ) {
                            led = !led;
                            //         debug( "...Pong\r\n" );
                            pc.printf("...Pong...\r\n");
                            // Send the next PING frame
                            strcpy( ( char* )Buffer, ( char* )PingMsg );
                            // We fill the buffer with numbers for the payload
                            for( i = 4; i < BufferSize; i++ ) {
                                Buffer[i] = i - 4;
                            }
                            wait_ms( 10 );
                            Radio.Send( Buffer, BufferSize );
                        } else if( strncmp( ( const char* )Buffer, ( const char* )PingMsg, 4 ) == 0 ) {
                            // A master already exists then become a slave
                            //         debug( "...Ping\r\n" );
                            pc.printf("...Ping...\r\n");
                            led = !led;
                            isMaster = false;
                            // Send th  e next PONG frame
                            strcpy( ( char* )Buffer, ( char* )PongMsg );
                            // We fill the buffer with numbers for the payload
                            for( i = 4; i < BufferSize; i++ ) {
                                Buffer[i] = i - 4;
                            }
                            wait_ms( 10 );
                            Radio.Send( Buffer, BufferSize );
                        } else { // valid reception but neither a PING or a PONG message
                            // Set device as master ans start again
                            isMaster = true;
                            Radio.Rx( RX_TIMEOUT_VALUE );
                        }
                    }
                } else {
                    if( BufferSize > 0 ) {
                        if( strncmp( ( const char* )Buffer, ( const char* )PingMsg, 4 ) == 0 ) {
                            led = !led;
                            //debug( "...Ping\r\n" );
                            pc.printf("...Ping...\r\n");
                            // Send the reply to the PING string
                            strcpy( ( char* )Buffer, ( char* )PongMsg );
                            // We fill the buffer with numbers for the payload
                            for( i = 4; i < BufferSize; i++ ) {
                                Buffer[i] = i - 4;
                            }
                            wait_ms( 10 );
                            Radio.Send( Buffer, BufferSize );
                        } else { // valid reception but not a PING as expected
                            // Set device as master and start again
                            isMaster = true;
                            Radio.Rx( RX_TIMEOUT_VALUE );
                        }
                    }
                }
                State = LOWPOWER;
                break;
            case TX:
                led = !led;
                if( isMaster == true ) {
                    //  debug( "Ping...\r\n" );
                    pc.printf("Ping...\r\n");
                } else {
                    //        debug( "Pong...\r\n" );
                    pc.printf("Pong...\r\n");
                }
                Radio.Rx( RX_TIMEOUT_VALUE );
                State = LOWPOWER;
                break;
            case RX_TIMEOUT:
                if( isMaster == true ) {
                    // Send the next PING frame
                    strcpy( ( char* )Buffer, ( char* )PingMsg );
                    for( i = 4; i < BufferSize; i++ ) {
                        Buffer[i] = i - 4;
                    }
                    wait_ms( 10 );
                    Radio.Send( Buffer, BufferSize );
                } else {
                    Radio.Rx( RX_TIMEOUT_VALUE );
                }
                State = LOWPOWER;
                break;
            case RX_ERROR:
                // We have received a Packet with a CRC error, send reply as if packet was correct
                if( isMaster == true ) {
                    // Send the next PING frame
                    strcpy( ( char* )Buffer, ( char* )PingMsg );
                    for( i = 4; i < BufferSize; i++ ) {
                        Buffer[i] = i - 4;
                    }
                    wait_ms( 10 );
                    Radio.Send( Buffer, BufferSize );
                } else {
                    // Send the next PONG frame
                     strcpy( ( char* )Buffer, ( char* )PongMsg );
                    for( i = 4; i < BufferSize; i++ ) {
                        Buffer[i] = i - 4;
                    }
                    wait_ms( 10 );
                    Radio.Send( Buffer, BufferSize );
                }
                State = LOWPOWER;
                break;
            case TX_TIMEOUT:
                Radio.Rx( RX_TIMEOUT_VALUE );
                State = LOWPOWER;
                break;
            case LOWPOWER:
                break;
            default:
                State = LOWPOWER;
                break;
        }
    }
 //   while(1);
}

void OnTxDone( void )
{
    Radio.Sleep( );
    State = TX;
//   debug_if( DEBUG_MESSAGE, "> OnTxDone\n\r" );
    pc.printf("> OnTxDone\r\n");

}

void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio.Sleep( );
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );
    RssiValue = rssi;
    SnrValue = snr;
    State = RX;
    //  debug_if( DEBUG_MESSAGE, "> OnRxDone\n\r" );
    pc.printf("> OnRxDone\r\n");
    pc.printf("size:%d rssi:%d snr:%d",size,rssi,snr);
    srprintf(payload,sizeof(payload));
    Radio.Rx( RX_TIMEOUT_VALUE );
}

void OnTxTimeout( void )
{
    Radio.Sleep( );
    State = TX_TIMEOUT;
//    debug_if( DEBUG_MESSAGE, "> OnTxTimeout\n\r" );
    pc.printf("> OnTxTimeout\r\n");
    Radio.Rx( RX_TIMEOUT_VALUE );
}

void OnRxTimeout( void )
{
    Radio.Sleep( );
    Buffer[ BufferSize ] = 0;
    State = RX_TIMEOUT;
//   debug_if( DEBUG_MESSAGE, "> OnRxTimeout\n\r" );
    pc.printf("> OnRxTimeout\r\n");
    Radio.Rx( RX_TIMEOUT_VALUE );
}

void OnRxError( void )
{
    Radio.Sleep( );
    State = RX_ERROR;
//   debug_if( DEBUG_MESSAGE, "> OnRxError\n\r" );
    pc.printf("> OnRxError\r\n");
    Radio.Rx( RX_TIMEOUT_VALUE );
}

