#ifndef __MemoryController_H_
#define __MemoryController_H_
 
#include "mbed.h"
 
class MemoryController
{
    
public:

    enum Commands_e 
    {
        RESET           = 0xFF,
        READADDRINIT    = 0x00,
        READADDREND     = 0x30,
        WRITEPAGEINIT   = 0x80,
        WRITEPAGEEND    = 0x10,
        ERASEBLOCKINIT  = 0x60,
        ERASEBLOCKEND   = 0xD1
    };
    
    union Data{
        uint8_t all;
        struct BitField_s
        {
            uint8_t dq1 : 1;
            uint8_t dq2 : 1;
            uint8_t dq3 : 1;
            uint8_t dq4 : 1;
            uint8_t dq5 : 1;
            uint8_t dq6 : 1;
            uint8_t dq7 : 1;
            uint8_t dq8 : 1;
        }bits;
    };

    struct DataAddress {
        char    col1;
        char    col2;
        char    page;
        char    block1;
        char    block2;
    } Address;
    
    MemoryController(DigitalOut CE,DigitalOut CLE,DigitalOut ALE,DigitalOut WE,DigitalOut RE,DigitalOut WP,PinName *DQ, DigitalIn RDY);
 
    /** @brief Destructor */
    ~MemoryController(void);
    
    /**
    * @brief    Set Dq Bus to char value (8 bits) used for command writes and address writes  
    * @param[IN]    set - Value to set DQ array to 
    * @param[OUT]   values - Array where values are held (8 bits of 1's and 0's to set DQ)
    */
    void setDQ(const char set, bool *values);

    /**
    * @brief    Used at initialization of Device
    */
    void ResetDevice();
    
    void startPageRead(DataAddress &address);
    
    void endPageRead();
    
    void ReadData(char *DataOut);
    
    void sendPageData(char *DataIn);
        
    void startPageWrite(DataAddress &address);
    
    void endPageWrite();
    
    
    /**
    * @brief    Used to send address for block erase (3 bytes)
    */
    void MemoryController::WriteAddressBlock(DataAddress &address);

    /**
    * @brief    Used to send address data for read/write page (5 bytes)
    */
    void MemoryController::WriteAddressPage(DataAddress &address);
    
    
    /**
    * @brief        Read Page of Data (4320 bytes)
    * @param[IN]    address -   address to read data from
    * @param[OUT]   DataOut -   Array to write data into
    */
    void ReadPage(DataAddress &address, char *DataOut);

    /**
    * @brief        Program 1 page (4320 bytes)
    * @param[IN]    adderess - address to program data to 
    * @param[IN]    DataIn   - Array pointer to data being written into page
    */
    void ProgramPage(DataAddress &address, char *DataIn);

    /**
    * @brief        Earse 1 block (128 pages)
    * @param[IN]    address -  Only row address characters for erasing block, no collumn needed  
    * @return       0 on success, 1 on failure
    */
    void EraseBlock(DataAddress &address);
    
    void setLogicStandby();
    
    void setLogicBusIdle();
    
    void setLogicCommandInput();
    
    void setLogicAddressInput();
    
    void setLogicDataInput();
    
    void setLogicDataOut();
    
    void setLogicWriteProtect();
    
private:
    PinName *m_DQ;
    DigitalOut m_CE, m_CLE, m_ALE, m_WE, m_RE, m_WP;
    DigitalIn m_RDY;
};
 
#endif /* __MemoryController_H_ */