/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "cc3000.h"
#include "main.h"

using namespace mbed_cc3000;

const char SSID[] = "yourap";
const char KEY[] = "yourkey";

/* cc3000 module declaration specific for user's board. Check also init() */
#if (MY_BOARD == WIGO)
cc3000 wifi(PTA16, PTA13, PTD0, SPI(PTD2, PTD3, PTC5), SSID, KEY, WPA2, false);
Serial pc(USBTX, USBRX);
#elif (MY_BOARD == WIFI_DIPCORTEX)
cc3000 wifi(p28, p27, p30, SPI(p21, p14, p37), SSID, KEY, WPA2, false);
Serial pc(UART_TX, UART_RX);
#elif (MY_BOARD == MBED_BOARD_EXAMPLE)
cc3000 wifi(p9, p10, p8, SPI(p5, p6, p7), SSID, KEY, WPA2, false);
Serial pc(USBTX, USBRX);
#elif (MY_BOARD == SPARKFUN_WIFI_SHIELD)
cc3000 wifi(D2, D7, D10, SPI(D11, D12, D13), SSID, KEY, WPA2, false); //SparkFun Board on Arduino pin definitions
Serial pc(USBTX, USBRX);
#elif (MY_BOARD == ADAFRUIT_WIFI_SHIELD)
cc3000 wifi(D3, D5, D10, SPI(D11, D12, D13), SSID, KEY, WPA2, false);
Serial pc(USBTX, USBRX);
#endif


// from http://developer.mbed.org/users/dflet/code/CC3000Test/
typedef struct scanResults {
    unsigned long numNetworksFound;
    unsigned long results;
    unsigned isValid:1;
    unsigned rssi:7;
    unsigned securityMode:2;
    unsigned ssidLength:6;
    unsigned short frameTime;
    unsigned char ssid_name[32];
    unsigned char bssid[6];
} scanResults;

int8_t listAPs() {
    int ret;
    scanResults sr;
    int apCounter;
    
    if ((ret = wifi._wlan.ioctl_get_scan_results(2000, (unsigned char *)&sr)) != 0) {
        printf("get scan results failed ret=%d\r\n", ret);
        return -1;
    }
    apCounter = sr.numNetworksFound;
    printf("APs found: %d\r\n", apCounter);

    do {
        if (sr.isValid) {
            char ssidbuf[32];
            memcpy(ssidbuf, sr.ssid_name, sr.ssidLength);
            ssidbuf[sr.ssidLength] = 0;
            printf("ssid=%s rssi=%3d\r\n", ssidbuf, sr.rssi);
        }

        if (--apCounter> 0) {
            if ((ret = wifi._wlan.ioctl_get_scan_results(2000, (unsigned char *)&sr)) != 0) {
                printf("get scan results failed ret=%d\r\n", ret);
                return -1;
            }
        }
    } while (apCounter > 0);
    printf(("End of AP list."));
    return 0;
}


/**
 *  \brief Connect to an AP, read firmware version, and list APs
 *  \param none
 *  \return int
 */
int main() {
    int i = 0;
    init(); /* board dependent init */
    pc.baud(115200);

    // allow time to get a serial console
    for (int i=10; i>=0; i--) {
        wait(0.5);
        pc.printf("Starting in %d\r\n", i);
    }

    printf("cc3000 WiFi demo.\r\n");
    wifi.init();
    printf("init done. connecting\r\n");
    if (wifi.connect() == -1) {
        while (++i < 10) {
            printf("Failed to connect. Please verify connection details and try again. \r\n");
            wait(1.0);
        }
    } else {
        while (++i < 10) {
            uint8_t v[2], ret;
            
            printf("IP address: %s \r\n",wifi.getIPAddress());
            wait(1.0);
            
            ret = wifi._nvmem.read_sp_version(v);
            printf("cc3000 sp version=%d.%d ret=%x\r\n", v[0], v[1], ret);

            ret = listAPs();
            wait(60.0);
            printf("\r\ncc3000 connected to the Internet. Demo completed.\r\n");
        }
    }

    wifi.disconnect();
}