//----------------------------------------------------------------------------
// The confidential and proprietary information contained in this file may
// only be used by a person authorised under and to the extent permitted
// by a subsisting licensing agreement from ARM Limited or its affiliates.
//
// (C) COPYRIGHT 2016 ARM Limited or its affiliates.
// ALL RIGHTS RESERVED
//
// This entire notice must be reproduced on all copies of this file
// and copies of this file may only be made by a person if such person is
// permitted to do so under the terms of a subsisting license agreement
// from ARM Limited or its affiliates.
//----------------------------------------------------------------------------
#include <sstream>
#include <string>
#include <time.h>

#include "mbed.h"
#include "OdinWiFiInterface.h"
#include "http_request.h"

#include "C12832.h"
#include "CCS811.h"
#include "Sht31.h"
#include "TSL2561.h"
#include "MMA7660.h"

#include "vl53l0x_api.h"
#include "vl53l0x_platform.h"
#include "vl53l0x_i2c_platform.h"
 
#define USE_I2C_2V8

// GLOBAL VARIABLES HERE
C12832  lcd(PE_14, PE_12, PD_12, PD_11, PE_9);
DigitalOut  led(PB_6, 1);
Sht31   temp_sensor(PF_0, PF_1);
CCS811  air_sensor(PF_0, PF_1);
TSL2561 light_sensor(PF_0, PF_1, TSL2561_ADDR_HIGH);
MMA7660 accel(PF_0, PF_1);

OdinWiFiInterface wifi;

InterruptIn post_button(PF_2);
InterruptIn get_put_button(PG_4);
volatile bool post_clicked = false;
volatile bool get_clicked = false;
volatile bool put_clicked = false;

// FUNCTION DEFINITIONS HERE
void lcd_print(const char* message) {
    lcd.cls();
    lcd.locate(0, 3);
    lcd.printf(message);
}


float * read_temp() {
    float t = temp_sensor.readTemperature();
    float h = temp_sensor.readHumidity();
    //char val[32];
    //sprintf(val, "TEMP: %3.2fC, HUM: %3.2f%%", t, h);
    //lcd_print(val);
    static float out[2];
    out[0] = t;
    out[1] = h;
    return out;
}

uint16_t * read_air() {
    air_sensor.init();
    uint16_t eco2, tvoc;
    air_sensor.readData(&eco2, &tvoc);
    //char val[32];
    //sprintf(val, "eCO2: %dppm, TVOC: %dppb", eco2, tvoc);
    //lcd_print(val);
    static uint16_t out[2];
    out[0] = eco2;
    out[1] = tvoc;
    return out;
}

int * read_light() {
    int vis = light_sensor.getLuminosity(TSL2561_VISIBLE);
    int infr = light_sensor.getLuminosity(TSL2561_INFRARED);
    //char val[32];
    //sprintf(val, "VIS: %d, INFR: %d ", vis, infr);
    //lcd_print(val);
    static int out[2];
    out[0] = vis;
    out[1] = infr;
    return out;
}


double * read_accel() {
    double x = accel.x();
    double y = accel.y();
    double z = accel.z();
    //char val[32];
    //sprintf(val, "x=%.2f y=%.2f z=%.2f", x, y, z);
    //lcd_print(val);
    static double out[3];
    out[0] = x;
    out[1] = y;
    out[2] = z;
    return out;
}



VL53L0X_Error WaitMeasurementDataReady(VL53L0X_DEV Dev) {
    VL53L0X_Error Status = VL53L0X_ERROR_NONE;
    uint8_t NewDatReady=0;
    uint32_t LoopNb;
    
    if (Status == VL53L0X_ERROR_NONE) {
        LoopNb = 0;
        do {
            Status = VL53L0X_GetMeasurementDataReady(Dev, &NewDatReady);
            if ((NewDatReady == 0x01) || Status != VL53L0X_ERROR_NONE) {
                break;
            }
            LoopNb = LoopNb + 1;
            VL53L0X_PollingDelay(Dev);
        } while (LoopNb < VL53L0X_DEFAULT_MAX_LOOP);
 
        if (LoopNb >= VL53L0X_DEFAULT_MAX_LOOP) {
            Status = VL53L0X_ERROR_TIME_OUT;
        }
    }
 
    return Status;
}
 
VL53L0X_Error WaitStopCompleted(VL53L0X_DEV Dev) {
    VL53L0X_Error Status = VL53L0X_ERROR_NONE;
    uint32_t StopCompleted=0;
    uint32_t LoopNb;
 
    if (Status == VL53L0X_ERROR_NONE) {
        LoopNb = 0;
        do {
            Status = VL53L0X_GetStopCompletedStatus(Dev, &StopCompleted);
            if ((StopCompleted == 0x00) || Status != VL53L0X_ERROR_NONE) {
                break;
            }
            LoopNb = LoopNb + 1;
            VL53L0X_PollingDelay(Dev);
        } while (LoopNb < VL53L0X_DEFAULT_MAX_LOOP);
 
        if (LoopNb >= VL53L0X_DEFAULT_MAX_LOOP) {
            Status = VL53L0X_ERROR_TIME_OUT;
        }
 
    }
 
    return Status;
}

void send_post() {
    post_clicked = true;
}
 
void send_put() {
    put_clicked = true;
}

int main() {

    // Setup Laser
    VL53L0X_Dev_t MyDevice;
    VL53L0X_Dev_t *pMyDevice = &MyDevice;
    VL53L0X_RangingMeasurementData_t    RangingMeasurementData;
    VL53L0X_RangingMeasurementData_t   *pRangingMeasurementData    = &RangingMeasurementData;
    
    // Initialize Comms laster
    pMyDevice->I2cDevAddr      = 0x52;
    pMyDevice->comms_type      =  1;
    pMyDevice->comms_speed_khz =  400;
    
    VL53L0X_RdWord(&MyDevice, VL53L0X_REG_OSC_CALIBRATE_VAL,0);
    VL53L0X_DataInit(&MyDevice); 
    uint32_t refSpadCount;
    uint8_t isApertureSpads;
    uint8_t VhvSettings;
    uint8_t PhaseCal;

    VL53L0X_StaticInit(pMyDevice); 
    VL53L0X_PerformRefSpadManagement(pMyDevice, &refSpadCount, &isApertureSpads); // Device Initialization
    VL53L0X_PerformRefCalibration(pMyDevice, &VhvSettings, &PhaseCal); // Device Initialization
    VL53L0X_SetDeviceMode(pMyDevice, VL53L0X_DEVICEMODE_CONTINUOUS_RANGING); // Setup in single ranging mode
    VL53L0X_SetLimitCheckValue(pMyDevice, VL53L0X_CHECKENABLE_SIGNAL_RATE_FINAL_RANGE, (FixPoint1616_t)(0.25*65536)); //High Accuracy mode, see API PDF
    VL53L0X_SetLimitCheckValue(pMyDevice, VL53L0X_CHECKENABLE_SIGMA_FINAL_RANGE, (FixPoint1616_t)(18*65536)); //High Accuracy mode, see API PDF
    VL53L0X_SetMeasurementTimingBudgetMicroSeconds(pMyDevice, 200000); //High Accuracy mode, see API PDF
    VL53L0X_StartMeasurement(pMyDevice);

    // Setup wifi
    lcd_print("Connecting...");
    int ret = wifi.connect(MBED_CONF_APP_WIFI_SSID, MBED_CONF_APP_WIFI_PASSWORD, NSAPI_SECURITY_WPA_WPA2);
    if (ret != 0) {
        lcd_print("Connection error.");
        return -1;
    }
    lcd_print("Successfully connected!");
    
    // Initialize reading variables
    int var=1, measure=0;
    int ave=0, sum=0;
    double * acc;
    int * light;
    uint16_t * air;
    float * temp;
    //auto time0 = chrono::steady_clock::now();
    
    int counter = 0;
    int buffer_size = 10;
    int buffer_time[buffer_size];
    double buffer_acc[buffer_size][3];
    int buffer_dist[buffer_size];
    int buffer_light[buffer_size][2];
    uint16_t buffer_air[buffer_size][2];
    float buffer_temp[buffer_size][2];
    
    post_button.rise(&send_post);
    get_put_button.rise(&send_put);
    
    Timer t;
    
    t.start();
    
    while (true) {
        
        // Get laser measurement
           while(var<=2){
                WaitMeasurementDataReady(pMyDevice);
                VL53L0X_GetRangingMeasurementData(pMyDevice, pRangingMeasurementData);
                measure=pRangingMeasurementData->RangeMilliMeter;
                sum=sum+measure;
                VL53L0X_ClearInterruptMask(pMyDevice, VL53L0X_REG_SYSTEM_INTERRUPT_GPIO_NEW_SAMPLE_READY);
                VL53L0X_PollingDelay(pMyDevice);
                var++;
                }
        ave=sum/var; // This is the measurement value
        var=1;
        sum=0; 

        // TIMINGS
        int now_time = t.read_ms();
        
        std::ostringstream timestr;
        timestr << "Elapsed time: " << now_time/1000 << "\n Press SW1 to send data";
        std::string tstr = timestr.str();
        
        //std::ostringstream timestr;
        //timestr << counter;
        //std:string tstr = timestr.str();
        //lcd_print(tstr.c_str());
        
        //buffer is sent via http when it is full
        buffer_time[counter] = now_time;
        //buffer_time[counter] = 0;
        buffer_dist[counter] = ave;
        // Get accelerometer measurement
        acc = read_accel();
        buffer_acc[counter][0] = acc[0];
        buffer_acc[counter][1] = acc[1];
        buffer_acc[counter][2] = acc[2];
        light = read_light();
        buffer_light[counter][0] = light[0];
        buffer_light[counter][1] = light[1];
        //temp = read_temp();
        //buffer_temp[counter][0] = temp[0];
        //buffer_temp[counter][0] = temp[1];
        //air = read_air();
        //buffer_air[counter][0] = air[0];
        //buffer_air[counter][0] = air[1];
        
        counter += 1;
        
        if (counter >= buffer_size) {
            counter = 0;
            lcd_print(tstr.c_str());
        
            if (post_clicked) {
                lcd_print("Sending Data... \nPress SW2 to stop.");
                //post_clicked = false;
                NetworkInterface* net = &wifi;
                HttpRequest* request = new HttpRequest(net, HTTP_POST, "http://10.25.1.118:5000");
                request->set_header("Content-Type", "application/json");
                std::ostringstream datastr;
                
                datastr << "{ \"time\" : [ " << buffer_time[0];
                for(int i=1; i<buffer_size; i++){
                    datastr << ", " << buffer_time[i];
                }
                datastr << " ], ";
                
                datastr << " \"dist\" : [ " << buffer_dist[0];
                for(int i=1; i<buffer_size; i++){
                    datastr << ", " << buffer_dist[i];
                }
                datastr << " ], ";
                
                datastr << " \"acc\" : [ ";
                datastr << "{ \"x\" : " << buffer_acc[0][0] << ", \"y\" : " << buffer_acc[0][1] << ", \"z\": " << buffer_acc[0][2] << "}" ;
                for(int i=1; i<buffer_size; i++){
                    datastr << ", { \"x\" : " << buffer_acc[i][0] << ", \"y\" : " << buffer_acc[i][1] << ", \"z\": " << buffer_acc[i][2] << "}" ;
                }
                
                // Light
                datastr << "], ";
                datastr << " \"visible\" : [ " << buffer_light[0][0];
                for(int i=1; i<buffer_size; i++){
                    datastr << ", " << buffer_light[i][0];
                }
                datastr << " ], ";
                datastr << " \"infared\" : [ " << buffer_light[0][1];
                for(int i=1; i<buffer_size; i++){
                    datastr << ", " << buffer_light[i][1];
                }    
                datastr << " ] ";        
                
                /*
                // Temperature
                datastr << " ], ";
                datastr << " \"temp\" : [ ";
                datastr << "{ \"temperature\" : " << buffer_temp[0][0] << ", \"humidity\" : " << buffer_temp[0][1] << "}" ;
                for(int i=1; i<buffer_size; i++){
                    datastr << ", { \"visible\" : " << buffer_temp[i][0] << ", \"infared\" : " << buffer_temp[i][1] << "}" ;
                }
                
                // Air
                datastr << " ], ";
                datastr << " \"air\" : [ ";
                datastr << "{ \"eco2\" : " << buffer_air[0][0] << ", \"tvoc\" : " << buffer_air[0][1] << "}" ;
                for(int i=1; i<buffer_size; i++){
                    datastr << ", { \"eco2\" : " << buffer_air[i][0] << ", \"tvoc\" : " << buffer_air[i][1] << "}" ;
                }
                */
                
                datastr << "}";
                    
                std::string dstr = datastr.str();
                HttpResponse* response = request->send(dstr.c_str(), strlen(dstr.c_str())); 
                
                //const char body[] = "{ \"key1\" : \"value1\" }";
                //HttpResponse* response = request->send(body, strlen(body));
                //lcd_print(response->get_body_as_string().c_str());
                delete request;
            }
        }
        
        if (put_clicked) {
            lcd_print("Stopped Sending Data...");
            put_clicked = false;
            post_clicked = false;
        }
        
        //wait_ms(10);
     
    }

}
