#ifndef FRONT_H
#define FRONT_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Menu.h"
#include "Sound.h"
#include "Envelope.h"
#include "Filter.h"

/** Front class
 * @author Luke Cartwright, University of Leeds
 * @brief manages front pannel of Synth
 * @date May 2020
*/

class Front
{
public:
    //variables
    int submenu;
    int a;
    int d;
    int s;
    int r;
    int dir;
    int envelope_in;
    int dir_1;
    uint16_t out;
    int frequency;

    //Methods
    /** Constructor */
    Front();

    /** Destructor */
    ~Front();

    /** Runs whole front pannel 
    * @function to process whole front pannel
    * Deals with ADSR and sound generation by calling other classes */
    void frontrun(N5110 &lcd, Gamepad &pad, int submenu);


private:
//variables
    int menuflag; //Flag to exit front function or change menu
    int used; //1->4 (A<D<S<R)
    bool use; //1=use 0=!use
    bool initial; //sets to run 1st time
    int adsr; //incrament correct val
    int ud; //up/down
    bool noteon; //trigger for play tone
    int release_flag; //sets 1 for if release on
    int filter_type;
    int oct;
    int lr;
    
    #ifdef CSV //COMMA SEPERATED VALUE
    int itterator; //Itterator for CSV
    #endif

//methods
    /** initial setup of front panel
    */
    void frontsetup(N5110 &lcd, Gamepad &pad, int submenu, int filter_type, bool initial);
    
    /** Initialises front
    * @ Runs all initial front functions
    */
    void initialise(Gamepad &pad, N5110 &lcd, int submenu);
    
    /**prints waveforms for front disp. to LCD
    */
    void printwav(N5110 &lcd, int submenu);

    /**prints single fader to LCD 
    */
    void printfader(N5110 &lcd,int x, int y);

    /**prints all ADSR sliders to LCD 
    */
    void printsliders(N5110 &lcd, int a, int d,int s,int r, int used);

    /**prints individual slider to LCD
    */
    void printslider(N5110 &lcd, int x_val, int y_val, bool use);
    
    /**Prints Filter Graphic
    * @prints correct graphic dependant on filter_type
    */
    void printfilter(N5110 &lcd, int filter_type);

    /**incraments correct adsr value
    * @outputs new ADSR value when called
    */
    int incrament_adsr(int used, int adsr, int ud);
    
    /**Incraments Filter type
    *outputs new filter value
    */
    int incrament_filter(int adsr, int ud);
    
    /**Converts Float to Frequency
    * @changes Pot 1 value to Frequency of tone (int)
    */
    int frequency_convert(Gamepad &pad, int oct);
    
    /**fader Select Change
    */
    int fader_select(int used, int lr);

    //pre compilation directive function
    #ifdef SLOW_TIME
    /**prints slow time front values output from loop
    * @prints values to pc if SLOW_TIME Macro used
    */
    void print_SLOWTIME_out (uint16_t out, bool noteon, int a, int d, int s, int r, int release_flag, int silent_flag);
    #endif
    
};

#endif