/*
 * Program implements MQTT client on a NUCLEO-L476RG board
 * using arm mbed-mqtt library and ESP-WROOM-02 WiFi modem.
 *
 * University of Belgrade - School of Electrical Engineering
 * Department of Electronics
 * Bulevar Kralja Aleksandra 73, 11120 Belgrade, Serbia
 *
 * November 2021.
 *
 */

#include "mbed.h"
#include "mb_pins.h"
#include "platform/mbed_thread.h"
#include "MQTTClientMbedOs.h"

#include "Adafruit_GFX.h"
#include "Adafruit_GFX_Config.h"
#include "Adafruit_SSD1306.h"



#define SMALL_WAIT_MS 10
#define VOLTAGE_SCALER 3.3f

//I2C adress
#define I2C_ADDRESS         0x3c
#define I2C_ADD_MBED    I2C_ADDRESS << 1

//OLED DIMENSIONS
#define OLED_HEIGHT_PX 64
#define OLED_WIDTH_PX 128

// Pointer to a WiFi network object:
WiFiInterface *wifi;
// Creating TCP socket:
TCPSocket socket;
// Creating MQTT client using the TCP socket;
MQTTClient client(&socket);
// Message handler:
MQTT::Message message;

I2C i2c_obj(MB_OLED_SDA, MB_OLED_SCL);
Adafruit_SSD1306_I2c myOLED(i2c_obj, PB_5, I2C_ADD_MBED, OLED_HEIGHT_PX, OLED_WIDTH_PX);

const char* topic_sub = "subpim";
const char* topic_pub = "pubpim";
// HiveMQ broker connectivity information:
const char* hostname = "broker.hivemq.com";
const int port = 1883;


void messageArrived(MQTT::MessageData& md)
{
    MQTT::Message &msg = md.message;
    printf("Browser message: %.*s \n", msg.payloadlen, (char*)msg.payload);
    message.qos = MQTT::QOS0;
    message.retained = false;
    message.dup = false;
    message.payload = (void*)msg.payload;
    message.payloadlen = msg.payloadlen + 1;
    client.publish(topic_pub, message);
    myOLED.clearDisplay();
    myOLED.setTextCursor(0, 0);
    myOLED.printf("%.*s \n", msg.payloadlen, (char*)msg.payload);
    myOLED.display(); 
    
}

int main()
{
    // Create a default network interface:
    wifi = WiFiInterface::get_default_instance();
    if (!wifi) {
        printf("ERROR: No WiFiInterface found.\n");
        return -1;
    }
    
    // Connect to the network with the parameters specified in 'mbed_app.json':
    printf("Connecting to %s... \n",MBED_CONF_APP_WIFI_SSID);
    int ret = wifi->connect(MBED_CONF_APP_WIFI_SSID, MBED_CONF_APP_WIFI_PASSWORD, NSAPI_SECURITY_WPA_WPA2);
    if(ret != 0){
        printf("\nConnection error: %d\n", ret);
        return -1;
    }
    // Print out the information aquired:
    printf("Connected to %s\n", MBED_CONF_APP_WIFI_SSID);
    
    // Open TCP socket using WiFi network interface:
    socket.open(wifi);
    
    // Connect to the HiveMQ broker:
    socket.connect(hostname,port);
    
    // Fill connect data with default values:
    MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
    // Change only ID and protocol version:
    data.MQTTVersion = 3;
    data.clientID.cstring = "pim-09";

    client.connect(data);
    client.subscribe(topic_sub, MQTT::QOS0, messageArrived);
    
    i2c_obj.frequency(400000);
    
    myOLED.begin();
    myOLED.clearDisplay();
    myOLED.display();
    while(1) {
        thread_sleep_for(SMALL_WAIT_MS);
        client.yield(1000);
    }
}
