#include "ChannelPlan.h"
#include "plans/ChannelPlan_EU868.h"
#include "MultitechDot.h"

MultitechDot *MultitechDot::get_instance(struct dot_config *config) {
    using namespace lora;

    ChannelPlan *plan = new ChannelPlan_EU868();
    plan->SetNumberOfChannels(1);
    plan->SetTxChannel(0);

    MultitechDot *dot = (MultitechDot *) mDot::getInstance(plan);

    dot->config(config);

    return dot;
}

void MultitechDot::config(struct dot_config *config) {
    _config = config;

    this->setLogLevel(config->log_level);

    if(config->join_mode == mDot::MANUAL) {
        _manual_config();
    } else if (config->join_mode == mDot::OTA) {
        _ota_config();
    }
}

struct dot_config *MultitechDot::get_config() {
    return _config;
}

void MultitechDot::_manual_config() {
    logInfo("MANUAL config");
    if (!this->getStandbyFlag()) {
        logInfo("mbed-os library version: %d", MBED_LIBRARY_VERSION);

        // start from a well-known state
        logInfo("defaulting Dot configuration");
        this->resetConfig();
        this->resetNetworkSession();

        this->setDisableDutyCycle(_config->disable_duty_cycle);
        this->setTxDataRate(_config->data_rate);

        // update configuration if necessary
        if (this->getJoinMode() != this->MANUAL) {
            logInfo("changing network join mode to MANUAL");
            if (this->setJoinMode(this->MANUAL) != this->MDOT_OK) {
                logError("failed to set network join mode to MANUAL");
            }
        }
        // in MANUAL join mode there is no join request/response transaction
        // as long as the Dot is configured correctly and provisioned correctly on the gateway, it should be able to communicate
        // network address - 4 bytes (00000001 - FFFFFFFE)
        // network session key - 16 bytes
        // data session key - 16 bytes
        // to provision your Dot with a Conduit gateway, follow the following steps
        //   * ssh into the Conduit
        //   * provision the Dot using the lora-query application: http://www.multitech.net/developer/software/lora/lora-network-server/
        //      lora-query -a 01020304 A 0102030401020304 <your Dot's device ID> 01020304010203040102030401020304 01020304010203040102030401020304
        //   * if you change the network address, network session key, or data session key, make sure you update them on the gateway
        // to provision your Dot with a 3rd party gateway, see the gateway or network provider documentation
        BaseDot::update_manual_config(
                _config->network_address,
                _config->network_session_key,
                _config->data_session_key,
                _config->frequency_sub_band,
                _config->public_network,
                _config->ack
        );

        // enable or disable Adaptive Data Rate
        this->setAdr(_config->adr);

        // Configure the join delay
        this->setJoinDelay(_config->join_delay);

        // save changes to configuration
        logInfo("saving configuration");
        if (!this->saveConfig()) {
            logError("failed to save configuration");
        }

        // display configuration
        this->display_config();
    } else {
        // restore the saved session if the dot woke from deepsleep mode
        // useful to use with deepsleep because session info is otherwise lost when the dot enters deepsleep
        logInfo("restoring network session from NVM");
        this->restoreNetworkSession();
    }
}


void MultitechDot::_ota_config() {
    logInfo("OTA config");
    if (!this->getStandbyFlag()) {
        logInfo("mbed-os library version: %d", MBED_LIBRARY_VERSION);

        // start from a well-known state
        logInfo("defaulting Dot configuration");
        this->resetConfig();
        this->resetNetworkSession();

        this->setDisableDutyCycle(_config->disable_duty_cycle);
        this->setTxDataRate(_config->data_rate);

        // make sure library logging is turned on
        this->setLogLevel(mts::MTSLog::INFO_LEVEL);

        // update configuration if necessary
        if (this->getJoinMode() != mDot::OTA) {
            logInfo("changing network join mode to OTA");
            if (this->setJoinMode(mDot::OTA) != mDot::MDOT_OK) {
                logError("failed to set network join mode to OTA");
            }
        }
        // in OTA and AUTO_OTA join modes, the credentials can be passed to the library as a name and passphrase or an ID and KEY
        // only one method or the other should be used!
        // network ID = crc64(network name)
        // network KEY = cmac(network passphrase)
        BaseDot::update_ota_config_id_key(
                _config->network_id,
                _config->network_key,
                _config->frequency_sub_band,
                _config->public_network,
                _config->ack
        );
        //update_ota_config_id_key(network_id, network_key, frequency_sub_band, network_type, ack);

        // configure network link checks
        // network link checks are a good alternative to requiring the gateway to ACK every packet and should allow a single gateway to handle more Dots
        // check the link every count packets
        // declare the Dot disconnected after threshold failed link checks
        // for count = 3 and threshold = 5, the Dot will ask for a link check response every 5 packets and will consider the connection lost if it fails to receive 3 responses in a row
        update_network_link_check_config(3, 5);

        // enable or disable Adaptive Data Rate
        this->setAdr(_config->adr);

        // Configure the join delay
        this->setJoinDelay(_config->join_delay);

        // save changes to configuration
        logInfo("saving configuration");
        if (!this->saveConfig()) {
            logError("failed to save configuration");
        }

        // display configuration
        display_config();
    } else {
        // restore the saved session if the dot woke from deepsleep mode
        // useful to use with deepsleep because session info is otherwise lost when the dot enters deepsleep
        logInfo("restoring network session from NVM");
        this->restoreNetworkSession();
    }
}

void MultitechDot::sleep_wake_rtc_or_interrupt(uint32_t delay_s, bool deepsleep) {
    if (deepsleep) {
        // for xDot, WAKE pin (connected to S2 on xDot-DK) is the only pin that can wake the processor from deepsleep
        // it is automatically configured when INTERRUPT or RTC_ALARM_OR_INTERRUPT is the wakeup source and deepsleep is true in the this->sleep call
    } else {
        // configure WAKE pin (connected to S2 on xDot-DK) as the pin that will wake the xDot from low power modes
        // other pins can be configured instead: GPIO0-3 or UART_RX
        this->setWakePin(WAKE);
    }

    logInfo("%s sleeping %lus or until interrupt on %s pin", deepsleep ? "deep" : "", delay_s,
            deepsleep ? "WAKE" : this->pinName2Str(this->getWakePin()).c_str());

    logInfo("application will %s after waking up", deepsleep ? "execute from beginning" : "resume");

    // lowest current consumption in sleep mode can only be achieved by configuring IOs as analog inputs with no pull resistors
    // the library handles all internal IOs automatically, but the external IOs are the application's responsibility
    // certain IOs may require internal pullup or pulldown resistors because leaving them floating would cause extra current consumption
    // for xDot: UART_*, I2C_*, SPI_*, GPIO*, WAKE
    // for mDot: XBEE_*, USBTX, USBRX, PB_0, PB_1
    // steps are:
    //   * save IO configuration
    //   * configure IOs to reduce current consumption
    //   * sleep
    //   * restore IO configuration
    if (!deepsleep) {
        // save the GPIO state.
        this->sleep_save_io();

        // configure GPIOs for lowest current
        this->sleep_configure_io();
    }

    // go to sleep/deepsleep and wake using the RTC alarm after delay_s seconds or rising edge of configured wake pin (only the WAKE pin in deepsleep)
    // whichever comes first will wake the xDot
    this->sleep(delay_s, this->RTC_ALARM_OR_INTERRUPT, deepsleep);

    if (!deepsleep) {
        // restore the GPIO state.
        this->sleep_restore_io();
        this->sleep_reset_hsi();
    }
}

void MultitechDot::sleep_reset_hsi() {
    // Enable the HSI (to clock the ADC)
    RCC_OscInitTypeDef RCC_OscInitStruct;
    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI;
    RCC_OscInitStruct.HSIState = RCC_HSI_ON;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_NONE;
    HAL_RCC_OscConfig(&RCC_OscInitStruct);
}
