/*
  AirQuality library v1.0
  2010 Copyright (c) Seeed Technology Inc.  All right reserved.

  Original Author: Bruce.Qin

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "Air_Quality.h"

AirQuality::AirQuality() {
    _s = 0;
    _sum_vol = 0;
    _heating_delay_s = 20;
}

void AirQuality::set_calc_avg_volt_period(uint16_t seconds) {
    _calc_avg_volt_period_s = seconds;
}

void AirQuality::set_sampling_period(uint16_t seconds) {
    _sampling_period_s = seconds;
}

void AirQuality::calcAvgVoltageFor(uint16_t seconds) {
    uint16_t s = (uint16_t) (seconds / _sampling_period_s);
    if (_s == s) {
        standard_vol = _sum_vol / s;
        _sum_vol = 0;
        _s = 0;
        debug_if(DEBUG_AIR_QUALITY, "Vol standard in %d seconds: %d\r\n", seconds, standard_vol);
    } else {
        _sum_vol += first_vol;
        _s++;
    }
}

void AirQuality::init(PinName pin) {
    _pin = pin;
    AnalogIn sensor(_pin);
    unsigned char minutes = 0;
    debug_if(DEBUG_AIR_QUALITY, "Air Quality Sensor Starting Up... (%d s)", _heating_delay_s);
    wait(_heating_delay_s);
    init_voltage = (uint16_t) (sensor.read() * 1000); // boost the value to be on a 0 -> 1000 scale for compatibility
    debug_if(DEBUG_AIR_QUALITY, "The initial voltage is %d%% of source ", init_voltage / 10);
    while (init_voltage) {
        if ((init_voltage > 10) && (init_voltage < 798)) {
            // the init voltage is ok
            first_vol = (uint16_t) (sensor.read() * 1000); //initialize first value
            last_vol = first_vol;
            standard_vol = last_vol;
            debug_if(DEBUG_AIR_QUALITY, "Sensor ready.");
            _has_error = false;
            break;
        } else if (init_voltage <= 10 || init_voltage > 798) {
            // The sensor is not ready, wait a bit for it to cool off
            minutes++;
            debug_if(DEBUG_AIR_QUALITY, "Sensor not ready (%d), try %d/5, waiting 60 seconds...", init_voltage,
                     minutes);
            wait(60);
            init_voltage = (uint16_t) (sensor.read() * 1000);
            if (minutes == 5) {
                // After 5 minutes warn user that the sensor may be broken
                minutes = 0;
                _has_error = true;
                debug_if(DEBUG_AIR_QUALITY, "Sensor Error! You may have a bad sensor. :-(");
            }
        } else
            break;
    }
}

air_quality_values AirQuality::slope(void) {
    air_quality_values ret = UNKNOWN;

    while (_was_sampled) {
        debug_if(DEBUG_AIR_QUALITY, "sensor_value: %d\r\n", first_vol);

        if (first_vol - last_vol > 400 || first_vol > 700) {
            debug_if(DEBUG_AIR_QUALITY, "Very high pollution! Force signal active.\r\n");
            ret = VERY_HIGH_POLLUTION;
        } else if ((first_vol - last_vol > 400 && first_vol < 700) || first_vol - standard_vol > 150) {
            debug_if(DEBUG_AIR_QUALITY, "High pollution!\r\n");
            ret = HIGH_POLLUTION;

        } else if ((first_vol - last_vol > 200 && first_vol < 700) || first_vol - standard_vol > 50) {
            debug_if(DEBUG_AIR_QUALITY, "Low pollution!\r\n");
            ret = LOW_POLLUTION;
        } else {
            debug_if(DEBUG_AIR_QUALITY, "No pollution\r\n");
            ret = NO_POLLUTION;
        }

        _was_sampled = false;
        calcAvgVoltageFor(_calc_avg_volt_period_s);
        return ret;
    }
    return ret;
}

void AirQuality::set_heating_delay(uint16_t seconds) {
    _heating_delay_s = seconds;
}

