/*
 * Dallas' DS1820 family temperature sensor.
 * This library depends on the OneWire library (Dallas' 1-Wire bus protocol implementation)
 * available at <http://developer.mbed.org/users/hudakz/code/OneWire/>
 *
 * Example of use:

#include "mbed.h"
#include "DS1820.h"

Serial serial(USBTX, USBRX);

int main() {
    serial.baud(115200);
    DS1820  ds1820(PB_2);  // substitute PB_2 with actual mbed pin name connected to the DS1820 data pin.

    if(ds1820.begin()) {
        ds1820.startConversion();   // start temperature conversion
        wait(2.0);                  // let DS1820 complete the temperature conversion
        while(1) {
            float temp = 0.0;
            ds1820.read(temp);
            serial.printf("temp = %3.1f\r\n", temp);     // read temperature
            ds1820.startConversion();     // start temperature conversion
            wait(1.0);                    // let DS1820 complete the temperature conversion
        }
    } else
        serial.printf("No DS1820 sensor found!\r\n");
}

 */

#include "DS1820.h"

/**
 * @brief   Constructs a generic DS1820 sensor
 * @note    begin() must be called to detect and initialize the actual model
 * @param   pin: Name of data pin
 * @retval
 */
DS1820::DS1820(PinName pin) : oneWire(pin) {
    present = false;
    model_s = false;
}

/**
 * @brief   Constructs a specific model
 * @note    No need to call begin() to detect and initialize the model
 * @param   model:  One character model name: 'S', 's', 'B' or 'b'
 *          pin:    Name of data pin
 * @retval
 */
DS1820::DS1820(char model, PinName pin) :
        oneWire(pin) {
    if ((model == 'S') or (model == 's')) {
        present = true;
        model_s = true;
    } else if ((model == 'B') or (model == 'b')) {
        present = true;
        model_s = false;
    } else
        present = false;
}

/**
 * @brief   Detects and initializes the actual DS1820 model
 * @note
 * @param
 * @retval  true:   if a DS1820 family sensor was detected and initialized
            false:  otherwise
 */
bool DS1820::begin(void) {
    oneWire.reset_search();
    wait_ms(250);
    if (!oneWire.search(addr)) {
        debug_if(DEBUG, "No addresses.\r\n");
        oneWire.reset_search();
        wait_ms(250);
        return false;
    }

    debug_if(DEBUG, "ROM =");
    for (uint8_t i = 0; i < 8; i++) {
        debug_if(DEBUG, " %x", addr[i]);
    }
    debug_if(DEBUG, "\r\n");

    if (OneWire::crc8(addr, 7) == addr[7]) {
        present = true;

        // the first ROM byte indicates which chip
        switch (addr[0]) {
            case 0x10:
                model_s = true;
                debug_if(DEBUG, "DS18S20 or old DS1820\r\n");
                break;

            case 0x28:
                model_s = false;
                debug_if(DEBUG, "DS18B20\r\n");
                break;

            case 0x22:
                model_s = false;
                debug_if(DEBUG, "DS1822\r\n");
                break;

            default:
                present = false;
                debug_if(DEBUG, "Device doesn't belong to the DS1820 family\r\n");
                return false;
        }
        return true;
    } else {
        return false;
    }
}

/**
 * @brief   Informs about presence of a DS1820 sensor.
 * @note    begin() shall be called before using this function
 *          if a generic DS1820 instance was created by the user.
 *          No need to call begin() for a specific DS1820 instance.
 * @param
 * @retval  true:   when a DS1820 sensor is present
 *          false:  otherwise
 */
bool DS1820::isPresent(void) {
    return present;
}

/**
 * @brief   Sets temperature-to-digital conversion resolution.
 * @note    The configuration register allows the user to set the resolution
 *          of the temperature-to-digital conversion to 9, 10, 11, or 12 bits.
 *          Defaults to 12-bit resolution for DS18B20.
 *          DS18S20 allows only 9-bit resolution.
 * @param   res:    Resolution of the temperature-to-digital conversion in bits.
 * @retval
 */
void DS1820::setResolution(uint8_t res) {
    // keep resolution within limits
    if (res > 12)
        res = 12;
    if (res < 9)
        res = 9;
    if (model_s)
        res = 9;

    oneWire.reset();
    oneWire.skip();
    oneWire.write(0xBE);            // to read Scratchpad
    for (uint8_t i = 0; i < 9; i++)  // read Scratchpad bytes
        data[i] = oneWire.read();

    data[4] |= (res - 9) << 5;      // update configuration byte (set resolution)
    oneWire.reset();
    oneWire.skip();
    oneWire.write(0x4E);            // to write into Scratchpad
    for (uint8_t i = 2; i < 5; i++)  // write three bytes (2nd, 3rd, 4th) into Scratchpad
        oneWire.write(data[i]);
}

/**
 * @brief   Starts temperature conversion
 * @note    The time to complete the converion depends on the selected resolution:
 *           9-bit resolution -> max conversion time = 93.75ms
 *          10-bit resolution -> max conversion time = 187.5ms
 *          11-bit resolution -> max conversion time = 375ms
 *          12-bit resolution -> max conversion time = 750ms
 * @param
 * @retval
 */
void DS1820::startConversion(void) {
    if (present) {
        oneWire.reset();
        oneWire.skip();
        oneWire.write(0x44);    //start temperature conversion
    }
}

/**
 * @brief   Reads temperature from chip's scratchpad.
 * @note    Verifies data integrity by calculating cyclic redundancy check (CRC).
 *          If the calculated CRC dosn't match the one stored in chip's scratchpad register
 *          the temperature variable is not updated and CRC error code is returned.
 * @param   temp: The temperature variable to be updated by this routine.
 *                (It's passed as reference to floating point.)
 * @retval  error code:
 *              0 - no errors ('temp' contains the temperature measured)
 *              1 - sensor not present ('temp' is not updated)
 *              2 - CRC error ('temp' is not updated)
 */
uint8_t DS1820::read(float &temp) {
    uint16_t temp_uint16 = 0;
    uint8_t result = readRaw(temp_uint16);
    bool temp_available = 0 == result;

    if (temp_available) {
        debug_if(DEBUG, "raw uint16 = 0x%02X \r\n", temp_uint16);
        // Convert to floating point value
        temp = toFloat(temp_uint16);
    }
}

/**
 * @brief   Reads temperature from chip's scratchpad.
 * @note    Verifies data integrity by calculating cyclic redundancy check (CRC).
 *          If the calculated CRC dosn't match the one stored in chip's scratchpad register
 *          the temperature variable is not updated and CRC error code is returned.
 * @param   temp: The temperature variable to be updated by this routine.
 *                (It's passed as reference to uint16_t)
 * @retval  error code:
 *              0 - no errors ('temp' contains the temperature measured)
 *              1 - sensor not present ('temp' is not updated)
 *              2 - CRC error ('temp' is not updated)
 */
uint8_t DS1820::readRaw(uint16_t &temp) {
    if (present) {
        oneWire.reset();
        oneWire.skip();
        oneWire.write(0xBE);            // to read Scratchpad
        for (uint8_t i = 0; i < 9; i++)  // reading scratchpad registers
            data[i] = oneWire.read();

        if (oneWire.crc8(data, 8) != data[8])    // if calculated CRC does not match the stored one
            return 2;                           // return with CRC error

        // Convert the raw bytes to a 16bit unsigned value
        uint16_t *p_word = reinterpret_cast < uint16_t * > (&data[0]);

        debug_if(DEBUG, "raw = %#x\r\n", *p_word);

        if (model_s) {
            *p_word = *p_word << 3;         // 9 bit resolution,  max conversion time = 750ms
            if (data[7] == 0x10) {

                // "count remain" gives full 12 bit resolution
                *p_word = (*p_word & 0xFFF0) + 12 - data[6];
            }

            // Convert the raw bytes to a 16bit signed fixed point value :
            // 1 sign bit, 7 integer bits, 8 fractional bits (two's compliment
            // and the LSB of the 16bit binary number represents 1/256th of a unit).
            *p_word = *p_word << 4;
            // Convert to floating point value
            temp = toFloat(*p_word);
            return 0;   // return with no errors
        } else {
            uint8_t cfg = (data[4] & 0x60); // default 12bit resolution, max conversion time = 750ms

            // at lower resolution, the low bits are undefined, so let's clear them
            if (cfg == 0x00)
                *p_word = *p_word & ~7;      //  9bit resolution, max conversion time = 93.75ms
            else if (cfg == 0x20)
                *p_word = *p_word & ~3;      // 10bit resolution, max conversion time = 187.5ms
            else if (cfg == 0x40)
                *p_word = *p_word & ~1;      // 11bit resolution, max conversion time = 375ms

            // Convert the raw bytes to a 16bit signed fixed point value :
            // 1 sign bit, 7 integer bits, 8 fractional bits (two's compliment
            // and the LSB of the 16bit binary number represents 1/256th of a unit).
            *p_word = *p_word << 4;
            // Convert to floating point value
            temp = *p_word;
            return 0;   // return with no errors
        }
    } else
        return 1;   // error, sensor is not present
}

/**
 * @brief   Converts a 16-bit signed fixed point value to floating point value
 * @note    The 16-bit unsigned integer represnts actually
 *          a 16-bit signed fixed point value:
 *          1 sign bit, 7 integer bits, 8 fractional bits (two’s compliment
 *          and the LSB of the 16-bit binary number represents 1/256th of a unit).
 * @param   16-bit unsigned integer
 * @retval  Floating point value
 */
float DS1820::toFloat(uint16_t word) {
    if (word & 0x8000)
        return (-float(uint16_t(~word + 1)) / 256.0f);
    else
        return (float(word) / 256.0f);
}


