/* mbed library for the mbed Lab Board  128*32 pixel LCD
 * use C12832 controller
 * Copyright (c) 2012 Peter Drescher - DC2PD
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

// 13.10.12    initial design
// 25.10.12    add autorefresh of screen
// 25.10.12    add standart font
// 20.12.12    add bitmap graphics

// optional defines :
// #define debug_lcd  1

#include "C12832.h"
#include "mbed.h"
#include "stdio.h"
#include "Small_7.h"
#include "Arial_9.h"

#define BPP    1       // Bits per pixel


C12832::C12832(PinName e,PinName rw, PinName a0,PinName cs, PinName d0, PinName d1,PinName d2, PinName d3,PinName d4,PinName d5,PinName d6,PinName d7,const char* name) : GraphicsDisplay(name), _e(e), _a0(a0),_cs(cs),_rw(rw),
        		_d(d0, d1, d2, d3,d4,d5,d6,d7) {

    _cs = 1;
    RESET();
    Delay(1000);
    CLEAR_ADC();	  //Normal disrect (SEG0-SEG128)
	SET_SHL();		// SHL 1,COM33-COM0
	SET_BIAS();   //  1/7 BIAS
	Power_Control(0x07);  //Booster circuit: ON   Voltage regulator circuit: ON  Voltage follower circuit: ON
	Regulor_Resistor_Select(0x05); //V0 voltage regulator internal resistance ratio register value and (1 + Rb/Ra) ratio (Reference value)  5.5
	Set_Contrast_Control_Register(19);
	Initial_Dispay_Line(0x00);	//Dispay Line 0
	wr_cmd(0xF8);	//select booster ratio
	wr_cmd(0x00);	//4X
	DISPLAY_ON();
    // clear and update LCD
	memset(buffer,0x00,512);  // clear display buffer
	copy_to_lcd();
	auto_up = 1;              // switch on auto update
	// dont do this by default. Make the user call
	claim(stdout);           // redirekt printf to lcd
	locate(0,0);
	set_font((unsigned char*)Small_7);  // standart font



}

void C12832::Delay(uint32_t n)
{
	uint32_t a;
	for (a = 0; a < n; a++);
}
//Specify DDRAM line for COM0 0~63
void C12832::Initial_Dispay_Line(unsigned char line)
{
    line|=0x40;
    wr_cmd(line);
	return;
}

void C12832::Power_Control(unsigned char vol)
{
	wr_cmd((0x28|vol));
	return;
}

/*  Regulor resistor select
**            1+Rb/Ra  Vo=(1+Rb/Ra)Vev    Vev=(1-(63-a)/162)Vref   2.1v
**            0  3.0       4  5.0(default)
**            1  3.5       5  5.5
**            2  4         6  6
**            3  4.5       7  6.4
*/
void C12832::Regulor_Resistor_Select(unsigned char r)
{
	wr_cmd((0x20|r));
	return;
}

//a(0-63) 32default   Vev=(1-(63-a)/162)Vref   2.1v
void C12832::Set_Contrast_Control_Register(unsigned char mod)
{
	wr_cmd(0x81);
	wr_cmd(mod);
	return;
}

int C12832::width()
{
     return 128;
}

int C12832::height()
{
    return 32;
}


void C12832::invert(unsigned int o)
{
    if(o == 0) wr_cmd(0xA6);
    else wr_cmd(0xA7);
}


void C12832::set_contrast(unsigned int o)
{
    contrast = o;
    wr_cmd(0x81);      //  set volume
    wr_cmd(o & 0x3F);
}

unsigned int C12832::get_contrast(void)
{
    return(contrast);
}


// write command to lcd controller

void C12832::wr_cmd(unsigned char cmd)
{
	_cs =0;
	_a0 = 0;
	_rw = 0;
	_d = cmd;
	LPC_GPIO2->FIODIR |= 0xFF;
	_e = 1;
	asm("nop");
	_e = 0;
	LPC_GPIO2->FIODIR &= ~0xFF;
	_rw=1;
	_cs=1;
}

// write data to lcd controller

void C12832::wr_dat(unsigned char dat)
{
	_cs = 0;
	_a0 = 1;
	_rw = 0;
	_d = dat;
	LPC_GPIO2->FIODIR |= 0xFF;
	_e = 1;
	asm("nop");
	_e = 0;
	LPC_GPIO2->FIODIR &= ~0xFF;
	_rw=1;
	_cs=1;
}

// set one pixel in buffer

void C12832::pixel(int x, int y, int color)
{
    // first check parameter
    if(x > 128 || y > 32 || x < 0 || y < 0) return;

    if(draw_mode == NORMAL) {
        if(color == 0)
            buffer[x + ((y/8) * 128)] &= ~(1 << (y%8));  // erase pixel
        else
            buffer[x + ((y/8) * 128)] |= (1 << (y%8));   // set pixel
    } else { // XOR mode
        if(color == 1)
            buffer[x + ((y/8) * 128)] ^= (1 << (y%8));   // xor pixel
    }
}

// update lcd

void C12832::copy_to_lcd(void)
{
    
    int i=0;
    
    //page 0
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB0);      // set page address  0
    _a0 = 1;
    for(i=0; i<128; i++) {
        wr_dat(buffer[i]);
    }

    // page 1
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB1);      // set page address  1
    _a0 = 1;
    for(i=128; i<256; i++) {
        wr_dat(buffer[i]);
    }

    //page 2
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB2);      // set page address  2
    _a0 = 1;
    for(i=256; i<384; i++) {
        wr_dat(buffer[i]);
    }

    //page 3
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB3);      // set page address  3
    _a0 = 1;

    _cs = 0;
    
    for(i=384; i<512; i++) {
        wr_dat(buffer[i]);
    }

}

void C12832::cls(void)
{
    memset(buffer,0x00,512);  // clear display buffer
    copy_to_lcd();
}




void C12832::setmode(int mode)
{
    draw_mode = mode;
}

void C12832::locate(int x, int y)
{
    char_x = x;
    char_y = y;
}



int C12832::columns()
{
    return width() / font[1];
}



int C12832::rows()
{
    return height() / font[2];
}



int C12832::_putc(int value)
{
    if (value == '\n') {    // new line
        char_x = 0;
        char_y = char_y + font[2];
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    } else {
        character(char_x, char_y, value);
        if(auto_up) copy_to_lcd();
    }
    return value;
}

void C12832::character(int x, int y, int c)
{
    unsigned int hor,vert,offset,bpl,j,i,b;
    unsigned char* zeichen;
    unsigned char z,w;

    if ((c < 31) || (c > 127)) return;   // test char range

    // read font parameter from start of array
    offset = font[0];                    // bytes / char
    hor = font[1];                       // get hor size of font
    vert = font[2];                      // get vert size of font
    bpl = font[3];                       // bytes per line

    if (char_x + hor > width()) {
        char_x = 0;
        char_y = char_y + vert;
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    }

    zeichen = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = zeichen[0];                          // width of actual char
    // construct the char into the buffer
    for (j=0; j<vert; j++) {  //  vert line
        for (i=0; i<hor; i++) {   //  horz line
            z =  zeichen[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) {
                pixel(x+i,y+j,0);
            } else {
                pixel(x+i,y+j,1);
            }

        }
    }

    char_x += w;
}


void C12832::set_font(unsigned char* f)
{
    font = f;
}

void C12832::set_auto_up(unsigned int up)
{
    if(up ) auto_up = 1;
    else auto_up = 0;
}

unsigned int C12832::get_auto_up(void)
{
    return (auto_up);
}

void C12832::print_bm(Bitmap bm, int x, int y)
{
    int h,v,b;
    char d;

    for(v=0; v < bm.ySize; v++) {   // lines
        for(h=0; h < bm.xSize; h++) { // pixel
            if(h + x > 127) break;
            if(v + y > 31) break;
            d = bm.data[bm.Byte_in_Line * v + ((h & 0xF8) >> 3)];
            b = 0x80 >> (h & 0x07);
            if((d & b) == 0) {
                pixel(x+h,y+v,0);
            } else {
                pixel(x+h,y+v,1);
            }
        }
    }

}


