/* ARM University Program Microcontroller Library
 * 
 * Environmental Sensing Service
 *
 * This software is distributed under an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 */

#ifndef __BLE_ENVIRONMENTAL_SENSING_SERVICE_H__
#define __BLE_ENVIRONMENTAL_SENSING_SERVICE_H__

#include <mbed.h>
#include "blecommon.h"
#include "Gap.h"
#include "GattServer.h"
#include "BLEDeviceInstanceBase.h"
#include "ble/BLE.h"

//#include "BLEDevice.h"

/* Environmental Sensing Service */
/* Service:  https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.environmental_sensing.xml */
/* Humidity characteristic: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.humidity.xml */
/* Temperature: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.temperature.xml */

class EnvironmentalSensingService {

public:
    typedef int16_t  TemperatureType_t;
    typedef uint16_t HumidityType_t;
    typedef uint32_t PressureType_t;
    typedef uint16_t WinddirectionType_t;
    /**
     * Constructor.
     *
     * param[in] _ble
     *               Reference to the underlying BLEDevice.
     * param[in] humidity percentage (16-bit unsigned, 2 decimals).
     *               initial value for the humidity value.
     * param[in] temperature in degrees Celsius (16-bit signed, 2 decimals).
     *               initial value for the temperature
     */
    /**
     * @brief   EnvironmentalService constructor.
     * @param   ble Reference to BLE device.
     * @param   temperature_en Enable this characteristic.
     * @param   humidity_en Enable this characteristic.
     * @param   pressure_en Enable this characteristic.
     */
    EnvironmentalSensingService(BLE& _ble) :
        ble(_ble),
        temperatureCharacteristic(GattCharacteristic::UUID_TEMPERATURE_CHAR, &temperature),
        humidityCharacteristic(0x2A6F, &humidity),
        pressureCharacteristic(GattCharacteristic::UUID_PRESSURE_CHAR, &pressure),
        winddirectionCharacteristic(0x2A71, &winddirection)
    {
        static bool serviceAdded = false; /* We should only ever need to add the information service once. */
        if (serviceAdded) {
            return;
        }
 
        GattCharacteristic *charTable[] = { &humidityCharacteristic,
                                            &pressureCharacteristic,
                                            &temperatureCharacteristic,
                                            &winddirectionCharacteristic };
 
        GattService EnvironmentalService(0x181A, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
 
        ble.gattServer().addService(EnvironmentalService);
        serviceAdded = true;
    }
 

    void updateHumidity(HumidityType_t newHumidityVal)
    {
        humidity = (HumidityType_t) (newHumidityVal * 100);
        ble.gattServer().write(humidityCharacteristic.getValueHandle(), (uint8_t *) &humidity, sizeof(HumidityType_t));
    }

    void updateTemperature(float newTemperatureVal)
    {
        temperature = (TemperatureType_t) (newTemperatureVal * 100);
        ble.gattServer().write(temperatureCharacteristic.getValueHandle(), (uint8_t *) &temperature, sizeof(TemperatureType_t));
    }
    void updatePressure(PressureType_t newPressureVal)
    {
        pressure = (PressureType_t) (newPressureVal * 10);
        ble.gattServer().write(pressureCharacteristic.getValueHandle(), (uint8_t *) &pressure, sizeof(PressureType_t));
    }
    void updateWinddirection(WinddirectionType_t winddirection) {
        ble.gattServer().write(winddirectionCharacteristic.getValueHandle(), (uint8_t *)&winddirection, sizeof(WinddirectionType_t));
    }      

        
private:
    BLE                  &ble;
    ReadOnlyGattCharacteristic<HumidityType_t>  humidityCharacteristic;
    ReadOnlyGattCharacteristic<TemperatureType_t> temperatureCharacteristic;
    ReadOnlyGattCharacteristic<PressureType_t> pressureCharacteristic;
    ReadOnlyGattCharacteristic<WinddirectionType_t>          winddirectionCharacteristic;
    PressureType_t    pressure;
    TemperatureType_t temperature;
    HumidityType_t    humidity;
    WinddirectionType_t winddirection;

};

#endif /* #ifndef __BLE_ENVIRONMENTAL_SENSING_SERVICE_H__*/


// *******************************ARM University Program Copyright © ARM Ltd 2015*************************************//
