/*
This file is part of CanFestival, a library implementing CanOpen Stack.

Copyright (C): Edouard TISSERANT and Francis DUPIN
mbed Port: sgrove

See COPYING file for copyrights details.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// Includes for the Canfestival driver
#include "canfestival.h"
#include "canopen_timer.h"
#include "mbed.h"

// make the system speed available for timer configuration
extern uint32_t SystemCoreClock;

// Define the timer registers
#define TimerAlarm        LPC_TIM1->MR0    // compare register to genterate an interrupt
#define TimerCounter      LPC_TIM1->TC     // free running timer counter register

/************************** Module variables **********************************/
// Store the last timer value to calculate the elapsed time
static TIMEVAL last_time_set = TIMEVAL_MAX;     

void initTimer(void)
/******************************************************************************
Initializes the timer, turn on the interrupt and put the interrupt time to zero
INPUT    void
OUTPUT    void
******************************************************************************/
{
    float prescaler = 0;
    TimerAlarm = 0;
    // power up the timer (if not already done)
    LPC_SC->PCONP |= (0x1 << 0x2);
    // reset the timer
    LPC_TIM1->TCR = 0x2;
    // set up the prescaler to create an 8us incrementing tick using Timer1
    switch ((LPC_SC->PCLKSEL0 >> 0x4) & 0x3){
            case 0x00:    prescaler = 0.000008 / (1.0 / (SystemCoreClock / 4.0));    break;
            case 0x01:    prescaler = 0.000008 / (1.0 / SystemCoreClock);            break;
            case 0x02:    prescaler = 0.000008 / (1.0 / (SystemCoreClock / 2.0));    break;
            case 0x03:    prescaler = 0.000008 / (1.0 / (SystemCoreClock / 8.0));    break;
    }
    LPC_TIM1->PR = static_cast<uint32_t>(prescaler);
    // MR0 is used as a match event to trigger an interrupt - managed through TimerAlarm
    LPC_TIM1->MCR |= (0x1 << 0);
    NVIC_EnableIRQ(TIMER1_IRQn);
    // start the timer
    LPC_TIM1->TCR = 0x1; 
    // Set timer for CANopen operation tick 8us , max time is 9+ hrs
    TimeDispatch();
}

void setTimer(TIMEVAL value)
/******************************************************************************
Set the timer for the next alarm.
INPUT    value TIMEVAL (unsigned long)
OUTPUT    void
******************************************************************************/
{
  TimerAlarm += (uint32_t)value;    // Add the desired time to timer interrupt time
}

TIMEVAL getElapsedTime(void)
/******************************************************************************
Return the elapsed time to tell the Stack how much time is spent since last call.
INPUT    void
OUTPUT    value TIMEVAL (unsigned long) the elapsed time
******************************************************************************/
{
    uint32_t timer = TimerCounter;    // Copy the value of the running timer
    // Calculate the time difference and return it
    return timer > last_time_set ? timer - last_time_set : last_time_set - timer;
}

///******************************************************************************
//Interruptserviceroutine Timer Compare for the stack CAN timer
//******************************************************************************/

extern "C" void TIMER1_IRQHandler()
{
    // clear all pending interrupts
    LPC_TIM1->IR = 0x3F;
    // store the time of the last interrupt occurance
    last_time_set = TimerCounter;
    // Call the time handler of the stack to adapt the elapsed time
    TimeDispatch();                          
}

