
#include "common.h"
#include "Atmel.h"
#include "Queue.h"
#include "RTCtime.h"

int AtmelHumanReadMode = 0;
int AtmelReadAddress = 0xff;
int AtmelReadInhibit = 0;
int AtmelReadFrameActive = 0;

// Two message queues.  AtmelLocalQueue is the internal queue for the read. It stores all events while enable is low.
// AtmelMessageQueue is the interface between the read frame and the main while loop. Real time access is controlled
// by the synchonize class.
Queue AtmelMessageQueue;
Queue AtmelLocalQueue;

//
AtmelStatus * DecodeAtmelBytes(UCHAR * ByteArray)
{
    /****************************************************************************
    Decode Atmel Touch Screen controller Data Bytes

    This Decodes the 10-bit Format as described in the Touch International Atmel
    Communications Protocol document for the I2C bus.

    ****************************************************************************/
    static AtmelStatus Status;

    memset(&Status,'\0',sizeof(Status));  // Init all fields to zero

    Status.FingerID = ByteArray[0];

    Status.X =  ByteArray[2]<<2;
    Status.X |=  (ByteArray[4]&0xC0) >>6;

    Status.Y =  ByteArray[3]<<2;
    Status.Y |=  (ByteArray[4]&0x0F) >>2;

    Status.Detect = ByteArray[1] & 0x80;
    Status.Press = ByteArray[1] & 0x40;
    Status.Release = ByteArray[1] & 0x20;
    Status.Move = ByteArray[1] & 0x10;
    Status.Vector = ByteArray[1] & 0x08;
    Status.Amp = ByteArray[1] & 0x04;
    Status.Suppress = ByteArray[1] & 0x02;

    Status.TouchArea = ByteArray[5];
    Status.TouchAmplitude = ByteArray[6];

    return &Status;
}

void AtmelDisplayReportStatus (Serial * PC, AtmelEventType * AtmelEvent)
{
    AtmelStatus * Status;


    Status = DecodeAtmelBytes(AtmelEvent->EventData);
    PC->printf("TouchEvent,");
    PC->printf("%08X%08X%08X,",  SIM->UIDMH, SIM->UIDML, SIM->UIDL); // 96 bit Unique ID
    PC->printf("%06d,", AtmelEvent->EventNumber);
    PC->printf("%02d/%02d/%04d,", AtmelEvent->EventTime.month, AtmelEvent->EventTime.day, AtmelEvent->EventTime.year); // Date
    PC->printf("%02d:%02d:%02d.%02d,", AtmelEvent->EventTime.hours, AtmelEvent->EventTime.minutes, AtmelEvent->EventTime.seconds, AtmelEvent->EventTime.hundredths); // Time
    PC->printf("%02d,",Status->FingerID);
    PC->printf("%d,%d,",Status->X, Status->Y); // Touch Position(X, Y), Origin is at bottom right.
    PC->printf("%d,%d,",Status->TouchArea, Status->TouchAmplitude); // Pixels affected and amplitude of pressure

    PC->printf("%s,%s,%s,%s,%s,%s,%s",  // Status flags
               Boolean(Status->Detect),
               Boolean(Status->Press),
               Boolean(Status->Release),
               Boolean(Status->Move),
               Boolean(Status->Vector),
               Boolean(Status->Amp),
               Boolean(Status->Suppress));

    PC->printf("\n");

}

void AtmelDisplayHumanStatus (Serial * PC, AtmelEventType * AtmelEvent)
{
    AtmelStatus * Status;



    Status = DecodeAtmelBytes(AtmelEvent->EventData);
    PC->printf("\n%02d:%02d:%02d.%02d\n",
               AtmelEvent->EventTime.hours,
               AtmelEvent->EventTime.minutes,
               AtmelEvent->EventTime.seconds,
               AtmelEvent->EventTime.hundredths); // Time
    PC->printf("Atmel Data: %06d\n            Finger#%02d (X,Y) = (%d,%d)\n",
               AtmelEvent->EventNumber,
               Status->FingerID,
               Status->X,
               Status->Y);
    PC->printf(  "            %-8s = %-7s\n","Detect",Boolean(Status->Detect));
    PC->printf(  "            %-8s = %-7s\n","Press",Boolean(Status->Press));
    PC->printf(  "            %-8s = %-7s\n","Release",Boolean(Status->Release));
    PC->printf(  "            %-8s = %-7s\n","Move",Boolean(Status->Move));
    PC->printf(  "            %-8s = %-7s\n","Vector",Boolean(Status->Vector));
    PC->printf(  "            %-8s = %-7s\n","Amp",Boolean(Status->Amp));
    PC->printf(  "            %-8s = %-7s\n","Suppress",Boolean(Status->Suppress));

    PC->printf("\n            TCHArea = %d\n",Status->TouchArea);
    PC->printf("            TCHAmplitude = %d\n",Status->TouchAmplitude);
}

void AtmelSendStatus (Serial * PC, AtmelEventType * AtmelEvent)
{
    if (AtmelHumanReadMode)
        AtmelDisplayHumanStatus(PC, AtmelEvent);
    else
        AtmelDisplayReportStatus(PC, AtmelEvent);

    ActivityIndicator = 0;

}

void AtmelHardwareReset(void)
{
    AtmelReset = 0;
    wait(0.5);
    AtmelReset = 1;

}

void AtmelReadEvent(AtmelEventType * EventRecord)
{
#define m_Detect   0x80
#define m_Press    0x40
#define m_Release  0x20
#define m_Move     0x01
#define m_Vector   0x08
#define m_Amp      0x04
#define m_Suppress 0x02
#define m_Reserved 0x01

    uint8 data[8];
    time_date_t  EventTime;
    GetRTclock(&EventTime);

    i2c.read(AtmelReadAddress, (char *)data, sizeof(data));


    memmove(&EventRecord->EventData, &data, sizeof(data));
    memmove(&EventRecord->EventTime, &EventTime, sizeof(time_date_t));
// Stats
    Statistics.AddEvent(); // Track number of actual, received events
    EventRecord->EventNumber = Statistics.GetEvent();

// Count individual flag events
    if (data[1] & m_Detect)
        Statistics.AddDetect();
    if (data[1] & m_Press)
        Statistics.AddPress();
    if (data[1] & m_Release)
        Statistics.AddRelease();
    if (data[1] & m_Move)
        Statistics.AddMove();
    if (data[1] & m_Vector)
        Statistics.AddVector();
    if (data[1] & m_Amp)
        Statistics.AddAmp();
    if (data[1] & m_Suppress)
        Statistics.AddSuppress();
    if (data[1] & m_Reserved)
        Statistics.AddReserved();
}



void AtmelSetAddress(int I2Caddress)
{
    AtmelReadAddress = I2Caddress <<1 | 0x01;
}

AtmelEventType EventRecord;

void AtmelDoRead(void)
{
    // Atmel controller set enable (~change) pin to low to signal new touch event
    while(!enable) {
        ActivityIndicator = 1;

        AtmelReadEvent(&EventRecord);
        AtmelLocalQueue.Put(&EventRecord, sizeof(AtmelEventType));
    }

    if (GuardCriticalSection.EnterMaster()) {
        while(AtmelLocalQueue.Get(&EventRecord, sizeof(AtmelEventType)))
            AtmelMessageQueue.Put(&EventRecord, sizeof(AtmelEventType));
        GuardCriticalSection.ExitMaster();
    }
}

void AtmelReadEvents(void)
{
    if(!AtmelReadInhibit) {
        AtmelReadFrameActive = 1;
        AtmelDoRead();

    }
    AtmelReadFrameActive = 0;
}



int AtmelDeviceReset(void)
{
    int ResetSuccess = 1;
    char data1[8], data2[8];

    while(AtmelReadFrameActive); // wait for read frame to finish.

    AtmelReadInhibit = 1;


    AtmelMessageQueue.CleanQueue();
    AtmelLocalQueue.CleanQueue();

    AtmelHardwareReset();

    wait(0.1);  // Let reset complete


    // At Reset, enable should go low; indicating the status messages
    if (enable) {
        ResetSuccess = 0;
    } else {
        //Reset generates two messages with FingerID = 01
        // First message = {01,90,DF,00,4C,00,00,00}
        // First message = {01,00,DF,00,4C,00,00,00}

        i2c.read(AtmelReadAddress, (char *)data1, sizeof(data1));
        i2c.read(AtmelReadAddress, (char *)data2, sizeof(data2));

        if(data1[0] != 0x01) ResetSuccess =0;
        if(data2[0] != 0x01) ResetSuccess =0;
        if(data1[1] != 0x90) ResetSuccess =0;
        if(data2[1] != 0x00) ResetSuccess =0;

        // Following the two FingerID=01 read, the ~Change (enable) discrete should go high.
        if (!enable) ResetSuccess = 0;

    }
    AtmelReadInhibit = 0;
    return ResetSuccess;
}



