
/*!
LT_I2C: Routines to communicate with ATmega328P's hardware I2C port.

@verbatim

LT_I2C contains the low level routines to communicate with devices using the
ATMega328's onboard hardware I2C port. Each routine checks the Two Wire Status
Register (TWSR) at the end of the transaction and returns 0 if successful and 1
if not successful.

I2C Frequency = (CPU Clock frequency)/(16+2(TWBR)*Prescaler)

TWBR-Two Wire Bit Rate Register
TWCR=Two Wire Control Register (TWINT TWEA TWSTA TWSTO TWWC TWEN - TWIE)
TWSR=Two Wire Status Register

Prescaler Values:
TWSR1  TWSR0  Prescaler
   0      0      1
   0      1      4
   1      0      16
   1      1      64

Examples:
CPU Frequency = 16Mhz on Arduino Uno
I2C Frequency  Prescaler  TWSR1  TWSR0  TWBR
  1khz         64         1      1      125
  10khz        64         1      1      12
  50khz        16         1      0      10
  100khz        4         0      1      18
  400khz        1         0      0      12

@endverbatim

REVISION HISTORY
$Revision: 5469 $
$Date: 2016-07-22 17:01:32 -0700 (Fri, 22 Jul 2016) $

Copyright (c) 2013, Linear Technology Corp.(LTC)
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of Linear Technology Corp.

The Linear Technology Linduino is not affiliated with the official Arduino team.
However, the Linduino is only possible because of the Arduino team's commitment
to the open-source community.  Please, visit http://www.arduino.cc and
http://store.arduino.cc , and consider a purchase that will help fund their
ongoing work.
*/

//! @defgroup LT_I2C LT_I2C: Routines to Communicate With ATmega328P's hardware I2C port.

/*! @file
    @ingroup LT_I2C
    Library for LT_I2C: Routines to Communicate With ATmega328P's hardware I2C port.
*/

//#include <Arduino.h>
#include <stdint.h>
//#include <util/delay.h>
#include "Linduino.h"
#include "LT_I2C.h"
#include "mbed.h"

//! CPU master clock frequency
#ifndef F_CPU
#define F_CPU 16000000UL
#endif
Serial pc1(USBTX, USBRX, 9600);


LT_I2C::LT_I2C() {
    i2c = new I2C(I2C_SDA, I2C_SCL);
    i2c->frequency(400000); //maximum from datasheet for ltc2991
}

LT_I2C::LT_I2C(PinName sda_, PinName scl_) {
    sda = sda_;
    scl = scl_;
    i2c = new I2C(sda_ , scl_);
    i2c->frequency(400000); //maximum from datasheet for ltc2991
}

// Read a byte of data at register specified by "command", store in "value"
int8_t LT_I2C::i2c_read_byte_data(uint8_t address, uint8_t command, uint8_t *value)
{
    int8_t ret = 0;
    
    char cmd1[1];
    cmd1[0] = command;
    char v[1];
    v[0] = 0;
    
    ret |= i2c->write((address<<1) | I2C_WRITE_BIT, cmd1, 1, true);  
    ret |= i2c->read((address<<1) | I2C_READ_BIT, v, 1, false);
    
    *value = v[0];
    
    if (ret == 0) {
        return 0; //success
    }
    return(1);                                       
}

// Write a byte of data to register specified by "command"
int8_t LT_I2C::i2c_write_byte_data(uint8_t address, uint8_t command, uint8_t value)
{
    int8_t ret = 0;
    char cmd[2];
    cmd[0] = command;
    cmd[1] = value;
    ret |= i2c->write((address<<1) | I2C_WRITE_BIT, cmd, 2, false);         

    if (ret == 0) {
        return 0; //success
    }
    return(1);
}

int8_t LT_I2C::i2c_read_word_data(uint8_t address, uint8_t command, uint16_t *value)
{
    int8_t ret = 0;
    

    union { 
       char b[2]; 
       uint16_t w;
    } data; 
    
    char writedata[2];
    writedata[0] = command; //msb
    writedata[1] = command+1; //lsb
    
    ret |= i2c->write((address << 1) | I2C_WRITE_BIT, &writedata[0], 1, true); 
    ret |= i2c->read((address << 1) | I2C_READ_BIT, &data.b[1], 1, false);
    ret |= i2c->write((address << 1) | I2C_WRITE_BIT, &writedata[1], 1, true); 
    ret |= i2c->read((address << 1) | I2C_READ_BIT, &data.b[0], 1, false);
    
    
    *value = data.w;
    
    if (ret == 0) {
        return 0;
    }
    return(1);
}

// Write a 16-bit word of data to register specified by "command"
int8_t LT_I2C::i2c_write_word_data(uint8_t address, uint8_t command, uint16_t value)
{
    int8_t ret = 0;
    
    union
    {
        uint8_t b[2];
        uint16_t w;
    } data;
    data.w = value;
    
    char cmd[3];
    cmd[0] = command;
    cmd[1] = data.b[1];
    cmd[2] = data.b[0];

    ret |= i2c->write((address<<1) | I2C_WRITE_BIT, cmd, 3, false);

    if (ret == 0) {
        return 0;
    }
    return(1);
}