#ifndef __BLEMASTERSERVICE_H__
#define __BLEMASTERSERVICE_H__

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "ble/DiscoveredCharacteristic.h"
#include "ble/DiscoveredService.h"
#include <UUID.h>
#include <BleSlaveService.h>



/* Max number of connectable nodes */
#define MAX_NUM_OF_NODES        7

/* brief Data type length */
#define ONE_BYTE_LEN         1
#define TWO_BYTES_LEN        2

#define ENV_DATA_LEN_SHORT  8
#define ENV_DATA_LEN_LONG  12
#define PRESS_DATA_LEN      4
#define HUM_DATA_LEN        2
#define TEMP_DATA_LEN       2
#define LED_DATA_LEN        ONE_BYTE_LEN
#define MIC_DATA_LEN        ONE_BYTE_LEN
#define LUX_DATA_LEN        TWO_BYTES_LEN
#define PRX_DATA_LEN        TWO_BYTES_LEN
#define ACC_DATA_LEN        6
#define GYR_DATA_LEN        6
#define MAG_DATA_LEN        6
#define SFUSION_DATA_LEN    6
#define X_DATA_LEN          2
#define Y_DATA_LEN          2
#define Z_DATA_LEN          2
#define WUP_DATA_LEN        0
#define STATUS_DATA_LEN     0

#define TSTAMP_LEN          2
#define NODE_ID_LEN         2
#define TYPE_ID_LEN         1

#define ATTR_HEAD_LEN       TSTAMP_LEN+NODE_ID_LEN+TYPE_ID_LEN
#define ENV_DATA_LEN        PRESS_DATA_LEN+HUM_DATA_LEN+TEMP_DATA_LEN+TEMP_DATA_LEN
#define MEMS_DATA_LEN       ACC_DATA_LEN+GYR_DATA_LEN+MAG_DATA_LEN

#define INT_VALUE           1
#define FLOAT_VALUE         2


/* Supported peripheral nodes ID */
#define NODE_ME1                0x01 /* MOTENV1 */
#define NODE_FL1                0x02 /* FLIGHT1 */
#define NODE_AM1                0x03 /* ALLMEMS1 */

#define BDADDR_SIZE             6
#define NODE_ID_B1              0 /* byte of the node BLE address used as 1st byte of the node ID */
#define NODE_ID_B2              1 /* byte of the node BLE address used as 2nd byte of the node ID */
#define NODE_ID_OFFSET          NODE_ID_B1

#define ME1_WUP_EVENT_ENABLED   1             /* To enable/disable Wake Up events from MOTENV1 nodes                                    */
#define AM1_WUP_EVENT_ENABLED   0             /* To enable/disable Wake Up events from ALLMEMS1 nodes                                   */
                                              /* It is present, without the need of any additional board, on the X-NUCLEO-IKS01A2.      */
#define MIC_PRESENT             (0x20 | 0x04) /* MIC on the X-NUCLEO-CCA02M1 (in this case the fw running on the node is the ALLMEMS1). */
#define PLUX_PRESENT            (0x03)        /* Proximity and Lux sensor on X-NUCLEO-6180XA1 (in this case the fw running on the node  */
                                              /* is the FLIGHT1).                                                                       */

#define BDADDR_SIZE             6
#define NODE_ID_B1              0 /* byte of the node BLE address used as 1st byte of the node ID */
#define NODE_ID_B2              1 /* byte of the node BLE address used as 2nd byte of the node ID */
#define NODE_ID_OFFSET          NODE_ID_B1

#define BUF_LEN 27

#define ENABLE  1
#define DISABLE 0

#define MANUF_SPECIFIC_TYPE     0xFF
#define STM32_NUCLEO            0x80
#define SENSOR_TILE             0x02
#define BLUE_COIN               0x03
#define WUP_POS                 0x0A
#define MIC_POS                 0x1A
#define PRX_POS                 0x19
#define ACC_POS                 0x17
#define SFUS_POS                0x07

/* Feature mask for Temperature1 */
#define FEATURE_MASK_TEMP1 0x00040000
/* Feature mask for Temperature2 */
#define FEATURE_MASK_TEMP2 0x00010000
/* Feature mask for Pressure */
#define FEATURE_MASK_PRESS 0x00100000
/* Feature mask for Humidity */
#define FEATURE_MASK_HUM   0x00080000
/* Feature mask for Accelerometer */
#define FEATURE_MASK_ACC   0x00800000
/* Feature mask for Gyroscope */
#define FEATURE_MASK_GYR   0x00400000
/* Feature mask for Magnetometer */
#define FEATURE_MASK_MAG   0x00200000
/* Feature mask for Sensor fusion */
#define FEATURE_MASK_SENSORFUSION 0x00000100
/* Feature mask for LED */
#define FEATURE_MASK_LED_EVENTS    0x20000000
/* Feature mask for WakeUp EVENT (here used as Motion Detector) */
#define FEATURE_MASK_WAKEUP_EVENTS 0x00000400
/* Feature mask for PROXIMITY EVENT  */
#define FEATURE_MASK_PROX          0x02000000
/* Feature mask for MICROPHONE */
#define FEATURE_MASK_MIC           0x04000000
/* Command to set the WakeUp notification (here used as Motion Detector) */
#define WAKEUP_NOTIFICATION_CMD    'w'
/* Command to set the notification frequency */
#define NOTIFICATION_FREQ_CMD      255
/* Notification frequency @50mS (default) */
#define NOTIFICATION_FREQ_50     0
/* Notification frequency @100mS */
#define NOTIFICATION_FREQ_100    1
/* Notification frequency @1S */
#define NOTIFICATION_FREQ_1000   10
/* Notification frequency @5S */
#define NOTIFICATION_FREQ_5000   50
/* Set the notification frequency to 100ms multiple */
#define SENDING_INTERVAL_100MS_MULTIPLE 10
/* Notification frequency when wifi is enabled */
#define NOTIFICATION_FREQ_WIFI_ON    NOTIFICATION_FREQ_1000 /* reduced to 1000 from NOTIFICATION_FREQ_5000 */
/* Notification frequency when wifi is disabled */
#define NOTIFICATION_FREQ_WIFI_OFF   NOTIFICATION_FREQ_1000
/* 1 to enable mems noifications */
#define ENABLE_MEMS 0
/* 1 to enable the scanning after disabling a notification */
#define SCAN_AFTER_NOTIFICATION 1

extern EventQueue   eventQ;



/* This var defines the central role
 * 0 if master
 * 1 if slave                     */
extern uint8_t role;

/* Pointer to var (uint8_t role)  */
//uint8_t * pRole;



/* Flag to indicate that the characteristic read is completed successful
 * 1                        successful
 * 0 or anything else       failed                                      */
extern uint8_t readCompleted;



/* Flag to indicate if the device discovery is completed successful
 * 1                        successful
 * 0 or anything else       failed                                      */
extern uint8_t discoveryCompleted;



/* Flag to indicate if the ch descriptor write is completed successful
 * 1                        successful
 * 0 or anything else       failed                                      */
extern uint8_t writeDescriptorCompleted;



/* Flag to indicate if the advertising is completed
 * 1                        completed
 * 0 or anything else       not completed                               */
extern uint8_t advEnds;


/* Flag to indicate if a change of notification is pending
 * 1                        pending
 * 0 or anything else       not pending                                 */
extern uint8_t notificationPending;


/* Flag to indicate if the ble stack is busy
 * 1                        busy
 * 0 or anything else       not busy                                    */
extern uint8_t stackBusy;



/* Supported peripheral nodes name */
//const char NODE_MOTENV[] =      "ME1V310";
const char NODE_MOTENV[] =      "ME1V320";
const char NODE_FLIGHT[] =      "FL1V310";
const char NODE_ALLMEMS[] =     "AM1V320";


/* Properties mask */
static uint8_t props_mask[] = {
  0x01,
  0x02,
  0x04,
  0x08,
  0x10,
  0x20,
  0x40,
  0x80
};


/* Enum of different states */
typedef enum {
  CONN_INIT,                                   // 0
  START_DEVICE_DISCOVERY,                      // 1
  DEVICE_FOUND,                                // 2
  DEVICE_NOT_FOUND,                            // 3
  DEVICE_DISCOVERY_COMPLETE,                   // 4
  START_DEVICE_CONNECTION,                     // 5
  DEVICE_CONNECTED,                            // 6
  DEVICE_CONNECTION_COMPLETE,                  // 7
  START_DISCOVERABLE_MODE,                     // 8
  DISCOVERABLE_MODE_SET,                       // 9
  START_SERVICE_DISCOVERY,                     //10
  SERVICE_DISCOVERED,                          //11
  SERVICE_DISCOVERY_COMPLETE,                  //12
  START_HARDWARE_SERV_CHARS_DISCOVERY,         //13
  HARDWARE_SERV_CHARS_DISCOVERED,              //14
  HARDWARE_SERV_CHARS_DISCOVERY_COMPLETE,      //15
  START_SOFTWARE_SERV_CHARS_DISCOVERY,         //16
  SOFTWARE_SERV_CHARS_DISCOVERED,              //17
  SOFTWARE_SERV_CHARS_DISCOVERY_COMPLETE,      //18
  START_CONFIGURATION_SERV_CHARS_DISCOVERY,    //19
  CONFIGURATION_SERV_CHARS_DISCOVERED,         //20
  CONFIGURATION_SERV_CHARS_DISCOVERY_COMPLETE, //21
  ENABLE_ME1_LED_NOTIFICATIONS,                //22
  ENABLE_ME1_WUP_NOTIFICATIONS,                //23
  NOTIFICATIONS_ENABLED,                       //24
  READ_INIT,                                   //25
  READING_PRX,                                 //26
  NOTIFY_PRX_TO_CLIENT,                        //27
  READING_ENVIRONMENTAL,                       //28
  NOTIFY_ENV_TO_CLIENT,                        //29
  READING_LED,                                 //30
  NOTIFY_LED_TO_CLIENT,                        //31
  READING_LUX,                                 //32
  NOTIFY_LUX_TO_CLIENT,                        //33
  READING_MIC,                                 //34
  NOTIFY_MIC_TO_CLIENT,                        //35
  READING_AGM,                                 //36
  NOTIFY_AGM_TO_CLIENT,                        //37
  READING_SFUSION,                             //38
  NOTIFY_SFUSION_TO_CLIENT,                    //39
  READING_PRESSURE,                            //40
  READING_HUMIDITY,                            //41
  READING_TEMPERATURE,                         //42
  ALL_DATA_READ,                               //43
  NOTIFICATIONS_DATA_READ,                     //44
  DISABLE_NOTIFICATIONS,                       //45
  READING_DISCONNECTION,                       //46
} Status_t;


/* (Service) Handle struct */
typedef struct {
  uint16_t start_h;
  uint16_t end_h;
} Handle_t;

typedef DiscoveredCharacteristic::Properties_t Properties_t;

typedef BLEProtocol::AddressBytes_t tBDAddr;

/* Peripheral device info struct */
typedef struct {
  uint8_t  dev_v;           // per device ID
  tBDAddr  bdaddr;          // per device addr
} devInfo_t;


/* Struct contains the peripheral devices informations  */
typedef struct {
  devInfo_t                 devInfo[MAX_NUM_OF_NODES];
  uint8_t                   discovery_enabled;
  bool                      device_found;
  uint8_t                   connDevices;
  uint8_t                   discDevices;
  uint8_t                   connDeviceIdx;
  uint8_t                   readDeviceIdx;
  uint16_t                  connection_handle[MAX_NUM_OF_NODES];
  bool                      is_connected[MAX_NUM_OF_NODES];
  bool                      is_disconnected[MAX_NUM_OF_NODES];
  bool                      is_unconnectable[MAX_NUM_OF_NODES];

  Handle_t                  gen_access_profile_handle[MAX_NUM_OF_NODES];
  Handle_t                  gen_attribute_profile_handle[MAX_NUM_OF_NODES];
  Handle_t                  hardware_service_handle[MAX_NUM_OF_NODES];
  Handle_t                  configuration_service_handle[MAX_NUM_OF_NODES];
  Handle_t                  software_service_handle[MAX_NUM_OF_NODES];

  // Characteristic handles
  uint16_t                  environmental_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  pressure_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  humidity_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  temperature_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  led_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  cfg_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  wup_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  mic_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  prx_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  lux_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  agm_char_handle[MAX_NUM_OF_NODES];
  uint16_t                  sfusion_char_handle[MAX_NUM_OF_NODES];

  uint8_t                   led_char_read[MAX_NUM_OF_NODES];
  uint8_t                   wup_char_read[MAX_NUM_OF_NODES];
  uint8_t                   mic_char_read[MAX_NUM_OF_NODES];
  uint8_t                   prx_char_read[MAX_NUM_OF_NODES];
  uint8_t                   agm_char_read[MAX_NUM_OF_NODES];
  uint8_t                   sfusion_char_read[MAX_NUM_OF_NODES];
  uint8_t                   wup_event[MAX_NUM_OF_NODES];
  uint8_t                   mic_event[MAX_NUM_OF_NODES];
  uint8_t                   prx_event[MAX_NUM_OF_NODES];
  uint8_t                   agm_event[MAX_NUM_OF_NODES];
  uint8_t                   sfusion_event[MAX_NUM_OF_NODES];
  uint8_t                   prx_on[MAX_NUM_OF_NODES];
  uint8_t                   agm_on[MAX_NUM_OF_NODES];
  uint8_t                   sfusion_on[MAX_NUM_OF_NODES];
  uint8_t                   mic_event_enabled;
  uint8_t                   prx_event_enabled;
  uint8_t                   agm_event_enabled;
  uint8_t                   sfusion_event_enabled;
  uint8_t                   wup_event_enabled;
  uint8_t                   acc_event_enabled;
  uint8_t                   gyr_event_enabled;
  uint8_t                   mag_event_enabled;
  Status_t                  status;

} PeripheralDevices_t;




/* List of type id */
typedef enum {
  UNKNOWN_TYPE_ID,  //0
  PRESS_TYPE_ID,    //1   (its data value is 4 bytes long)
  HUM_TYPE_ID,      //2   (its data value is 2 bytes long)
  TEMP_TYPE_ID,     //3   (its data value is 2 bytes long)
  LED_TYPE_ID,      //4   (its data value is 1 byte long)
  PRX_TYPE_ID,      //5   (its data value is 2 bytes long)
  WUP_TYPE_ID,      //6   (no data value)
  MICLEVEL_TYPE_ID, //7   (its data value is 1 byte long)
  LUX_TYPE_ID,      //8   (its data value is 2 bytes long)
  ACC_TYPE_ID,      //9   (its data value is 6 bytes long)
  GYR_TYPE_ID,      //10  (its data value is 6 bytes long)
  MAG_TYPE_ID,      //11  (its data value is 6 bytes long)
  STATUS_TYPE_ID,   //12  (no data value)
  SFUSION_TYPE_ID   //13  (its data value is 8 bytes long)
                    //  (increase the enum to add new data type IDs)
} Data_Type_ID_t;



/* Struct of the generic DiscoveredCharacteristic node */
struct DiscoveredCharacteristicNode{
    //data
    DiscoveredCharacteristic data;
    //pointer to next node
    DiscoveredCharacteristicNode * next;
};


/* Header pointer of the DiscoveredCharacteristic list */
extern DiscoveredCharacteristicNode * headCharacteristic[MAX_NUM_OF_NODES];
extern DiscoveredCharacteristicNode * tmp[MAX_NUM_OF_NODES];


/* Primary Service UUID expected from Sensor demo peripherals */
extern UUID::ShortUUIDBytes_t GENERIC_ACCESS_PROFILE_UUID;
extern UUID::ShortUUIDBytes_t GENERIC_ATTRIBUTE_PROFILE_UUID;




/* Services UUID */        //  1b:c5:d5:a5:02:00:b4:9a:e1:11:XX:XX:XX:XX:XX:XX
extern UUID::LongUUIDBytes_t HARDWARE_SERVICE_UUID;

extern UUID::LongUUIDBytes_t CONFIG_SERVICE_UUID;

extern UUID::LongUUIDBytes_t SOFTWARE_SERVICE_UUID;




/* Characteristics UUID */
/* ENVIRONMENTAL */
extern UUID::LongUUIDBytes_t ENVIRONMENTAL_CHAR_UUID;

/* ENVIRONMENTAL for Sensor Tile */
extern UUID::LongUUIDBytes_t ENVIRONMENTAL_ST_CHAR_UUID;

/* ACC + GYRO + MAG */
extern UUID::LongUUIDBytes_t ACCGYROMAG_CHAR_UUID;

/* MEMS SENSOR FUSION */
extern UUID::LongUUIDBytes_t SFUSION_CHAR_UUID;

/* LED */
extern UUID::LongUUIDBytes_t LED_CHAR_UUID;

/* WAKEUP */
extern UUID::LongUUIDBytes_t WAKEUP_EVENT_CHAR_UUID;

/* MICROPHONE */
extern UUID::LongUUIDBytes_t MIC_EVENT_CHAR_UUID;

/* PROXIMITY */
extern UUID::LongUUIDBytes_t PROXIMITY_CHAR_UUID;

/* LUX */
extern UUID::LongUUIDBytes_t LUX_CHAR_UUID;

/* CONFIGURATION */
extern UUID::LongUUIDBytes_t CONFIG_CHAR_UUID;


/*-----  -----*/
extern uint8_t wifi_data[256];
extern uint8_t new_data;
extern uint8_t *data;
extern uint8_t wifi_present;

extern char print_msg_buff[512];
//extern UART_HandleTypeDef UartMsgHandle;

//extern uint16_t connection_handle = 0;

extern uint8_t attribute_value[20];
extern uint8_t star_attr_value[256];
//extern uint8_t bnrg_expansion_board_type = IDB04A1; /* at startup, suppose the X-NUCLEO-IDB04A1 is used */
extern uint8_t notification_freq;


/*---- Main functions declaration ----*/
/* Complete the initialization of ble module */
void bleInitComplete                        (BLE::InitializationCompleteCallbackContext *params);

/* scheduleBleEventsProcessing */
void scheduleBleEventsProcessing            (BLE::OnEventsToProcessCallbackContext* context);

/* Initialization error handling */
void onBleInitError                         (BLE &ble, ble_error_t error);

void checkStatus                                (void);


/*---- Master service functions declarations ----*/
/* Scan for new peripheral nodes and establish new connections */
void connectionProcess                          (void);

/* Read data from peripheral nodes */
void readingProcess                             (void);

/* Create a null gatt read pointer */
GattReadCallbackParams * nullGattReadCallbackP  (uint16_t connection_handle);


/* Print the central mac addr */
void printMacAddress                            (void);

/* Complete the initialization of ble module */
void bleInitComplete                            (BLE::InitializationCompleteCallbackContext *params);

/* Callback functions */
void disconnectionCallback                      (const Gap::DisconnectionCallbackParams_t *);

void connectionCallback                         (const Gap::ConnectionCallbackParams_t *params);

void advertisementCallback                      (const Gap::AdvertisementCallbackParams_t *params);

void onStopScan                                 (Gap::TimeoutSource_t t);

void serviceDiscoveryCallback                   (const DiscoveredService *service);

void characteristicDiscoveryCallback            (const DiscoveredCharacteristic *characteristicP);

void discoveryTerminationCallback               (Gap::Handle_t connectionHandle);

/* Method called after the reading of a characteristic */
void readCharacteristicCallback                 (const GattReadCallbackParams *response);

/* Method called after a periph descriptor is written */
void perDescriptorWrittenCallback               (const GattWriteCallbackParams* event);



/* Save the found peripheral device in the struct containing all the
   found peripheral devices. */
void saveDeviceFound                            (uint8_t adv_type, BLEProtocol::AddressBytes_t addr,
                                                 uint8_t data_length, const uint8_t* data_RSSI, uint8_t pos, uint8_t dev_v,
                                                 uint8_t wup_event, uint8_t mic_event, uint8_t prx_event,
                                                 uint8_t agm_event, uint8_t sfusion_event);

/* Connect the device as Master to a peripheral device */
void connectPeripheral                          (void);

/* Service discovery */
void discoverServices                           (void);

/* Initialize the struct where all peripheral devices are stored */
void initProcess                                (void);

/* Start the discovery of new peripheral nodes */
void startDiscovery                             (void);

/* This function retrieves the peripheral device index from the connection handle */
void getDeviceFromConnHandle                    (uint16_t handle, uint8_t *index, tBDAddr devAddr);

/* This method reads the characteristic in input */
void readSensorData                             (const DiscoveredCharacteristic &characteristic,
                                                 uint16_t connection_handle, uint8_t index);

/* Set the new status */
void setNewStatus                               (void);

/* Method that enables characteristic notifications */
void enableNotifications                        (uint16_t conn_handle, uint8_t index, uint8_t dev_v, uint8_t set_mode);

/* Write a characteristic descriptor */
void writeCharacDescriptor                      (uint16_t conn_handle, uint16_t attr_handle, uint8_t attr_len,
                                                 uint8_t* attr_value);

/* Get error */
ble_error_t writeCharacDescriptorWithError      (uint16_t conn_handle, uint16_t attr_handle, uint8_t attr_len,
                                                 uint8_t* attr_value);

/* Write a characteristic value without waiting for any response */
void writeCharacValueWithoutResp                (uint16_t conn_handle, uint16_t attr_handle, uint8_t attr_len,
                                                 uint8_t* attr_value);

/* Get error */
ble_error_t writeCharacValueWithoutRespWithError       (uint16_t conn_handle, uint16_t attr_handle, uint8_t attr_len,
                                                        uint8_t* attr_value);

/* Builds a Jason format string */
void Build_MEMS_Packet                          (uint8_t *attr_value, char *data_type, tBDAddr devAddr, uint8_t num_type);


/* Create new attr value for the master */
void Create_New_Attr_Value                      (uint8_t *tstamp, tBDAddr  devAddr, uint8_t data_type_id, uint8_t *data, uint8_t data_length);

/* Method called when there is a notification from the sever */
void onNotificationCallback                     (const GattHVXCallbackParams* event);

void scanTimeOut                                (void);
/*----------------------------------------------------------------------------*/




/* DiscoveredCharacteristic List methods */

/* Creating a new DiscoveredCharacteristicNode (struct r308)*/
DiscoveredCharacteristicNode * createDCNode     (const DiscoveredCharacteristic &ch, DiscoveredCharacteristicNode *next);

/* Prepend a DiscoveredCharacteristicNode (struct r308)*/
DiscoveredCharacteristicNode * prependDCNode    (DiscoveredCharacteristicNode *head, const DiscoveredCharacteristic *characteristic);

/* Append a DiscoveredCharacteristicNode (struct r308)*/
DiscoveredCharacteristicNode * appendDCNode     (DiscoveredCharacteristicNode *head, const DiscoveredCharacteristic *characteristic);

/* Number of DiscoveredCharacteristic nodes */
int countElements                               (DiscoveredCharacteristicNode *head);

/* Search for a DiscoveredCharacteristic node by declaration handle */
DiscoveredCharacteristicNode * searchDCNode     (DiscoveredCharacteristicNode *head, uint16_t declHandle);

/* Delete the DCN list */
void deleteDCNList                              (DiscoveredCharacteristicNode *head);

/*----------------------------------------------------------------------------*/







#endif /*__BLEMASTERSERVICE_H__*/
