
/* 
Generated: Enet_SPI_LPC1768.bin

mbed test software for Ethernet HTTPServer with simultaneous SPI for 4 data byte updates

Version Date    Author  Change    
0.2:    06/10/11    LS  Add block checksum character to SPI data
0.3:    07/10/11    LS  Add SPI read from cmd_buf
0.4:    13/10/11    LS  Integrated with HTTP server example code
0.5:    18/10/11    LS  Initial file handling
0.6:    26/10/11    LS  File IO handled by SPI_server class
0.78:   17/11/11    LS  Add SNMP files for system info reporting, but HTTP svr.bind disabled
0.79    12/12/11    LS  Reads network addresses from file & uses enterprise address
0.80    18/12/11    LS  Save last number read in read_addrs_from_file(). 
0.81    22/12/11    LS  Add new_cmd_idx to indicate which parameter has been set (0 for none). 
0.82    22/12/11    LS  Add blinker to show comms activity 
0.83    23/12/11    LS  Clear cmd_buf[0] after SPI been read 4 times 
0.84    27/12/11    LS  snmp_fw_version & f3k_fw_version saved as 4 char strings 
0.85    29/12/11    LS  Update f3k_sys_set_value() & f3k_sys_set_test() to enable net address change
0.90    29/12/11    LS  Add 3-step routine for passing commands over SPI bus from cmd_buf
0.94    10/01/12    LS  Add led_link  = LPC_GPIO1->FIOPIN & (1<<25) in netservice.cpp
0.95    11/01/12    LS  Fix f3k_speed reporting error in private_mib.c
0.96    12/01/12    LS  Move led_link  = LPC_GPIO1->FIOPIN & (1<<25) to eth_poll in eth_drv.cpp
0.98    26/04/12    LS  Increase snmp_publiccommunity[7] to [21] and allow this to be set from file
0.99    08/06/12    LS  #define SNMP_TRAP_DESTINATIONS 1. Declare th_ad[4] for host trap address
                        Add init_traps() to ensure head-head link status trap is sent
        21/06/12        Add check for changes to IO bytes and chk_SPI_active() every 60 seconds
1.00    22/06/12    LS  Save SNMP changes to net addresses & mask4_IO_registers in net_adrs.txt

http://192.168.99.99/rpc/offset,604 writes 604 to reg_offset
http://192.168.99.99/rpc/led2/write,1 writes 1 to led2
http://192.168.99.99/files/IO1000PX.htm uploads file IO1000PX.htm
*/

#include "mbed.h" 
#include "EthernetNetIf.h"
#include "HTTPServer.h"
#include "lwip/snmp_structs.h"
#include "lwip/snmp_msg.h"
#include "IO_file.h"

#define LPC1768         0x60          // SPI address of LPC1768 module
#define READ_IO         0x01          // bit in address field for data reads
#define SPI_BYTES       4             // # of data bytes in SPI message
#define INV_SPI_CNT     0xfffff       // invalid (over-range SPI count
#define NUM_READ        8             // # of SPI replies for valid reads
#define NUM_INDX        4             // # of SPI replies for valid cmd_buf index
#define MAX_CMD         4             // size of CMD buffer
#define NO_CMD          0xff          // data on cmd_buf[0] when NO_CMD to be sent
#define NO_UPDATE       0             // NO change to Ethernet addresses
#define UPDATED         0xff          // Ethernet addresses updated in file
#define NET_ADRS_FILE   "/webfs/net_adrs.txt"


const char * snmp_fw_version = "1.a";

/* default ethernet address setup */             
unsigned char ip_ad[4] = {192,168,99,99};
unsigned char msk_ad[4] = {255,255,255,0};
unsigned char gwy_ad[4] = {192,168,99,253};
unsigned char dns_ad[4] = {192,168,99,253};
unsigned char th_ad[4] = {146,38,105,49};   // v0.99: host address for SNMP traps
unsigned char net_addrs_set = false;        // flags net address update from SNMP 
unsigned char net_addrs_chng = NO_UPDATE;   // flags net address changed in main CPU 
unsigned char byte_changed = false;         // true if any IO regs have changed
unsigned char all_IO_updated = false;       // true if all 32 IO regs have been read
char hex_mask[12];                          // IO mask as hex string
unsigned long mask4_IO_registers = 0x3fffffff;  // mask for IO regs ignoring IO31 & 32
unsigned long spi_count = 0;                // last SPI_COUNT saved in SPI interrupt
unsigned char spi_active = true;            // avoid false trap message at start

static u8_t syslocation[32]; /* v0.9b */
u8_t sysloc_len;

SPISlave spi_slave(p11, p12, p13, p14); // mosi, miso, sclk, ssel
  
Ticker comms_tick;
Ticker sys_tick;                      // system tick for SNMP sysuptime report
Ticker spi_tick;                      // tick for monitoring SPI activity
Timeout cold_start_delay;

DigitalOut led1(LED1, "led1");        // mimics head-head link status o/p in main CPU I/O
DigitalOut led2(LED2, "led2");
DigitalOut led3(LED3, "led3");
DigitalOut led4(LED4, "led4");
DigitalOut led_link(p29);             // Green link status: ON=good, Blink=Activity
DigitalOut led_speed(p30);            // Yellow link speed: ON for 100Mb/s 

unsigned char blink_led = false;            
unsigned char trap_init = false;      // true after cold_start trap sent          

LocalFileSystem fs( "webfs" );

IpAddr mip, mnm, mgw, mdns;           // instantiate from core\ipaddr.cpp

//EthernetNetIf eth;                    // use if DNS sets up address

HTTPServer svr;
SPI_Server fsvr( "fsvr" );

Serial pc( USBTX, USBRX );        // for diagnostic serial comms to PC via USB

/****************************************************************************/ 
/*                      SPI interface data and routines                     */
/****************************************************************************/
/* commands sent to main CPU cpu from cmd_buf[0] and 
   waiting commands stored in higher indices until sent */
unsigned char cmd_buf[ MAX_CMD ];         // store for commands from Ethernet
unsigned char new_cmd_idx;


/****************************************************************************/ 
/*              SPI_Server: 
* Description:  Instantiates Class to hold & access all data read from RAM by SPI
*               Called by FSHandler::doGet() in services\http\server\impl
* Parameters:   name of object instantiated
* Returns:      NONE
*/
/****************************************************************************/ 
SPI_Server::SPI_Server(const char * name)
{
int idx;
  for ( idx = 0; idx < MAX_RAM; idx++ )
  {
    ram_img[ idx ] = (MAX_RAM - idx) & 0xff;
  }
  reg_offset = 0;
}

SPI_Server::~SPI_Server()
{
  if(fp)  fclose(fp);
}

/****************************************************************************/ 
/*              update_IO_file: 
* Description:  Reads data from ram_img[] into IO1000PX.htm for web access
*               Called by FSHandler::doGet() in services\http\server\impl
* Globals used: NONE 
* Parameters:   NONE
* Returns:      0 if OK, 1 if failed to open files
*/
/****************************************************************************/ 
int SPI_Server::update_IO_file()
{
int c, i;                           // characters read/written in files
unsigned short chnl, dval;          // data from 1000PX

  DBG("\r\nIn SPI_Server::update_IO_file()\r\n");
  FILE *fr = fopen("/webfs/hdr.htm", "r");
  if(!fr) {
    fprintf(stderr, "\r\nFile /webfs/hdr.htm could not be opened!\r\n");
    exit(1);
  }
  else
  {
    fp = fopen( "/webfs/IO1000PX.htm", "w" );
    if(!fp) {
      fprintf(stderr, "\r\nFile /webfs/IO1000PX.htm could not be opened to write!\r\n" );
      exit(1);
    }
    else
    {
      while ( (c = fgetc(fr)) != EOF )
      {
        fputc( c, fp );
      }
      fclose(fr);
  DBG("\r\nIn SPI_Server::update_IO_file() - header copied\r\n");
      for ( i = 0; i < 16; i++ )
      {
        chnl = reg_offset + i;
        dval = 256 * ram_img[ 2*chnl ] + ram_img[ 2*chnl + 1 ]; 
        fprintf(fp, "    <TR VALIGN=TOP>\r\n");
        fprintf(fp, "        <TD WIDTH=124>\r\n");
        fprintf(fp, "            <P>%02d</P>\r\n", chnl );
        fprintf(fp, "        </TD>\r\n");
        fprintf(fp, "        <TD WIDTH=124>\r\n");
        fprintf(fp, "            <P>%d</P>\r\n", dval );
        fprintf(fp, "        </TD>\r\n");
        fprintf(fp, "        <TD WIDTH=124>\r\n");
        fprintf(fp, "            <P>%02x</P>\r\n", (dval/256) );
        fprintf(fp, "        </TD>\r\n");
        fprintf(fp, "        <TD WIDTH=124>\r\n");
        fprintf(fp, "            <P>%02x</P>\r\n", (dval & 0xff) );
        fprintf(fp, "        </TD>\r\n");
        fprintf(fp, "    </TR>\r\n");
      }
      fprintf(fp, "</TABLE>\r\n<P><BR><BR>\r\n</P>\r\n</BODY>\r\n</HTML>\r\n");
    }
  }
  fclose(fp);
//  fclose(fr);
  return 0;
}

int SPI_Server::read( void )
{
  DBG("\r\nIn SPI_Server::read()\r\n");
  return reg_offset;
}

void SPI_Server::write( int new_offset )
{
  DBG("\r\nIn SPI_Server::write()\r\n");
  if ( (new_offset >= 0) && (new_offset < MAX_RAM) ) reg_offset = new_offset; 
}

unsigned char SPI_Server::get_byte( unsigned short idx )
{
  return ram_img[ idx ];
}

void SPI_Server::put_byte( unsigned short idx, unsigned char new_byte )
{
  if ( idx < MAX_RAM )  
  ram_img[ idx ] = new_byte; 
}

unsigned short SPI_Server::get_error( void )
{
  // return error register value from main CPU 
  return (256 * ram_img[ 1786 ]) + ram_img[ 1787 ];
}
  
unsigned short SPI_Server::get_SPI_count( void )
{
  // return SPI write count from main CPU - indicates main CPU running OK 
  return (256 * ram_img[ SPI_COUNT ]) + ram_img[ SPI_COUNT + 1 ];
}

unsigned char * SPI_Server::net_address_ptr( void )
{
    return &ram_img[ NET_IP_ADDRESS ];
}

unsigned char * SPI_Server::trap_host_ptr( void )
{
    return &ram_img[ TRAP_HOST_ADDRESS ];
}

#if 0
//#ifdef MBED_RPC trap_host_ptr
const struct rpc_method *SPI_Server::get_rpc_methods() { 
    static const rpc_method methods[] = {
        { "read", rpc_method_caller<int, SPI_Server, &SPI_Server::read> },
        { "write", rpc_method_caller<SPI_Server, int, &SPI_Server::write> },
        RPC_METHOD_SUPER(Base)                          //Stream)
    };
    return methods;
}

struct rpc_class *SPI_Server::get_rpc_class() {
    static const rpc_function funcs[] = {
        { "new", rpc_function_caller<const char*, int, &Base::construct<SPI_Server, int> > },
        RPC_METHOD_END
    };
    static rpc_class c = { "SPI_Server", funcs, NULL };
    return &c;
}
#endif

void check4write_offset( char * req )
{
unsigned char l, c, i;
char v[ 20 ] = {0};
char s[ 20 ] = {0};
char * p;

  strcpy( v, "/offset," );
  p = strstr( req, v );
  strcpy( s, p );
  if ( p != NULL )
  {
    i = strlen(v);
    l = strlen(s);
    if ( (l > i) && (l < (i + 6))  )
    {
      l -= i;
      for ( c = 0; c < l; c++ )
      {
        v[ c ] = s[ i + c ]; 
      } 
      v[ c ] = 0;
      fsvr.write( atoi(v) );
    } 
  }  
}

/****************************************************************************/ 
/*              get_SPI_data: 
* Description:  Implements transaction as SPI slave.
*               Reads 4 data bytes from SPI bus into destbuf for SPI write, or
*               Puts 1 byte from srcbuf onto SPI for SPI read 
*               Called by main when spi_slave.receive interrupts
*               LPC1768 SPI slave handles a maximum of 8 bytes per transfer
* Globals used: NONE  
* Parameters:   IN/OUT: unsigned char * srcbuf : buffer to read from
* Returns:      int buf_idx : index to SPI data in destbuf/srcbuf
*/
/****************************************************************************/ 
int get_SPI_data( unsigned char * srcbuf )
{
static unsigned char rd_cnt;    // v0.83: count of times valid data put on bus
int buf_idx = 0xffff;           // index to save SPI data into destbuf
int idx;                        // index to compare with net address block
unsigned char spi_buf[ 8 ];     // temporary buffer for SPI data received
unsigned char byte_cnt;         // count of SPI data bytes received
unsigned char spiv;             // SPI value - initial slave address
unsigned char bcc = 0;          // received block check character
unsigned char bcc_ok = false;   // default to reject messages without bcc

    spiv = spi_slave.read();   // Read byte from master
    if ( (spiv & 0xfe) == (LPC1768 & 0xfe) )
    {
      if ( spiv & READ_IO )
      {
        led4 = 1;
        led3 = 0;
        if ( *srcbuf < NO_CMD )
        {                             // new data to be read by main CPU cpu
          if ( ++rd_cnt >= NUM_READ )
          {                           // v0.83: clear data after 4 reads
            spi_slave.reply( NO_CMD );  // put invalid data in SPI buffer
            *srcbuf = NO_CMD;
            rd_cnt = 0;
          }
          else if ( rd_cnt >= NUM_INDX )
          {                           // next put data from srcbuf in SPI buffer
            spi_slave.reply( *srcbuf );  
          }
          else                        // put new_cmd_idx on SPI bus first
          {
            spi_slave.reply( NO_CMD - new_cmd_idx );  // put new_cmd_idx in SPI buffer
          }
        } // end if ( *srcbuf < 0xff )
        else
        {
          rd_cnt = 0;                 // re-initialise read count
          spi_slave.reply( NO_CMD );  // put invalid data in SPI buffer
        }
      }
      else                                  // NOT READ_IO - data write
      {
        buf_idx =  spi_slave.read();        // Read ms index from master
        bcc ^= buf_idx;                     // Start checksum calculation
        buf_idx *=  256;                    // Make ms index from master
        buf_idx +=  spi_slave.read();       // Read ls index from master
        bcc ^= (buf_idx & 0xff);
        byte_cnt = 0;
        while( spi_slave.receive() ) { 
          if( byte_cnt < SPI_BYTES ) {      // Read 4 data bytes
            spi_buf[ byte_cnt ] = spi_slave.read();
            bcc ^=  spi_buf[ byte_cnt++ ];
          }
          else if ( byte_cnt == SPI_BYTES ) {
            spiv = spi_slave.read();
            bcc_ok = ( bcc == spiv );
          }
        } // end while ( spi_slave.receive() )
        if ( bcc_ok && (buf_idx < MAX_RAM) )
        {
            for( byte_cnt = 0; byte_cnt < SPI_BYTES; byte_cnt++ )
            {
               idx = buf_idx + byte_cnt;
               if ( idx <= MAX_IO_REG_BYTE )
               {                        // check for IO changed from last saved
                 if ( (spi_buf[ byte_cnt ] != fsvr.get_byte( idx )) && !byte_changed ) 
                 {
                   if ( mask4_IO_registers & (1 << (idx/2)) )
                   {                    // masked-in register has changed
                     byte_changed  = true;
                   }
                 }
                 if ( idx == MAX_IO_REG_BYTE )
                 {                      // after each I/O scan
                   all_IO_updated = true;
                   spi_count = fsvr.get_SPI_count();  // save last SPI_count value
                 }
               }
               else if ( idx == NET_ADDRESS_CHANGE )
               {                    // v0.93: check for net address update from main CPU CPU
                 if ( spi_buf[ byte_cnt ] == NO_UPDATE )
                 {
                    net_addrs_chng = NO_UPDATE;
                 }
                 else if ( net_addrs_chng != UPDATED )
                 {
                    net_addrs_chng = spi_buf[ byte_cnt ];
                 }
               }
               fsvr.put_byte( idx, spi_buf[ byte_cnt ] ); // save byte read from SPI
            }
            led3 = 0;
        }
        else                        // checksum failed
        {
            led3 = 1;
        }
        led4 = 0;
      } // end else not READ_IO
    } // end  if ( (spiv & 0xfe) == (LPC1768 & 0xfe) )
  return buf_idx;
}

/****************************************************************************/ 
/*              chk_SPI_active: 
* Description:  checks spi_count has been updated in SPI interrupt
*               to be called by timer, independently of SPI interrupt
* Globals used: spi_active, spi_count
* Parameters:   NONE
* Returns:      NONE
*/
/****************************************************************************/
void chk_SPI_active( void )
{
  if ( spi_count == INV_SPI_CNT )
  {
    spi_active = false;                 // flag SPI failure
  }
  else
  {
    spi_count = INV_SPI_CNT;            // set to be overwritten by next SPI update
    spi_active = true;                  // SPI updated OK
  }
}

/****************************************************************************/ 
/*              get_hex_mask: 
* Description:  converts long mask4_IO_registers to hexadecimal string
* Globals used: mask4_IO_registers, hex_mask
* Parameters:   NONE
* Returns:      pointer to hex_mask
*/
/****************************************************************************/
char * get_hex_mask( void )
{
unsigned char i;
  for ( i = 0; i < 12; i++ )
  {
    hex_mask[ i ] = 0;
  }
  sprintf(hex_mask, "%08x", mask4_IO_registers);
  return hex_mask;
}

/****************************************************************************/ 
/*              convert2decimal: 
* Description:  converts hexadecimal string to long mask4_IO_registers
* Globals used: mask4_IO_registers
* Parameters:   u8_t * uc_ptr: pointer to hex-formatted mask string
*               u16_t len: length of string
* Returns:      NONE
*/
/****************************************************************************/
void convert2decimal( u8_t * uc_ptr, u16_t len )
{
char ch, i;
unsigned long temp_sum = 0;

  for ( i = 0; i < len; i++ )
  {
    ch = (char)*(uc_ptr + i);
    if ( islower( ch ) )
    {
      ch = ch - 'a' + 10;
    }
    else if ( isupper( ch ) )
    {
      ch = ch - 'A' + 10;
    }
    else if ( isdigit( ch ) )
    {
      ch = ch - '0';
    }
    else
    {
      temp_sum = 0xfffffff;
      ch = 15;
      i = len;                          // end on error;
    }
    temp_sum = 16 * temp_sum + ch;
  }
  mask4_IO_registers = temp_sum;
}

void send_ram( void )
{
static int idx;
char i;

  if ( (idx & 0x7ff) == 0 )
  {
      pc.printf("CMD data %x", idx );
      for ( i = 0; i < MAX_CMD; i++ )
      {
        pc.printf(", %x", cmd_buf[ i ] );
      }
      pc.printf( "\r\n" );  
      pc.printf("RAM data %x", idx & 0x7ff );
      for ( i = 0; i < 8; i++ )
      {
        pc.printf(", %x", fsvr.get_byte(idx++ & 0x7ff) );
      }
      pc.printf( "\r\n" );
  }
  else
  {
      pc.printf("RAM data %x", idx & 0x7ff );
      for ( i = 0; i < 8; i++ )
      {
        pc.printf(", %x", fsvr.get_byte(idx++ & 0x7ff) );
      }
      pc.printf( "\r\n" );
  }
}

/****************************************************************************/ 
/*              write_addrs_to_file: 
* Description:  Writes net addresses to /webfs/net_adrs.txt
* Globals used: NET_ADRS_FILE 
* Parameters:   NONE
* Returns:      0 if OK, 1 if failed to open files
*/
/****************************************************************************/ 
int write_addrs_to_file()
{
  FILE *fp = fopen( NET_ADRS_FILE, "w" );
  if(!fp) {
    fprintf(stderr, "\r\nFile /webfs/net_adrs.txt could not be opened to write!\r\n");
    exit(1);
  }
  else                              // file opened OK, so save addresses
  {
    fprintf(fp, "IP address %u.%u.%u.%u \r\n", ip_ad[0],ip_ad[1],ip_ad[2],ip_ad[3]);
    fprintf(fp, "Address mask %u.%u.%u.%u \r\n", msk_ad[0],msk_ad[1],msk_ad[2],msk_ad[3]);
    fprintf(fp, "Gwy address %u.%u.%u.%u \r\n", gwy_ad[0],gwy_ad[1],gwy_ad[2],gwy_ad[3]);
    fprintf(fp, "DNS address %u.%u.%u.%u \r\n", dns_ad[0],dns_ad[1],dns_ad[2],dns_ad[3]);
    fprintf(fp, "%s\r\n", snmp_publiccommunity );   // v0.98: save community string
    fprintf(fp, "Trap host address %u.%u.%u.%u \r\n", th_ad[0],th_ad[1],th_ad[2],th_ad[3]);
    fprintf(fp, "%u \r\n", mask4_IO_registers );
    fclose(fp);
  } // file opened OK
  return 0;
}

/****************************************************************************/ 
/*              copy_trap_host_address: v0.99
* Description:  Copies src array  to th_ad
* Globals used: th_ad
*               net_addrs_chng set to UPDATED 
* Parameters:   IN: unsigned char * src - source array of 4 bytes
* Returns:      NONE
*/
/****************************************************************************/ 
void copy_trap_host_address( unsigned char * src )
{
unsigned char i;
  for ( i = 0; i < 4; i++ )
  {
    if ( src[ i ] != 0 )
    {
      {
        th_ad[ i ] = src[ i ];
      }
    }
  }
  net_addrs_chng = UPDATED;
}

/****************************************************************************/ 
/*              copy_net_addresses_from: v0.93
* Description:  Copies src array  to ip_ad, msk_ad, gwy_ad, dns_ad
* Globals used: ip_ad, msk_ad, gwy_ad, dns_ad
* Parameters:   IN: unsigned char * src - source array of 16 bytes
* Returns:      NONE
*/
/****************************************************************************/ 
void copy_net_addresses_from( unsigned char * src )
{
unsigned char i;
  for ( i = 0; i < 4; i++ )
  {
    if ( src[ i ] != 0 )
    {
      ip_ad[ i ] = src[ i ];
    }
  }
  for ( i = 4; i < 8; i++ )
  {
    if ( src[ i ] != 0 )
    {
      msk_ad[ i-4 ] = src[ i ];
    }
  }
  for ( i = 8; i < 12; i++ )
  {
    if ( src[ i ] != 0 )
    {
      gwy_ad[ i-8 ] = src[ i ];
    }
  }
  for ( i = 12; i < 16; i++ )
  {
    if ( src[ i ] != 0 )
    {
      dns_ad[ i-12 ] = src[ i ];
    }
  }
}

/****************************************************************************/ 
/*              parse4numbers: 
* Description:  Reads file fr and separates out max_num numbers delimited by
*               any non-digit characters
* Globals used:  
* Parameters:   IN: FILE *fr - file to read
*               OUT: unsigned char * numb - array of numbers read 
*               IN: unsigned char max_num - number of numbers to read
* Returns:      NONE
*/
/****************************************************************************/ 
void parse4numbers( FILE *fr, unsigned char * numb, unsigned char max_num )
{
int c, i, new_num;                           
char digit_found = false;
char number_completed = false;

    i = new_num = 0;
    while ( ((c = fgetc(fr)) != EOF) && (i < max_num) )
    {
      numb[ i ] = 0;
      if ( c < '0' || c > '9' )
      {                             // not a numeric character
        if ( digit_found )
        {
          number_completed = true;
          if ( i >= (max_num - 1) )
          {                         // max_num distinct numbers read
            break;
          }
        }
      }
      else                          // numeric character
      {
        digit_found = true;
        if ( number_completed )
        {                           // save number into array
          numb[ i++ ] = new_num & 0xff;
          new_num = 0;
        }
        new_num = 10*new_num + c - '0';
        number_completed = false;
      } // else numeric character
    } // end while
    if ( (number_completed || (c == EOF)) && (i < max_num) )
    {                               // 0.80: save last number read
      numb[ i++ ] = new_num & 0xff;
    }
}

/****************************************************************************/ 
/*              read_addrs_from_file: 
* Description:  Reads data from /webfs/net_adrs.txt
* Globals used: NET_ADRS_FILE 
* Parameters:   NONE
* Returns:      0 if OK, 1 if failed to open files
*/
/****************************************************************************/ 
int read_addrs_from_file()
{
int c, i;                           
unsigned char numbers[17];
char community[ MAX_COMMUNITY ];

  for ( i = 0; i < MAX_COMMUNITY; i++ )
  {
    community[ i ] = 0;
  }
  FILE *fr = fopen( NET_ADRS_FILE, "r" );
  if(!fr) {
    fprintf(stderr, "\r\nFile /webfs/net_adrs.txt could not be opened!\r\n");
    exit(1);
  }
  else                              // file opened OK
  {
    parse4numbers( fr, numbers, 16 );
    copy_net_addresses_from( numbers ); // v0.93: new subroutine
    /* v0.98: now read the community string from the file after reading end of line */
    while ( ((c = fgetc(fr)) != EOF) && ((c == ' ') || (c == '\r') || (c == '\n')) );
    i = 0;
    do
    {
      c = (char)c;
      if ( (c != ' ') && (c != '\r') && (c != '\n') && (i < MAX_COMMUNITY-1) )
      {
        community[ i++ ] = c;
      }
      else                              // string terminated
      {
        break;
      } 
    } while ( ((c = fgetc(fr)) != EOF) && (c != 0) );
    parse4numbers( fr, numbers, 4 );    // now read in Trap host address
    copy_trap_host_address( numbers );
    fscanf( fr, "%lu", &mask4_IO_registers );
    fclose(fr);
  } // file opened OK
  if ( strlen(community) > 3 )
  {                                     // string length 4 to 20 characters
    strcpy( snmp_publiccommunity, community );
    pc.printf("community: %s %d \r\n", snmp_publiccommunity, strlen(snmp_publiccommunity) );
  }
  else
  {
    pc.printf("community too short: %s %d \r\n", community, strlen(community) );
  }
  return 0;
}

/****************************************************************************/ 
/*              blinker: 
* Description:  Blinks led_link for 10 calls
* Globals used: led_link, blink_led,  
* Parameters:   NONE
* Returns:      NONE
*/
/****************************************************************************/ 
void blinker( void )
{
static unsigned char blink_cnt;

  if ( blink_led && blink_cnt++ < 10 )
  {
    led_link = !led_link;            
  }
  else
  {
    blink_cnt = 0;
    blink_led = false;
  }
}

/****************************************************************************/ 
/*              init_traps: 
* Description:  initialises trap flags and sends cold start trap when done
* Globals used: trap_init set true after cold start 
* Parameters:   NONE
* Returns:      NONE
*/
/****************************************************************************/ 
void init_traps( void )
{
  snmp_coldstart_trap();
  trap_init = true;
}

/****************************************************************************/ 
/*              check_all_traps: 
* Description:  test states for all trap events and sends appropriate trap
* Globals used: led2: set to ON if SPI not active
*               byte_changed, all_IO_updated, spi_active
* Parameters:   NONE
* Returns:      NONE
*/
/****************************************************************************/ 
void check_all_traps( void )
{
    if ( !led1 && (fsvr.get_byte( H_H_LINK_LSBYTE ) || fsvr.get_byte( H_H_LINK_MSBYTE )) )
    {                           // change of link state
        led1 = 1;               // flag: link is up when led1 on
        snmp_linkup_trap();
    }
    else if ( (fsvr.get_byte( H_H_LINK_LSBYTE ) == 0) && (fsvr.get_byte( H_H_LINK_MSBYTE ) == 0) && led1 )
    {                           // change of link state
        led1 = 0;
        snmp_linkdown_trap();
    }
    if ( byte_changed && all_IO_updated )
    {                           // any masked-in I/O register changed
        f3kIOchange_trap();
        all_IO_updated = byte_changed = false;
    }
    if ( spi_active && led2 )
    {
        led2 = 0;               // flag: SPI is up when led2 off
        snmp_SPIup_trap();
    }
    else if ( !spi_active && !led2 )
    {
        led2 = 1;               // flag: SPI is down when led2 on
        snmp_SPIdown_trap();
    }            
}


/****************************************************************************/ 
/****************************************************************************/ 
int main() {
int idx;

  new_cmd_idx = 0xaa;
  for ( idx = 0; idx < MAX_CMD; idx++ )
  {                                 // put meaningful non-valid data in cmd_buf
    cmd_buf[ idx ] = new_cmd_idx;
  }
  pc.printf("=================== Version %s ====================\r\n", snmp_fw_version);
//  pc.printf("Press any key to start...\r\n");
  led3 = 1;

  FSHandler::mount("/webfs", "/files"); //Mount /webfs path on /files web path
  FSHandler::mount("/webfs", "/"); //Mount /webfs path on web root path
  read_addrs_from_file();
        
/* Force fixed Ethernet address */
EthernetNetIf eth(
  IpAddr( ip_ad[0], ip_ad[1], ip_ad[2], ip_ad[3] ), //IP Address
  IpAddr( msk_ad[0], msk_ad[1], msk_ad[2], msk_ad[3] ), //Network Mask
  IpAddr( gwy_ad[0], gwy_ad[1], gwy_ad[2], gwy_ad[3] ), //Gateway
  IpAddr( dns_ad[0], dns_ad[1], dns_ad[2], dns_ad[3] )  //DNS Server
);

  Base::add_rpc_class<DigitalOut>();
#if 0
  Base::add_rpc_class<SPI_Server>();
#endif

//  pc.getc();                            // wait for keyboard
  led3 = 0;
  
  strcpy( (char *)syslocation, (const char *)syslocation_default );
  sysloc_len = strlen( (const char *)syslocation );
  snmp_set_syslocation( syslocation, &sysloc_len );
   
  pc.printf("\r\nSetting up...\r\n");
  EthernetErr ethErr = eth.setup();
  if( ethErr )
  {
    pc.printf("Error %d in Ethernet setup.\r\n", ethErr);
//    return -1;
  }
  else                  // Ethernet Setup OK
  {
    mip = eth.getIp();
    led3 = 1;
    led_speed = LPC_GPIO1->FIOPIN & (1<<26);       // GPIO1.26 = "LED_SPEED".    
    led_link  = LPC_GPIO1->FIOPIN & (1<<25);       // GPIO1.25 = "LED_LINK". 
    pc.printf("\r\nEthernet Setup OK\r\n");
  
    svr.addHandler<RPCHandler>("/rpc");
    svr.addHandler<FSHandler>("/files");
    svr.addHandler<FSHandler>("/"); //Default handler
    //Example : Access to mbed.htm : http://a.b.c.d/mbed.htm or http://a.b.c.d/files/mbed.htm
      
//    svr.bind(80); disable for SNMP development
    
//    snmp_init(); done in lwip/core/init.c
    
    pc.printf("Listening...\n\r");
    
    snmp_trap_dst_ip_set( ONLY_TRAP, (ip_addr_t *)&th_ad[0] );
    snmp_trap_dst_enable( ONLY_TRAP, ENABLED );
    snmp_coldstart_trap();              // to force ARP call
    
    led3 = led4 = 0;
    spi_slave.format(8,3);        // Setup:  byte data, high steady state clock, 2nd edge capture
  
    sys_tick.attach( &snmp_inc_sysuptime, 0.01 );
    spi_tick.attach( &chk_SPI_active, 60.0 );
    comms_tick.attach( &blinker, 0.07 ); // tick to blink activity LED
    cold_start_delay.attach( &init_traps, 30.0 ); // send cold_start trap after 30 seconds


    // Listen indefinitely to Ethernet and SPI
    while ( true ) 
    {
        Net::poll();                    // check for Ethernet requests
        
        if ( trap_init )
        {
            check_all_traps();
        }
        
        if( spi_slave.receive() ) {
            // Data available - needs chip select line to activate
            get_SPI_data( cmd_buf );
        } // end if( spi_slave.receive() )
        if ( net_addrs_set )
        {                               // v0.99: address changed from SNMP i/f
            write_addrs_to_file();
            net_addrs_set = false;
        }
    } // end while ( true )
  
  } // else Ethernet Setup OK
  return 0;
}
