/*
  DS1337.h - library for DS1337 rtc
*/

// ensure this library description is only included once
#ifndef DS1337_h
#define DS1337_h

// include types & constants of Wiring core API
#include "mbed.h"

typedef int8_t    byte;

// indices within the rtc_bcd[] buffer
#define DS1337_SEC  0
#define DS1337_MIN  1
#define DS1337_HR   2
#define DS1337_DOW  3
#define DS1337_DATE 4
#define DS1337_MTH  5
#define DS1337_YR   6

#define DS1337_BASE_YR      2000

#define DS1337_CTRL_ID      0xD0



// Define register bit masks
#define DS1337_CLOCKHALT    (1<<7)

#define DS1337_LO_BCD       0xf
#define DS1337_HI_BCD       0xf0

#define DS1337_HI_SEC       0x70
#define DS1337_HI_MIN       0x70
#define DS1337_HI_HR        0x30
#define DS1337_LO_DOW       0x07
#define DS1337_HI_DATE      0x30
#define DS1337_HI_MTH       0x30
#define DS1337_HI_YR        0xf0

#define DS1337_ARLM1        0x07
#define DS1337_ARLM1_LO_SEC 0xf
#define DS1337_ARLM1_HI_SEC 0x70
#define DS1337_ARLM1_LO_MIN 0x70
#define DS1337_ARLM1_HI_MIN 0xf

#define DS1337_SP           0x0E
#define DS1337_SP_EOSC      (1<<7)
#define DS1337_SP_RS2       (1<<4)
#define DS1337_SP_RS1       (1<<3)
#define DS1337_SP_INTCN     (1<<2)
#define DS1337_SP_A2IE      (1<<1)
#define DS1337_SP_A1IE      (1<<0)

#define DS1337_STATUS       0x0F
#define DS1337_STATUS_OSF   (1<<7)
#define DS1337_STATUS_A2F   (1<<1)
#define DS1337_STATUS_A1F   (1<<0)

/* Definitions for alarm repeat */
/* The private variable alarm_repeat holds the user's alarm repeat preference. However, the DS1337 encodes these in the topmost bit(s) of the 4 alarm registers. */
/* Splattering these bits across the alarm regs is handled in the writeAlarm() function. */
/* If DY/DT is set, the day field is interpreted as a DayOfWeek (1 ~ 7), else it is interpreted as a DayOfMonth.*/

/* user alarm_repeat bit mask:
       7   6   5    4       3      2       1     0
      [x   x   x   A1M4   DY/DT   A1M3   A1M2   A1M1]
*/

#define EVERY_SECOND       B00010111
#define EVERY_MINUTE       B00010110
#define EVERY_HOUR         B00010100
#define EVERY_DAY          B00010000
#define EVERY_WEEK         B00001000
#define EVERY_MONTH        B00000000


/* typedef struct {
  unsigned int year;
  unsigned char month;
  unsigned char day;
  unsigned char dayOfWeek;
  unsigned char hour;
  unsigned char minute;
  unsigned char second;
} TIME; */


// library interface description
class DS1337 {
    // user-accessible "public" interface
public:

    void readTime(void);
    unsigned char getSeconds();
    unsigned char getMinutes();
    unsigned char getHours();
    unsigned char getDays();
    unsigned char getDayOfWeek();
    unsigned char getMonths();
    unsigned int getYears();

    void setSeconds(unsigned char);
    void setMinutes(unsigned char);
    void setHours(unsigned char);
    void setDays(unsigned char);
    void setDayOfWeek(unsigned char);
    void setMonths(unsigned char);
    void setYears(unsigned int);
    
    void setTime();


    void    start(void);
    void    stop(void);
    unsigned char getRegister(unsigned char registerNumber);
    void    setRegister(unsigned char registerNumber, unsigned char registerValue);
    //void  unsetRegister(unsigned char registerNumber, unsigned char registerMask);

    // library-accessible "private" interface
private:

    byte    time_set;
    byte alarm_repeat;
    byte    rtc_bcd[7]; // used prior to read/set DS1337 registers;
    void    read(void);
    void    save(void);
    byte bcd2bin(byte);
    byte bin2bcd(byte);
};

//extern DS1337 RTC;

#endif
