/*
 * mbed library for NTC analog pin that translates an A:D input
 *     voltage to a temperature value.
 *
 *
 *     topology:  Vref -> SeriesRes -> A:D -> NTC -> GND
 *
*/

#ifndef        MBED_NTC
#define        MBED_NTC

    /** 
     * Private data structure for NTC data values.
     * 
    **/
    typedef struct {
        float vref;         /*!< Referense voltage*/
        float ad_resolution; /*!< A:D resolution*/
        int ntc_res;        /*!< NTC resistance value at 25C*/
        int ntc_tol;        /*!< NTC initial tolerance %*/
        int ntc_beta_0050;  /*!< NTC Beta, B0/50*/
        int ntc_beta_2550;  /*!< NTC Beta, B25/50*/
        int ntc_beta_2580;  /*!< NTC Beta, B25/80*/
        int ntc_beta_2585;  /*!< NTC Beta, B25/85*/
        int ntc_beta_25100; /*!< NTC Beta, B25/100*/
        int ntc_beta_other; /*!< NTC Beta, other value vs temp*/
        int ntc_beta_tol;   /*!< NTC Beta tolerance %*/
        int sres_res;       /*!< Series resistor value*/
        int sres_tol;       /*!< Series resistor tolerance %*/
        int sres_ppm;       /*!< Series resistor tempco in ppm*/
    } NTC_TypeDef;

    /** 
     * Default values for data structure above
     * 
    **/
    const NTC_TypeDef ntc_std_paramtr = {
        // Vref
        3.0f,               // Vref
        16384.0f,           // A:D resolution
        // muRata NCP15XH103J03RC
        20000,              // NTC resistance
        3,                  // NTC initial tolerance
        0,                  // NTC B0/50, none
        3300,               // NTC B25/50
        3322,               // NTC B25/80
        3333,               // NTC B25/85
        3344,               // NTC B25/100
        0,                  // NTC other Beta value
        2,                  // NTC beta tolerance
        // 3.32k 1% 100ppm
        4750,               // Series resistor value
        1,                  // Series resistor tolerance
        200                 // Series Resistor tempco ppm
    };

    /** NTC temperature A:D conversion
     *
     * @code
     *  #include "mbed.h"
     *  #include "ntc.h" 
     *
     *  #define NTC_VREF        3.3f
     *  #define NTC_AD_RESOL    65536.0f
     *
     *  // Connections:
     *
     *  //      3.3V (or other Vref)
     *  //     --+--
     *  //       |
     *  //      -+-
     *  //     |   |
     *  //     |   |
     *  //     |   | Series
     *  //     |   | Resistor
     *  //     |   |
     *  //      -+-
     *  //       |
     *  //       +---> To A:D
     *  //       |
     *  //      -+-
     *  //     |   |
     *  //     |   |
     *  //     |   | NTC
     *  //     |   | 
     *  //     |   |
     *  //      -+-
     *  //       |
     *  //     --+--
     *  //      ---  Ground
     *  //       -
     *
     *  
     *  const NTC_TypeDef ntc_my_paramtr = {
     *    // Vref
     *    NTC_VREF,           // Vref
     *    NTC_AD_RESOL,       // A:D 16-bit resolution
     *    // muRata NCP15XH103J03RC
     *    10000,              // NTC resistance
     *    5,                  // NTC initial tolerance
     *    0,                  // NTC B0/50 (none)
     *    3380,               // NTC B25/50
     *    3428,               // NTC B25/80
     *    3434,               // NTC B25/85
     *    3355,               // NTC B25/100
     *    0,                  // NTC other beta (none)
     *    1,                  // NTC beta tolerance
     *    // 3.32k 1% 100ppm
     *    3320,               // Series resistor value
     *    1,                  // Series resistor tolerance
     *    100                 // Series Resistor tempco ppm
     *  };    
     *
     *  NTC ntc(A1, &ntc_my_paramtr);                   //initialize NTC temperature A:D
     * 
     *  main() {
     *      printf("\r\n\r\n-------------------------------------------\r\n");
     *      printf("NTC Res: %5d   B0/50: %4d   B25/50: %4d   B25/80: %4d   B25/85: %4d   B25/100: %4d   B_OTHER: %4d   SeriesR: %d\r\n", 
     *                  ntc.get_ntc_res(), ntc.get_ntc_beta(NTC::B0_50), ntc.get_ntc_beta(NTC::B25_50), ntc.get_ntc_beta(NTC::B25_80), ntc.get_ntc_beta(NTC::B25_85), 
     *                  ntc.get_ntc_beta(NTC::B25_100), ntc.get_ntc_beta(NTC::B_OTHER),  ntc.get_series_res());
     *      uint16_t ad = ntc.read_ad_reg();
     *      printf("NTC A:D Val: %5d   Volt A:D: %.6f   NTC-R_now: %7.1f    Temp: %+.2f\r\n", ad, 
     *              NTC_VREF / NTC_AD_RESOL * (float)ad, ntc.get_ntc_res_viaAD(ad), ntc.get_ntc_temp(NTC::B25_85, ad));
     *
     *      while(1) {
     *          printf("Temp: %+.2f\r\n", ntc.get_ntc_temp(NTC::B25_85));
     *          wait(2.0);
     *      }
     *          
     *  }         
     * @endcode
     */
 
class NTC
{
public:
    /** 
     * NTC Beta curves to choose from for Temperature calculations
     * 
    **/
    enum ntcBetaCurve {
        B0_50,   /*!<  use B0/50 curve */
        B25_50,  /*!<  use B25/50 curve */
        B25_80,  /*!<  use B25/80 curve */
        B25_85,  /*!<  use B25/85 curve */
        B25_100, /*!<  use B25/100 curve */
        B_OTHER, /*!<  use other curve */
    };

    /** Configure gpio pin
      * @param A:D I/O pin
      * @param parameters for NTC (NTC_TypeDef)
      */
    NTC(PinName p_ana, const NTC_TypeDef *ntc_parameter);
    
    /** Configure gpio pin
      * @param A:D I/O pin
      * uses default parameters
      */
    NTC(PinName p_ana);
    
    /** Read A:D register value
      * @param none
      * @return A:D value from 0-65535
      */
    uint16_t read_ad_reg();
    
    /** Get the NTC resistance value
      * @param none
      * @return resistance of NTC at 25C
      */
    int get_ntc_res();
    
    /** Get the NTC beta value
      * @param ntcBetaCurve selection
      * @param B25_85 default value
      * @return beta curve value
      */
    int get_ntc_beta(int beta = B25_85);
       
    /** Get the resistance value for the series resistor
      * @param none
      * @return resistance of series resistor
      */
    int get_series_res();
    
    /** calculate NTC resistance based on A:D reading
      * @param none
      * @param din optional read from read_ad_reg().
      * @param din -> if missing or 0, read_ad_reg() value is used for calculation
      * @param din -> if != 0, then value of din is used for calculation
      * @return resistance of NTC
      */
    float get_ntc_res_viaAD(uint16_t din = 0);
      
    /** calculate NTC temperaure
      * @param curve Beta curve to use. See ntcBetaCurve {};
      * @param din optional read from read_ad_reg()
      * @param din -> if missing or 0, read_ad_reg() value is used for calculation
      * @param din -> if != 0, then value of din is used for calculation
      * @return temperature of NTC
      * @return -100 if NTC Beta selected = 0
      */
    float get_ntc_temp(int curve, uint16_t din = 0);
    
protected:
    AnalogIn  _ana;

private:
    NTC_TypeDef ntc_set_data;
    uint16_t rawdata;

};

#endif  //  MBED_NTC