#ifndef I2CDEVICES_H
#define I2CDEVICES_H

#include "DevI2C.h"

//if the use of low-level I2C routines is required, uncomment out the next two lines 
//#define I2CLOWLEVEL     1   
//#warning "using low level I2C routines"

/** Software routines that show the I2C devices connected to the I2C bus. The
 *  routine also sets flags that indicate certain I2C devices have been 
 *  detected.  Those flags are within a "typedef" structure.
 *
 * NOTE: Requires the external definition of the user console: "RawSerial pc;"
 *
 * @code
 * #include "mbed.h"
 * #include "i2cDevices.h"
 *
 * RawSerial pc(USBTX, USBRX);
 * DigitalOut led1(LED1);
 *
 * #define SDA         PTE0     //i2c bus for KL25Z
 * #define SCL         PTE1
 *
 * I2C i2c1(SDA, SCL);
 * i2c_discovery i2c_d(SDA, SCL, "i2c_d");
 * i2c_discovery::i2cDeviceFlags i2c_flags = {};
 *
 * int main() {
 *     pc.baud(230400);
 *     i2c1.frequency(400000);
 *     pc.print("\r\n\r\nbegin....\r\n");
 *     i2c_d.find_i2c(i2c_flags);
 *     if(i2c_flags.PCA9955_0 == true) {
 *         pc.printf("Device detected!\r\n");
 *     } else {
 *         pc.printf("Device not found!!!\r\n");
 *     }
 *     while(1) {
 *         led1 = !led1;
 *         wait(1.0);
 *     }
 * }
 *
 * @endcode
 */

//used to define terminal text output color
#define COLOR_TERM
#ifdef COLOR_TERM
    #define DOBLUE      "\033[0;34;2m"
    #define DOMAGENTA   "\033[0;35;2m"
    #define DONONE      "\033[0m"
#else
    #define DOBLUE      ""
    #define DOMAGENTA   ""
    #define DONONE      ""
#endif

//list of i2c device possibilities
#define I2C_0x02    "PCA9955-GGG"
#define I2C_0x06    "SHT1x"
#define I2C_0x0C    "AK8975"

#define I2C_0x1C    "MAG3110"

#define I2C_0x20    "VEML6040, VEML6075"

#define I2C_0x38    "MMA8451-0, FXOS8700CQ-2"
#define I2C_0x3A    "MMA8451-1, FXOS8700CQ-1"
#define I2C_0x3C    "FXOS8700CQ-0, LIS3MDL"
#define I2C_0x3E    "FXOS8700CQ-3"

#define I2C_0x42    "MCP23017-b"
#define I2C_0x44    "MCP23017-c"
#define I2C_0x46    "MCP23017-d"

#define I2C_0x50    "MAX5387-ggg"
#define I2C_0x52    "MAX5387-ggv, VL53L0X"
#define I2C_0x54    "MAX5387-gvg"
#define I2C_0x56    "MAX5387-gvv"
#define I2C_0x58    "MAX5387-vgg"
#define I2C_0x5A    "MAX5387-vgv"
#define I2C_0x5C    "MAX5387-vvg"
#define I2C_0x5E    "MAX5387-vvv"

#define I2C_0x70    "VEML6070-ld/c"
#define I2C_0x72    "VEML6070-hd"

#define I2C_0x80    "Si7005, Si7021, HTU21D, HDC1080, INA2xx/MAX7300-gg"
#define I2C_0x82    "INA2xx/MAX7300-gv"
#define I2C_0x84    "INA2xx/MAX7300-gd"
#define I2C_0x86    "INA2xx/MAX7300-gc"
#define I2C_0x88    "INA2xx/MAX7300-vg"
#define I2C_0x8A    "INA2xx/MAX7300-vv"
#define I2C_0x8C    "INA2xx/MAX7300-vd"
#define I2C_0x8E    "INA2xx/MAX7300-vc"

#define I2C_0x90    "AD7416-a, LM75, INA2xx/MAX7300-dg"
#define I2C_0x92    "MAX9768, INA2xx/MAX7300-dv"
#define I2C_0x94    "INA2xx/MAX7300-dd"
#define I2C_0x96    "INA2xx/MAX7300-dc"
#define I2C_0x98    "INA2xx/MAX7300-cg"
#define I2C_0x9A    "INA2xx/MAX7300-cv"
#define I2C_0x9C    "INA2xx/MAX7300-cd"
#define I2C_0x9E    "INA2xx/MAX7300-cc"

#define I2C_0xA0    "24FC512-000, 24AA02-b0"
#define I2C_0xA2    "24AA02-b1"
#define I2C_0xA4    "24AA02-b2"
#define I2C_0xA6    "24AA02-b3"
#define I2C_0xA8    "24AA02-b4"
#define I2C_0xAA    "MMA8491, 24AA02-b5, BQ27xx0, BQ34Z100-G1"
#define I2C_0xAC    "24AA02-b6, M24SR64-Y"
#define I2C_0xAE    "24AA02-b7"

#define I2C_0xB4    "CCS811-0"
#define I2C_0xB6    "CCS811-1"
#define I2C_0xBA    "LPS22HB"
#define I2C_0xBE    "HTS221"

#define I2C_0xC0    "MPL3115A2, MLX90620-ram, PCA9955-0"
#define I2C_0xC2    "SCD30"

#define I2C_0xD0    "M41T11, DS1307, DS3232M, MCU9150-0"
#define I2C_0xD2    "MCU9150-1"
#define I2C_0xD4    "LSM6DSL"
#define I2C_0xDE    "MCP7941x-rtc"

#define I2C_0xE0    "PCA9955-AllCall"
#define I2C_0xEC    "MS5611-cs1, BME280-cs0, PCA9955-SubCall"
#define I2C_0xEE    "MS5611-cs0, BME280-cs1, BMP-05"

/** Create an i2c controller class
 *
 * @param i2c class class
 *
 */
class i2c_discovery
{
    public:
    /** Structure which is used to exchange indicate certain
     *  devices of interest is detected or not by this routine.
     */    
    typedef struct {
        bool rtcDS_ok;          /*!< i2c address 0xd0 */
        bool ms5611_ok;         /*!< i2c address 0xee */
        bool bme280_cs0_ok;     /*!< i2c address 0xec */
        bool bme280_cs1_ok;     /*!< i2c address 0xee */
        bool mma8451cs1_ok;     /*!< i2c address 0x3a */
        bool HTU21D_ok;         /*!< i2c address 0x80 */
        bool INA219_MAX7300_gg_ok; /*!< i2c address 0x80 */
        bool INA230_MAX7300_gg_ok; /*!< i2c address 0x80 */
        bool PCA9955_GGG_ok;    /*!< i2c address 0xc0 */
        bool VEML60xx_ok;       /*!< i2c address 0x20 */
        bool VEML6070_ok;       /*!< i2c address 0x70 */
        bool eeprom000_ok;      /*!< i2c address 0xa0 */
        bool mlx90620_ok;       /*!< i2c address 0xc0 */
        bool SCD30_ok;          /*!< i2c address 0xc2 */
        bool bq27210_ok;        /*!< i2c address 0xaa */
        bool ccs811_0_ok;       /*!< i2c address 0xb4 */
        bool ccs811_1_ok;       /*!< i2c address 0xb6 */
        bool mcp7941x_ok;       /*!< i2c address 0xde */
    } i2cDeviceFlags;

    /**
     * Constructor, uses default I2C clock speed
     * - Fixed at I2C address 0x80
     * - I2C speed set to 400KHz
     *
     * @param sda - mbed I2C interface pin
     * @param scl - mbed I2C interface pin
     * @param name - name of i2c_discovery
     */
    i2c_discovery(PinName sda, PinName scl, const char* name);
    
    /**
     * Clear I2C device Flags
     * 
     * @param i2cDeviceFlags pointer
     *
     * @return --none--
    */    
    void clearFlags(i2cDeviceFlags& i2cFlags);
    
    /**
     * Go and search for I2C devices
     * 
     * @param i2cDeviceFlags pointer
     *
     * @return count - number of i2c devices detected
    */     
    int find_i2c(i2cDeviceFlags& i2cFlags);
    
    private:

    int showI2C(i2cDeviceFlags& i2cFlags, char addr);
    int dispI2C(i2cDeviceFlags& i2cFlags);
    
    protected:
    
    DevI2C _i2c;

};

#endif
