#include "mbed.h"
#include "ds3232m.h"

#define RTCI2CADDRESS           0xd0        //i2c address of the DS3232M
#define DS3232SECONDS           0x00        //seconds register
#define DS3232DAYOFWEEK         0x03        //day of the week register
#define DS3232ALM1BOTTOM        0x07        //alarm 1 registers (not used)
#define DS3232ALM2BOTTOM        0x0b        //alarm 2 registers (not used)
#define DS3232CTLREG            0x0e        //control register
#define DS3232STATREG           0x0f        //status register
#define DS3232TEMPERATURE       0x11        //temperature registers
#define DS3232TESTREG           0x13        //test register (do not use)
#define DS3232RAMBOTTOM         0x14        //where user RAM starts
#define DS3232RAMCRC16          0xfc        //crc16 registers (top 2 bytes of user RAM)
#define DS3232RAMTOP            0xfe        //where user RAM ends. Note: bug in F746 only allows for a I2C xfer of 255 max!!!!

//control register
#define DS3232_EOSCN_BIT        0x80        //turn on/off oscillator in batt backup mode, 0 = on
#define DS3232_BBSQW_BIT        0x40        //battery backed 1Hz enable, 1 = enable
#define DS3232_CONV_BIT         0x20        //start temperature conversion, 1 = start
#define DS3232_INTCN_BIT        0x04        //interrupt control, 1 = interrupt mode, 0 = 1Hz mode

//status register
#define DS3232_OSF_BIT          0x80        //oscillator stopped, 1 = stopped
#define DS3232_BB32KHZ_BIT      0x40        //battery backed 32KHz enable, 1 = enable(writable bit)
#define DS3232_EN32KHZ_BIT      0x08        //32KHz enable, 1 = enable(writable bit)
#define DS3232_BUSY_BIT         0x04        //busy doing a temperature A:D conversion, 1 = busy

char RtcCtlReg = 0;                         //mirror of control register
char RtcStatReg = 0;                        //mirror of status register

//char RTCbuffer[256];                        //buffer may contain up to 256 bytes, depending on command used

//--------------------------------------------------------------------------------------------------------------------------------------//
// constructor with fixed frequency 

ds3232m::ds3232m(PinName sda, PinName scl) {
    // Create a new I2C object
    _i2c_ = new I2C(sda, scl);
    // Set the frequency to standard 400kHz
    _i2c_->frequency(400000);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// constructor with I2C frequency selection

ds3232m::ds3232m(PinName sda, PinName scl, int i2cFrequency) {
    _i2c_ = new I2C(sda, scl);
    _i2c_->frequency(i2cFrequency);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// deconstructor

ds3232m::~ds3232m() {
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// set time up into the DS3232M

void ds3232m::setTime(Time_rtc& dsSTR) {
    dsSTR.RTCbuffer[0] = DS3232SECONDS;
    dsSTR.RTCbuffer[1] = DecToBCD(dsSTR.sec);
    dsSTR.RTCbuffer[2] = DecToBCD(dsSTR.min);
    dsSTR.RTCbuffer[3] = DecToBCD(dsSTR.hour);
    dsSTR.RTCbuffer[4] = dsSTR.wday;
    dsSTR.RTCbuffer[5] = DecToBCD(dsSTR.date);
    dsSTR.RTCbuffer[6] = DecToBCD(dsSTR.mon);
    dsSTR.RTCbuffer[7] = DecToBCD(dsSTR.year - 2000);
    
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 8, false);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// get time from the DS3232M and stick it into the mbed's RTC

void ds3232m::getTime(Time_rtc& dsSTR) {
    dsSTR.RTCbuffer[0] = DS3232SECONDS;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    _i2c_->read(RTCI2CADDRESS, dsSTR.RTCbuffer, 7, false);
    
    dsSTR.sec = BCDToDec(dsSTR.RTCbuffer[0]);
    dsSTR.min = BCDToDec(dsSTR.RTCbuffer[1]);
    dsSTR.hour = ds3232m::BCDToDec(dsSTR.RTCbuffer[2]);
    dsSTR.wday = dsSTR.RTCbuffer[3]; 
    dsSTR.date = ds3232m::BCDToDec(dsSTR.RTCbuffer[4]);
    dsSTR.mon = ds3232m::BCDToDec(dsSTR.RTCbuffer[5]);
    dsSTR.year = ds3232m::BCDToDec(dsSTR.RTCbuffer[6]) + 2000;
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// retrieve the control and status registers

void ds3232m::getControlStatusRegs(Time_rtc& dsSTR) {
    dsSTR.RTCbuffer[0] = DS3232CTLREG;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    _i2c_->read(RTCI2CADDRESS, dsSTR.RTCbuffer, 2, false);
    RtcCtlReg = dsSTR.RTCbuffer[0];
    RtcStatReg = dsSTR.RTCbuffer[1];
}
    
//--------------------------------------------------------------------------------------------------------------------------------------//
// enables/disables the main oscillator during battery backup mode. true = enable during battery backup mode

void ds3232m::enableBattClock(Time_rtc& dsSTR, bool batt) {
    getControlStatusRegs(dsSTR);
    if(batt == true) {
        RtcCtlReg &= ~DS3232_EOSCN_BIT;
    } else {
        RtcCtlReg |= DS3232_EOSCN_BIT;
    }

    dsSTR.RTCbuffer[0] = DS3232CTLREG;
    dsSTR.RTCbuffer[1] = RtcCtlReg;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 2, false);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// check to see if DS3232M's temperature conversion cycle is bust or not
// only 1 per second max

bool ds3232m::checkTempBusy(Time_rtc& dsSTR) {
    getControlStatusRegs(dsSTR);
    if(RtcCtlReg & DS3232_CONV_BIT) return true;     //is already busy
    if(RtcStatReg & DS3232_BUSY_BIT) return true;    //is already busy
    return false;
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// start a temperature conversion cycle
// only 1 per second max

bool ds3232m::startTempCycle(Time_rtc& dsSTR) {
    getControlStatusRegs(dsSTR);
    if((checkTempBusy(dsSTR)) == true) return false;     //is already busy
    RtcCtlReg |= DS3232_CONV_BIT;

    dsSTR.RTCbuffer[0] = DS3232CTLREG;
    dsSTR.RTCbuffer[1] = RtcCtlReg;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 2, false);
    return true;
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// get temperature
// if returned value = 255.0, then temperature conversion was still busy

float ds3232m::getTemperature(Time_rtc& dsSTR) {
    getControlStatusRegs(dsSTR);
    if((checkTempBusy(dsSTR)) == false) return(255.0);     //is already busy

    dsSTR.RTCbuffer[0] = DS3232TEMPERATURE;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    _i2c_->read(RTCI2CADDRESS, dsSTR.RTCbuffer, 2, false);
    int8_t temp3232a = dsSTR.RTCbuffer[0];
    uint8_t temp3232b = dsSTR.RTCbuffer[1];

    float temp3232 = (float)temp3232a;
    //for positive temp
    if((temp3232b == 0x40) && (temp3232a >= 0)) temp3232 += 0.25;
    if (temp3232b == 0x80) temp3232 += 0.5;
    if((temp3232b == 0xc0) && (temp3232a >= 0)) temp3232 += 0.75;
    //for negative temp
    if((temp3232b == 0x40) && (temp3232a < 0)) temp3232 += 0.75;
    if((temp3232b == 0xc0) && (temp3232a < 0)) temp3232 += 0.25;
    return (temp3232);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// get seconds register

uint8_t ds3232m::getSeconds(Time_rtc& dsSTR) {
    dsSTR.RTCbuffer[0] = DS3232SECONDS;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    _i2c_->read(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, false);
    int seconds3232 = dsSTR.RTCbuffer[0];
    int xbcd = BCDToDec(seconds3232);
    return(xbcd);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// get day of the week register (1 = Monday... 7 = Sunday)

uint8_t ds3232m::getDayOfWeek(Time_rtc& dsSTR) {
    dsSTR.RTCbuffer[0] = DS3232DAYOFWEEK;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    _i2c_->read(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, false);
    int dow3232 = dsSTR.RTCbuffer[0];
    return(BCDToDec(dow3232));
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// set day of the week register (1 = Monday... 7 = Sunday)

void ds3232m::putDayOfWeek(Time_rtc& dsSTR, uint8_t dow3232) {
    dow3232 = dow3232 & 7;
    dsSTR.RTCbuffer[0] = DS3232DAYOFWEEK;
    dsSTR.RTCbuffer[1] = DecToBCD(dow3232);
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 2, false);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// clears all user memory inside the DS3232M.  Top 2 locations - reserved for CRC, also set to 00 since CRC value is 0x00 0x00

void ds3232m::clearRAM(Time_rtc& dsSTR) {
    for(int i = DS3232SECONDS; i <= DS3232RAMTOP; i++) dsSTR.RTCbuffer[i] = 0;
    dsSTR.RTCbuffer[0] = DS3232RAMBOTTOM;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, (DS3232RAMTOP - DS3232RAMBOTTOM + 2), false);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// retrieves user data from DS3232M RAM

uint8_t ds3232m::getUserRAM(char *buffer, Time_rtc& dsSTR, int offset, int length) {
    if((offset + length) > DS3232RAMCRC16) return(DS3232_OVERFLOWERROR);
    if(offset < DS3232RAMBOTTOM) return(DS3232_OFFSETERROR);
    if(length == 0) return(DS3232_LENZEROERROR);
    dsSTR.RTCbuffer[0] = offset;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    _i2c_->read(RTCI2CADDRESS, buffer, length, false);
    if(LoadRTCRam(dsSTR) == false) return(DS3232_CRCERROR);
    return(DS3232_NOERROR);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// puts user data into DS3232M RAM. CRC16 value added to the top 2 RAM location

uint8_t ds3232m::putUserRAM(char *buffer, Time_rtc& dsSTR, int offset, int length) {
    if((offset + length) > DS3232RAMCRC16) return(DS3232_OVERFLOWERROR);
    if(offset < DS3232RAMBOTTOM) return(DS3232_OFFSETERROR);
    if(length == 0) return(DS3232_LENZEROERROR);
    for(int i = 0; i < 32; i++) dsSTR.RTCbuffer[i] = NULL;
    dsSTR.RTCbuffer[0] = offset;
    dsSTR.RTCbuffer[1] = NULL;
    strcat(dsSTR.RTCbuffer, buffer);
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, length + 1, false);
    addCRC16(dsSTR);
    return(DS3232_NOERROR);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// enable/disable 32KHz output with run on battery mode option

void ds3232m::set32KhzOutput(Time_rtc& dsSTR, bool ena, bool batt) {
    getControlStatusRegs(dsSTR);
    if(ena == true) {
        RtcStatReg |= DS3232_EN32KHZ_BIT;
        if(batt == true) {
            RtcStatReg |= DS3232_BB32KHZ_BIT;
        } else {
            RtcStatReg &= ~DS3232_BB32KHZ_BIT;
        }
    } else {
        RtcStatReg &= ~DS3232_EN32KHZ_BIT;
        RtcStatReg &= ~DS3232_BB32KHZ_BIT;
    }

    dsSTR.RTCbuffer[0] = DS3232STATREG;
    dsSTR.RTCbuffer[1] = RtcStatReg;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 2, false);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// enable/disable 1Hz output with run on battery mode option

void ds3232m::set1hzOutput(Time_rtc& dsSTR, bool ena, bool batt) {
    getControlStatusRegs(dsSTR);
    if(ena == true) {
        RtcCtlReg &= ~DS3232_INTCN_BIT;
        if(batt == true) {
            RtcCtlReg |= DS3232_BBSQW_BIT;
        } else {
            RtcCtlReg &= ~DS3232_BBSQW_BIT;
        }
    } else {
        RtcCtlReg |= DS3232_INTCN_BIT;
        RtcCtlReg &= ~DS3232_BBSQW_BIT;
    }

    dsSTR.RTCbuffer[0] = DS3232CTLREG;
    dsSTR.RTCbuffer[1] = RtcCtlReg;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 2, false);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// CRC table and routine taken from Emilie Laverge's CRC16 library
//--------------------------------------------------------------------------------------------------------------------------------------//
// lookup table for calculating crc16

uint16_t crc16table[256] = { 
        0x0000, 0x8005, 0x800F, 0x000A, 0x801B, 0x001E, 0x0014, 0x8011,
        0x8033, 0x0036, 0x003C, 0x8039, 0x0028, 0x802D, 0x8027, 0x0022,
        0x8063, 0x0066, 0x006C, 0x8069, 0x0078, 0x807D, 0x8077, 0x0072,
        0x0050, 0x8055, 0x805F, 0x005A, 0x804B, 0x004E, 0x0044, 0x8041,
        0x80C3, 0x00C6, 0x00CC, 0x80C9, 0x00D8, 0x80DD, 0x80D7, 0x00D2,
        0x00F0, 0x80F5, 0x80FF, 0x00FA, 0x80EB, 0x00EE, 0x00E4, 0x80E1,
        0x00A0, 0x80A5, 0x80AF, 0x00AA, 0x80BB, 0x00BE, 0x00B4, 0x80B1,
        0x8093, 0x0096, 0x009C, 0x8099, 0x0088, 0x808D, 0x8087, 0x0082,
        0x8183, 0x0186, 0x018C, 0x8189, 0x0198, 0x819D, 0x8197, 0x0192,
        0x01B0, 0x81B5, 0x81BF, 0x01BA, 0x81AB, 0x01AE, 0x01A4, 0x81A1,
        0x01E0, 0x81E5, 0x81EF, 0x01EA, 0x81FB, 0x01FE, 0x01F4, 0x81F1,
        0x81D3, 0x01D6, 0x01DC, 0x81D9, 0x01C8, 0x81CD, 0x81C7, 0x01C2,
        0x0140, 0x8145, 0x814F, 0x014A, 0x815B, 0x015E, 0x0154, 0x8151,
        0x8173, 0x0176, 0x017C, 0x8179, 0x0168, 0x816D, 0x8167, 0x0162,
        0x8123, 0x0126, 0x012C, 0x8129, 0x0138, 0x813D, 0x8137, 0x0132,
        0x0110, 0x8115, 0x811F, 0x011A, 0x810B, 0x010E, 0x0104, 0x8101, 
        0x8303, 0x0306, 0x030C, 0x8309, 0x0318, 0x831D, 0x8317, 0x0312,
        0x0330, 0x8335, 0x833F, 0x033A, 0x832B, 0x032E, 0x0324, 0x8321,
        0x0360, 0x8365, 0x836F, 0x036A, 0x837B, 0x037E, 0x0374, 0x8371,
        0x8353, 0x0356, 0x035C, 0x8359, 0x0348, 0x834D, 0x8347, 0x0342,
        0x03C0, 0x83C5, 0x83CF, 0x03CA, 0x83DB, 0x03DE, 0x03D4, 0x83D1,
        0x83F3, 0x03F6, 0x03FC, 0x83F9, 0x03E8, 0x83ED, 0x83E7, 0x03E2,
        0x83A3, 0x03A6, 0x03AC, 0x83A9, 0x03B8, 0x83BD, 0x83B7, 0x03B2,
        0x0390, 0x8395, 0x839F, 0x039A, 0x838B, 0x038E, 0x0384, 0x8381,
        0x0280, 0x8285, 0x828F, 0x028A, 0x829B, 0x029E, 0x0294, 0x8291,
        0x82B3, 0x02B6, 0x02BC, 0x82B9, 0x02A8, 0x82AD, 0x82A7, 0x02A2,
        0x82E3, 0x02E6, 0x02EC, 0x82E9, 0x02F8, 0x82FD, 0x82F7, 0x02F2,
        0x02D0, 0x82D5, 0x82DF, 0x02DA, 0x82CB, 0x02CE, 0x02C4, 0x82C1,
        0x8243, 0x0246, 0x024C, 0x8249, 0x0258, 0x825D, 0x8257, 0x0252,
        0x0270, 0x8275, 0x827F, 0x027A, 0x826B, 0x026E, 0x0264, 0x8261,
        0x0220, 0x8225, 0x822F, 0x022A, 0x823B, 0x023E, 0x0234, 0x8231,
        0x8213, 0x0216, 0x021C, 0x8219, 0x0208, 0x820D, 0x8207, 0x0202
};

//--------------------------------------------------------------------------------------------------------------------------------------//
// calculate a crc value for the DS3232M user ram area.  CRC value calculated from address 0x14 - 0xfd

uint16_t ds3232m::calculateCRC16(char input[], Time_rtc& dsSTR, int offset, int length) {
    uint16_t CRC16s = 0;
    input+= offset;
    for(int i = offset; i < (length + offset); i++) {
        uint16_t tableValue = crc16table[((CRC16s >> 8) ^ *(char *)input++) & 0x00ff];
        CRC16s = (CRC16s << 8) ^ tableValue;
    }
    dsSTR.c_crc = CRC16s;
    return CRC16s;
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// This function gets the entire contents of the RTC. Returns flase if a CRC error occured

bool ds3232m::LoadRTCRam(Time_rtc& dsSTR) {
    dsSTR.RTCbuffer[0] = DS3232SECONDS;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    _i2c_->read(RTCI2CADDRESS, dsSTR.RTCbuffer, DS3232RAMTOP, false); //BUG in F746 only allows for xfer of 255 bytes max!!!

    dsSTR.s_crc = (dsSTR.RTCbuffer[DS3232RAMCRC16] << 8) | dsSTR.RTCbuffer[DS3232RAMCRC16 + 1];
    calculateCRC16(dsSTR.RTCbuffer, dsSTR, DS3232RAMBOTTOM, (DS3232RAMCRC16 - DS3232RAMBOTTOM));
    
    ////reload the time registers
    //dsSTR.RTCbuffer[0] = DS3232SECONDS;
    //_i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 1, true);
    //_i2c_->read(RTCI2CADDRESS, dsSTR.RTCbuffer, 7, false);

    if(dsSTR.c_crc != dsSTR.s_crc) return false;
    return true;
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// This function calculates and saves CRC data to the end of RTC's RAM.  
// CRC calculated address range is 0x14 - 0xfd. CRC data is placed in 0xfe and 0xff

void ds3232m::addCRC16(Time_rtc& dsSTR) {
    LoadRTCRam(dsSTR);
    dsSTR.c_crc = calculateCRC16(dsSTR.RTCbuffer, dsSTR, DS3232RAMBOTTOM, (DS3232RAMCRC16 - DS3232RAMBOTTOM)); //0x14 is offset from 0 in buffer, 230 is length //
    dsSTR.RTCbuffer[0] = DS3232RAMCRC16;
    dsSTR.RTCbuffer[1] = dsSTR.c_crc >> 8;
    dsSTR.RTCbuffer[2] = dsSTR.c_crc & 0xff;
    _i2c_->write(RTCI2CADDRESS, dsSTR.RTCbuffer, 3, false);
}

