// Borch BME280 Barometer, Humidity and Temperature sensor IC

#include "bme280.h"

//--------------------------------------------------------------------------------------------------------------------------------------//
// Constructor, to allow for user to select i2c address based on CSB pin

bme280::bme280(PinName sda, PinName scl, CSBpolarity CSBpin)  : _i2c(sda, scl) {
        _i2c.frequency(400000);
        i2cWAddr = BME280_WADDR;
        i2cRAddr = BME280_RADDR;
        if(CSBpin == CSBpin_1) {
            i2cWAddr++;
            i2cWAddr++;
            i2cRAddr++;
            i2cRAddr++;
        }
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// deconstructor

bme280::~bme280() {
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// I2C start.  Returns "ack" from slave

int bme280::_i2c_start(uint8_t i2c_addr) {
    int ack;
    _i2c.start();
    ack = _i2c_write(i2c_addr);
    return(ack);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// I2C stop

void bme280::_i2c_stop() {
    _i2c.stop();
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// I2C write a byte.  Returns "ack" from slave

uint8_t bme280::_i2c_write(uint8_t data) {
    int ack = _i2c.write(data);
    return(ack);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// I2C read byte and sending ACK. Returns data byte.

uint8_t bme280::_i2c_readACK() {
    uint8_t rdata = _i2c.read(1);
    return(rdata);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// I2C read byte and sending NACK. Returns data byte.

uint8_t bme280::_i2c_readNACK() {
    uint8_t rdata = _i2c.read(0);
    return(rdata);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Get BME280 ID register

uint8_t bme280::getBmeID() {
#if defined  BMEi2cLOWLEVEL
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_CHIP_ID_REG);
    _i2c_start(i2cRAddr);
    uint8_t rdata = _i2c_readNACK();
    _i2c_stop();
    return(rdata);
#else
    bme280Buffer[0] = BME280_CHIP_ID_REG;
    _i2c.write(i2cWAddr, bme280Buffer, 1, true);
    _i2c.read(i2cRAddr, bme280Buffer, 1, false);
    uint8_t rdata = bme280Buffer[0];
    return(bme280Buffer[0]);
#endif
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Soft reset the chip

uint8_t bme280::resetBme() {
#if defined  BMEi2cLOWLEVEL
    uint8_t rdata = _i2c_start(i2cWAddr);
    if(rdata) return(rdata);
    _i2c_write(BME280_RST_REG);
    _i2c_write(BME280_RESET_VALUE);
    _i2c_stop();
#else
    bme280Buffer[0] = BME280_RST_REG;
    bme280Buffer[1] = BME280_RESET_VALUE;
    uint8_t rdata = _i2c.write(i2cWAddr, bme280Buffer, 2, false);
#endif
    return(rdata);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Get BME280 status register.  Returns register value

uint8_t bme280::getBmeStatus() {
#if defined  BMEi2cLOWLEVEL
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_STAT_REG);
    _i2c_start(i2cRAddr);
    uint8_t rdata = _i2c_readNACK();
    _i2c_stop();
    return(rdata);
#else
    bme280Buffer[0] = BME280_STAT_REG;
    _i2c.write(i2cWAddr, bme280Buffer, 1, true);
    _i2c.read(i2cRAddr, bme280Buffer, 1, false);
    return(bme280Buffer[0]);
#endif
}    

//--------------------------------------------------------------------------------------------------------------------------------------//
// Get BME280 PTH values.  Saves raw data is data structure.  Returns 0 if successful, !0 if status was busy - pass thru from getBmeStatus();

uint8_t bme280::getBmeRawData(bme_data& bmed) {
    uint8_t rdata = getBmeStatus();
    if(rdata) return(rdata);
    bmed.raw_hum = 0;
#if defined  BMEi2cLOWLEVEL
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_PRESSURE_MSB_REG);
    _i2c_start(i2cRAddr);

    //                  MSB first               LSB second               XLSB third
    bmed.raw_baro = ((_i2c_readACK() << 12) | (_i2c_readACK() << 4) | (_i2c_readACK()));
    bmed.raw_temp = ((_i2c_readACK() << 12) | (_i2c_readACK() << 4) | (_i2c_readACK()));
    bmed.raw_hum =  ((_i2c_readACK() << 8) | (_i2c_readNACK()));
    _i2c_stop();
#else
    bme280Buffer[0] = BME280_PRESSURE_MSB_REG;
    _i2c.write(i2cWAddr, bme280Buffer, 1, true);
    _i2c.read(i2cRAddr, bme280Buffer, 8, false);
    //                    MSB first                LSB second               XLSB third
    bmed.raw_baro = ((bme280Buffer[0] << 12) | (bme280Buffer[1] << 4) | (bme280Buffer[2]));
    bmed.raw_temp = ((bme280Buffer[3] << 12) | (bme280Buffer[4] << 4) | (bme280Buffer[5]));
    bmed.raw_hum =  ((bme280Buffer[6] << 8)  | (bme280Buffer[7]));
#endif
    return(0);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
//Convert BME280 PTH values.  Takes raw data from data structure and applies calibration values to it.  

void bme280::convertBmeRawData(bme_data& bmed, bme_cal& bmec) { 
    
    //Returns temperature in DegC, resolution is 0.01 DegC. Output value of “5123” equals 51.23 DegC.
    //t_fine carries fine temperature as global value
    int var1t = ((((bmed.raw_temp >> 3) - ((int)bmec.dig_T1 << 1))) * ((int)bmec.dig_T2)) >> 11;
    int var2t = (((((bmed.raw_temp >> 4) - ((int)bmec.dig_T1)) * ((bmed.raw_temp >> 4) - ((int)bmec.dig_T1))) >> 12) *
        ((int)bmec.dig_T3)) >> 14;
    bmec.t_fine = var1t + var2t;
    bmed.corr_temp = (bmec.t_fine * 5 + 128) >> 8;
    bmed.bme_temp = (double)bmed.corr_temp / 100.0;
    
    //Returns pressure in Pa as unsigned 32 bit integer in Q24.8 format (24 integer bits and 8 fractional bits).
    //Output value of “24674867” represents 24674867/256 = 96386.2 Pa = 963.862 hPa
    int64_t var1p, var2p, p;
    var1p = ((int64_t)bmec.t_fine) - 128000;
    var2p = var1p * var1p * (int64_t)bmec.dig_P6;
    var2p = var2p + ((var1p * (int64_t)bmec.dig_P5) << 17);
    var2p = var2p + (((int64_t)bmec.dig_P4) << 35);
    var1p = ((var1p * var1p * (int64_t)bmec.dig_P3 )>> 8) + ((var1p * (int64_t)bmec.dig_P2) << 12);
    var1p = (((((int64_t)1) << 47) + var1p)) * ((int64_t)bmec.dig_P1) >> 33;
    if (var1p == 0) return;     // avoid exception caused by division by zero
    p = 1048576 - bmed.raw_baro;
    p = (((p << 31) - var2p) * 3125)/var1p;
    var1p = (((int64_t)bmec.dig_P9) * (p >> 13) * (p >> 13)) >> 25;
    var2p = (((int64_t)bmec.dig_P8) * p) >> 19;
    p = ((p + var1p + var2p) >> 8) + (((int64_t)bmec.dig_P7) << 4);
    bmed.corr_baro = p >> 8;
    bmed.bme_baro = (double)bmed.corr_baro / 100.0;

    //Returns humidity in %RH as unsigned 32 bit integer in Q22.10 format (22 integer and 10 fractional bits).
    //Output value of “47445” represents 47445/1024 = 46.333 %RH
    int v_x1_u32r = (bmec.t_fine - ((int)76800));
    v_x1_u32r = (((((bmed.raw_hum << 14) - (((int)bmec.dig_H4) << 20) - (((int)bmec.dig_H5) * v_x1_u32r)) +
        ((int)16384)) >> 15) * (((((((v_x1_u32r * ((int)bmec.dig_H6)) >> 10) * (((v_x1_u32r *
        ((int)bmec.dig_H3)) >> 11) + ((int)32768))) >> 10) + ((int)2097152)) *
        ((int)bmec.dig_H2) + 8192) >> 14));
    v_x1_u32r = (v_x1_u32r - (((((v_x1_u32r >> 15) * (v_x1_u32r >> 15)) >> 7) * ((int)bmec.dig_H1)) >> 4));
    v_x1_u32r = (v_x1_u32r < 0 ? 0 : v_x1_u32r);
    v_x1_u32r = (v_x1_u32r > 419430400 ? 419430400 : v_x1_u32r);
    bmed.corr_hum = (uint32_t)(v_x1_u32r >> 12);
    bmed.bme_hum = (double)bmed.corr_hum / 1024.0;  //was: / 1000.0
}

//--------------------------------------------------------------------------------------------------------------------------------------//
//Convert BME280 PTH values.  Takes raw data from data structure and applies calibration values to it.  
//Note: This is the floating point version.

void bme280::convertBmeRawDataFloat(bme_data& bmed, bme_cal& bmec) { 

    //Returns temperature in DegC, double precision. Output value of “51.23” equals 51.23 DegC.
    //t_fine carries fine temperature as global value
    double var1, var2;
    var1 = (((double)bmed.raw_temp) / 16384.0 - ((double)bmec.dig_T1) / 1024.0) * ((double)bmec.dig_T2);
    var2 = ((((double)bmed.raw_temp) / 131072.0 - ((double)bmec.dig_T1) / 8192.0) *
        (((double)bmed.raw_temp) / 131072.0 - ((double)bmec.dig_T1) / 8192.0)) * ((double)bmec.dig_T3);
    bmec.t_fine = (int)(var1 + var2);
    bmed.corr_temp = 0;
    bmed.bme_temp = (var1 + var2) / 5120.0;

    //Returns pressure in Pa as double. Output value of “96386.2” equals 96386.2 Pa = 963.862 hPa
    double p;
    var1 = ((double)bmec.t_fine / 2.0) - 64000.0;
    var2 = var1 * var1 * ((double)bmec.dig_P6) / 32768.0;
    var2 = var2 + var1 * ((double)bmec.dig_P5) * 2.0;
    var2 = (var2 / 4.0)+(((double)bmec.dig_P4) * 65536.0);
    var1 = (((double)bmec.dig_P3) * var1 * var1 / 524288.0 + ((double)bmec.dig_P2) * var1) / 524288.0;
    var1 = (1.0 + var1 / 32768.0)*((double)bmec.dig_P1);
    if (var1 == 0.0) {
        bmed.corr_baro = 0;
        bmed.bme_baro = 0.0;
        return; //avoid exception caused by division by zero
    }
    p = 1048576.0 - (double)bmed.raw_baro;
    p = (p - (var2 / 4096.0)) * 6250.0 / var1;
    var1 = ((double)bmec.dig_P9) * p * p / 2147483648.0;
    var2 = p * ((double)bmec.dig_P8) / 32768.0;
    p = p + (var1 + var2 + ((double)bmec.dig_P7)) / 16.0;
    bmed.corr_baro = 0;
    bmed.bme_baro = p / 100.0;

    //Returns humidity in %rH as as double. Output value of “46.332” represents 46.332 %rH
    double var_H;
    var_H = (((double)bmec.t_fine) - 76800.0);
    var_H = (bmed.raw_hum - (((double)bmec.dig_H4) * 64.0 + ((double)bmec.dig_H5) / 16384.0 * var_H)) *
        (((double)bmec.dig_H2) / 65536.0 * (1.0 + ((double)bmec.dig_H6) / 67108864.0 * var_H *
        (1.0 + ((double)bmec.dig_H3) / 67108864.0 * var_H)));
    var_H = var_H * (1.0 - ((double)bmec.dig_H1) * var_H / 524288.0);
    if (var_H > 100.0) {
        var_H = 100.0;
    } else if (var_H < 0.0) {
        var_H = 0.0;
    }
    bmed.corr_hum = 0;
    bmed.bme_hum = var_H;
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Initialize the chip

uint8_t bme280::initBme(bme_cal& bmec) {
#if defined  BMEi2cLOWLEVEL
    //initialize the chip
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_CTRL_HUMIDITY_REG);
    _i2c_write(BME280_CTRL_HUMIDITY_REG_DATA);
    _i2c_stop();
    
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_CTRL_MEAS_REG);
    _i2c_write(BME280_CTRL_MEAS_REG_DATA);
    _i2c_stop();
    
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_CONFIG_REG);
    _i2c_write(BME280_CONFIG_REG_DATA);
    _i2c_stop();
  
    //read back config registers
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_CTRL_HUMIDITY_REG);
    _i2c_start(i2cRAddr);
    bmec.ctrl_hum_reg  = _i2c_readACK();
    uint8_t status     = _i2c_readACK();
    bmec.ctrl_meas_reg = _i2c_readACK();
    bmec.config_reg    = _i2c_readNACK();
    _i2c_stop();
    
    //now get the calibration registers
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_CAL_DATA_START_1);
    _i2c_start(i2cRAddr);
    //                  LSB first             MSB second
    bmec.dig_T1 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_T2 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_T3 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P1 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P2 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P3 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P4 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P5 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P6 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P7 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P8 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_P9 = (_i2c_readACK() + (_i2c_readACK() << 8));
    uint8_t rdata  = (_i2c_readACK());    //dummy read of address 0xa0
    bmec.dig_H1 = (_i2c_readNACK());
    _i2c_stop();
    
    //finally, get the Humid calibration registers
    _i2c_start(i2cWAddr);
    _i2c_write(BME280_CAL_DATA_START_2);
    _i2c_start(i2cRAddr);
    bmec.dig_H2 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_H3 = (_i2c_readACK());
    bmec.dig_H4 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_H5 = (_i2c_readACK() + (_i2c_readACK() << 8));
    bmec.dig_H6 = (_i2c_readNACK());
    _i2c_stop();
    
#else
    //initialize the chip
    bme280Buffer[0] = BME280_CTRL_HUMIDITY_REG;
    bme280Buffer[1] = BME280_CTRL_HUMIDITY_REG_DATA;
    _i2c.write(i2cWAddr, bme280Buffer, 2, false);
    
    bme280Buffer[0] = BME280_CTRL_MEAS_REG;
    bme280Buffer[1] = BME280_CTRL_MEAS_REG_DATA;
    _i2c.write(i2cWAddr, bme280Buffer, 2, false);
    
    bme280Buffer[0] = BME280_CONFIG_REG;
    bme280Buffer[1] = BME280_CONFIG_REG_DATA;
    _i2c.write(i2cWAddr, bme280Buffer, 2, false);
    
    //read back config registers
    bme280Buffer[0] = BME280_CTRL_HUMIDITY_REG;
    _i2c.write(i2cWAddr, bme280Buffer, 1, true);
    _i2c.read(i2cRAddr, bme280Buffer, 4, false);
    bmec.ctrl_hum_reg  = bme280Buffer[0];
//    uint8_t status     = bme280Buffer[1];
    bmec.ctrl_meas_reg = bme280Buffer[2];
    bmec.config_reg    = bme280Buffer[3];
    
    //now get the calibration registers, Temp and Press first
    bme280Buffer[0] = BME280_CAL_DATA_START_1;
    _i2c.write(i2cWAddr, bme280Buffer, 1, true);
    _i2c.read(i2cRAddr, bme280Buffer, 26, false);
    //                  LSB first             MSB second
    bmec.dig_T1 = (bme280Buffer[0] | (bme280Buffer[1] << 8));
    bmec.dig_T2 = (bme280Buffer[2] | (bme280Buffer[3] << 8));
    bmec.dig_T3 = (bme280Buffer[4] | (bme280Buffer[5] << 8));
    bmec.dig_P1 = (bme280Buffer[6] | (bme280Buffer[7] << 8));
    bmec.dig_P2 = (bme280Buffer[8] | (bme280Buffer[9] << 8));
    bmec.dig_P3 = (bme280Buffer[10] | (bme280Buffer[11] << 8));
    bmec.dig_P4 = (bme280Buffer[12] | (bme280Buffer[13] << 8));
    bmec.dig_P5 = (bme280Buffer[14] | (bme280Buffer[15] << 8));
    bmec.dig_P6 = (bme280Buffer[16] | (bme280Buffer[17] << 8));
    bmec.dig_P7 = (bme280Buffer[18] | (bme280Buffer[19] << 8));
    bmec.dig_P8 = (bme280Buffer[20] | (bme280Buffer[21] << 8));
    bmec.dig_P9 = (bme280Buffer[22] | (bme280Buffer[23] << 8));
//    uint8_t rdata  = (bme280Buffer[24]);    //dummy read of address 0xa0
    bmec.dig_H1 = (bme280Buffer[25]);
    
    //finally, get the Humid calibration registers
    bme280Buffer[0] = BME280_CAL_DATA_START_2;
    _i2c.write(i2cWAddr, bme280Buffer, 1, true);
    _i2c.read(i2cRAddr, bme280Buffer, 8, false);
    bmec.dig_H2 = (bme280Buffer[0] | (bme280Buffer[1] << 8));
    bmec.dig_H3 = (bme280Buffer[2]);
    bmec.dig_H4 = ((bme280Buffer[4] & 15) | (bme280Buffer[3] << 4));
    bmec.dig_H5 = (((bme280Buffer[4] >> 4) & 15) | (bme280Buffer[5] << 4));
    bmec.dig_H6 = (bme280Buffer[6]);
    
#endif 
    return(0);
    }

//--------------------------------------------------------------------------------------------------------------------------------------//
// Return corrected altitude (in feet) from barometer at sea level (in mB)

float bme280::getAltitudeFT(bme_data& bmed, float sea_pressure) {
    return(float)((1 - (pow((bmed.bme_baro / (double)sea_pressure), 0.190284))) * 145366.45);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Return corrected barometer, based on altitude (in feet)
    
float bme280::getSeaLevelBaroFT(bme_data& bmed, float known_alt) {
    return(pow(pow((bmed.bme_baro * MB_INHG_DOUBLE), 0.190284) + 0.00001313 * (double)known_alt , 5.2553026) * INHG_MB_DOUBLE);
} 

//--------------------------------------------------------------------------------------------------------------------------------------//
// Return corrected barometer, based on altitude (in meters)

float bme280::getSeaLevelBaroM(bme_data& bmed, float known_alt) {
    return(pow(pow((bmed.bme_baro * MB_INHG_DOUBLE), 0.190284) + 0.00001313 * (double)known_alt * FEET_METERS , 5.2553026) * INHG_MB_DOUBLE);
} 

//--------------------------------------------------------------------------------------------------------------------------------------//
// Return dew point. More accurate, slower

float bme280::getDewPt(bme_data& bmed) {
    // dewPoint function NOAA
    // reference: http://wahiduddin.net/calc/density_algorithms.htm    
    double A0= 373.15 / (273.15 + (double)bmed.bme_temp);
    double SUM = -7.90298 * (A0 -1);
    SUM += 5.02808 * log10(A0);
    SUM += -1.3816e-7 * (pow(10, (11.344 * (1 - 1/A0))) -1) ;
    SUM += 8.1328e-3 * (pow(10,(-3.49149 * (A0 -1))) -1) ;
    SUM += log10(1013.246);
    double VP = pow(10, SUM -3) * bmed.bme_hum;
    double T = log(VP / 0.61078);   // temp var
    return (241.88 * T) / (17.558 - T);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Return dew point. Less accurate, faster

float bme280::getDewPtFast(bme_data& bmed) {
    // delta max = 0.6544 wrt dewPoint()
    // 5x faster than dewPoint()
    // reference: http://en.wikipedia.org/wiki/Dew_point
    double bmeDtzA = 17.271;
    double bmeDtzB = 237.7;
    double bmeDtzC = (bmeDtzA * bmed.bme_temp) / (bmeDtzB + bmed.bme_temp) + log(bmed.bme_hum / 100.0);
    double bmeDtzD = (bmeDtzB * bmeDtzC) / (bmeDtzA - bmeDtzC);
    return (bmeDtzD);
}



