#ifndef PrintBuffer_H
#define PrintBuffer_H

#include "mbed.h"

/** Print Buffer Routine. Displays X number of lines in a buffer.
 * each line broken int 2 groups of HEX characters followed by 2 groups of ASCII characters.
 * Illegal ascii characters are represented as a cyan colored dot "."
 *
 * example; dump myBuffer, 4 lines starting at line 18
 *
 * dump("This is myBuffer:", 4, 0x120, myBuffer);   //Note: this line will be colored magenta below
 *
 * This is myBuffer: - lines: 4   starting at: 0x0120
 *
 * 0120  00 04 09 29 00 00 00 00   00 00 00 00 00 00 00 00   ...).... ........
 *
 * 0130  00 00 00 00 00 00 00 00   00 00 00 00 00 00 4c 00   ........ ......L.
 *
 * 0140  00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00   ........ ........
 *
 * 0150  00 00 00 00 00 00 00 00   00 00 00 00 65 6e 64 00   ........ ....end.
 *
 * @code
 * #include "mbed.h"
 * #include "PrintBuffer.h" 
 * 
 * PrintBuffer pb("pb");  
 * DigitalOut led1(LED1, "led1");
 * DigitalOut led2(LED2, "led2");
 * Serial pc(USBTX, USBRX);
 *
 * int gDebug = 1;
 * char* myBuffer = new char[256];       //example buffers
 * char* myOtherBuffer = new char[64];
 *
 * int main() {
 *     pc.baud(921600); 
 *     for(int i = 0; i < 256; i++) {    //example, fill myBuffer with 0, 1, 2, 3, 4...256
 *         myBuffer[i] = i;
 *     }
 *     for(int i = 0; i < 64; i++) {     //example, fill myOtherBuffer with 192, 193, 194, 195...256
 *         myOtherBuffer[i] = i + 192;
 *     }
 *     pc.printf("example, 16 lines of myBuffer, starting at 0x0000\n");
 *     pb.dump(16, 0, myBuffer);
 *     pc.printf("example, 3 lines of myBuffer, starting at 0x0034\n");
 *     pb.dump(3, 0x34, myBuffer);
 *     pc.printf("example, 4 lines of myOtherBuffer, starting at 0x0000\n");
 *     pb.dump(4, 0, myOtherBuffer);
 *     led2 = 1;
 *     while(true) {
 *           wait(0.5);
 *           led1 = !led1;
 *           led2 = !led2;
 *     }
 * }
 * @endcode
 */

class PrintBuffer {

public:

    /** Create a PrintBuffer object with a name attached 
     *
     * @param constructor, - for PrintBuffer
     */
    PrintBuffer(const char* name);
    
    /** Print out hex / ascii data using pc.printf using format below
     *
     * @param three, (int # of lines to print), (int starting address), (c c name of buffer to print)
     */
    int dump(const char* title, int BufferLines, int BufferOffset, const char buffer[]);
    
    /** Print out hex / ascii data using pc.printf using format below
     *
     * @param three, (int # of lines to print), (int starting address), (c u8_t name of buffer to print)
     */
    int dump_t(const char* title, int BufferLines, int BufferOffset, const uint8_t buffer[]);
    
    /** Print out hex / ascii data using pc.printf using format below
     *
     * @param three, (int # of lines to print), (int starting address), (c unsigned char name of buffer to print)
     */
    int dump_uc(const char* title, int BufferLines, int BufferOffset, const unsigned char buffer[]);
    
    /** Print out hex / ascii data using pc.printf using format below
     *
     * @param four, (int # of lines to print), (int starting address), (c unsigned char name of buffer to print), (int starting address)
     */
    int dump_a(const char* title, int BufferLines, int BufferOffset, const char buffer[], int addr);
    
private:

    int BufferLines;

}; 
 
#endif