#include "mbed.h"
#include "TSISensor.h"                      //touch sensor
#include "MMA8451Q.h"                       //accelerometer

int revision = 100;

Serial pc(USBTX, USBRX);                    //tx, rx

MMA8451Q acc(PTE25, PTE24, 0x3a);           //accelerometer
I2C i2c0(PTE0, PTE1);                       //i2c bus 0
I2C i2c1(PTE25, PTE24);                     //i2c bus 1

#define LED_ON      0
#define LED_OFF     1
DigitalOut rled(LED_RED);                   //KL25Z red led
DigitalOut gled(LED_GREEN);                 //KL25Z green led
// DO NOT Use LED_BLUE - pin PTD1 already in use by WiFi chip

int gDebug = 1;                             //global debug print flag

//--------------------------------------------------------------------------------------------------------------------------------------//
//RTC specific variables and constants

#define TBUFLEN             10              //size of smaller ascii time buffers
time_t ctTime;                              //time structure
int DST = 0;                                //Daylight Saving Time (or as defined in .ini file)
int TZone = -8;                             //Time Zone from UTC (or as defined in .ini file)
char timebuf_hm[TBUFLEN];                   //local time format buffer - 21:16
char timebuf_hms[TBUFLEN];                  //local time format buffer - 21:16:43
char timebuf_mdy[TBUFLEN];                  //local time format buffer - 04/01/14
char timebuf_dMyy[TBUFLEN + 4];             //local time format buffer - 01-Apr-2014

int BootTime = 0;                           //RTC at boot up
int StartTime = 0;                          //time we powered up
int SysUpTime = 0;                          //time we've been alive
int EventTime = 0;                          //for 143 debug, next time to save data in EVENTS.CSV
int RTCupdate = 0;                          //time until next NTP update timer

//--------------------------------------------------------------------------------------------------------------------------------------//
// Print build date / time  Note: display is UTC, not local time
// Adafruit CC3000 WiFi Shield with uFL Connector for Ext Antenna
// http://www.adafruit.com/products/1534

void printBuildDate() {
    pc.printf("KL25Z / WWVB Tests v%d  build: " __DATE__ " " __TIME__ "(UTC)  K Braun\n", revision);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// initial splash display

void initSplash() {
    pc.printf("\n\n");
    pc.printf("-------------------------------------------------------------------\n");
    printBuildDate();
}

//--------------------------------------------------------------------------------------------------------------------------------------//
//Detect I2C device chain

extern I2C i2c0;
extern I2C i2c1;

int dispI2C(int i2cchain) {
    int count = 0;
    for (int address = 2; address < 256; address +=2) {
        if(((i2cchain == 0) && (!i2c0.write(address, NULL, 0))) || ((i2cchain == 1) && (!i2c1.write(address, NULL, 0)))) {
            wait_us(50);
            pc.printf("   - i2c chain %d: device found at address 0x%02X  \n", i2cchain, address);
            count++;
        }        
    }
    return(count);
}
//--------------------------------------------------------------------------------------------------------------------------------------//
//Detect I2C device chain

void find_i2c(int i2cbus) {
    pc.printf("Searching for devices on i2c bus %d ...\n", i2cbus);
    int count = dispI2C(i2cbus);
    pc.printf(" - %2d devices found on i2c bus %d\n", count, i2cbus);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Update time

void UpdateTime() {
    ctTime = time(NULL) + ((TZone + DST) * 3600);   //timezone and dst offsets
    SysUpTime = ctTime - StartTime;
    strftime(timebuf_mdy, TBUFLEN, "%m/%d/%y", localtime(&ctTime));
    strftime(timebuf_dMyy, TBUFLEN * 2, "%d-%b-%Y", localtime(&ctTime));
    strftime(timebuf_hm, TBUFLEN, "%H:%M", localtime(&ctTime));
    strftime(timebuf_hms, TBUFLEN, "%H:%M:%S", localtime(&ctTime));
}


//--------------------------------------------------------------------------------------------------------------------------------------//
//general housekeeping items whilw waiting in loops

void HouseKeeping() {
    UpdateTime();
}

//--------------------------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------------------------//
// *  \brief HTTP client demo
// *  \param  none
// *  \return int

int main() {
    rled = LED_OFF;
    gled = LED_OFF;
    ctTime = time(NULL);
    StartTime = ctTime;                     //get time we started up
    BootTime = ctTime;                      //save boot time for later (RTC debug)

    pc.baud(115200);
    initSplash();
    
    //look for devices on i2c buses
    find_i2c(0);
    find_i2c(1);

    //check to see if the RTC is operational.  
    //NOTE: bootloader must be MBED, not FRDMKL25Z. Otherwise pin PTC1 (J10-12) will not have a clock signal for the RTC
    if(gDebug) pc.printf("Checking the RTC...  ");
    wait_ms(1500);
    ctTime = time(NULL);
    if(gDebug) pc.printf("ctTime: %d   StartTime: %d\r\n", ctTime, StartTime);
    if((StartTime == ctTime) || (ctTime <= 1000000)) {
        pc.printf("*** Local RTC stopped, initializing the RTC.  !!CHECK BATTERY!!\r\n");
        pc.printf("*** Note: Time is incorrect, needs to be updated!!!\r\n");
        set_time(1357000000);                //set time to something close, Jan01 2013 00:26:40, but still way off
    } else {
        if(gDebug) pc.printf("Local RTC is running\r\n");
    } 
    UpdateTime();
    StartTime = ctTime;
    if(gDebug) pc.printf("Date: %s   ", timebuf_dMyy);
    if(gDebug) pc.printf("Time: %s \n", timebuf_hms);
    
    //set up the cc3000 paramaters
    TSISensor touchSensor;
        
    float accels[3];
    float resting,reading;
    float THRESHOLD = 0.15;
    float position;
    
    wait_ms(500);
    acc.getAccAllAxis( accels );
    resting = abs(accels[0]) + abs(accels[1]) + abs(accels[2]);
    pc.printf("Set up resting accelerometer - %.2f\r\n", resting);
    wait_ms(100);
    rled = LED_OFF;
    gled = LED_OFF;
    pc.printf("Ready\r\n");
    int bumps = 0;
    while(1) {
        HouseKeeping();
        acc.getAccAllAxis( accels );
        reading = abs(accels[0]) + abs(accels[1]) + abs(accels[2]);
        if (abs(resting - reading) > THRESHOLD ){
            rled = LED_ON;
            gled = LED_ON;
            wait_ms(100);
            bumps++;
            position = touchSensor.readPercentage();
            //pc.printf( "BUMPED  %d times,  resting %.2f   read %.2f  position %.2f \r\n", bumps, resting, reading, position );
            //pc.printf("Date: %s   ", timebuf_dMyy);
            //pc.printf("Time: %s \n", timebuf_hms);
            rled = LED_OFF;
            wait_ms(100);
            gled = LED_OFF;
            rled = LED_ON;
            wait_ms(100);
            rled = LED_OFF;
            wait_ms(100);
        }
    
    }
    
}

