#include "mbed.h"
#include "button_group.hpp"
#include "LCDclockSet.h"                    //various display graphics for clock

#if defined(TARGET_DISCO_F746NG)
#endif

//------------------------
//hardware equates

LCD_DISCO_F746NG lcd;                       // Object for LCD display
TS_DISCO_F746NG ts;                         // Object for touch pannel
RawSerial pc(USBTX, USBRX);                 // terminal console
DigitalOut gled(LED1);

LCDclockSet ss("ss");                       //graphic display items

TS_StateTypeDef TS_State;

//------------------------
//LCD specific variables and constants

uint8_t lcd_text[30];

//------------------------
//RTC specific variables and constants

#define TBUFLEN                     10      //size of smaller ascii time buffers
time_t ctTime;                              //time structure
int DST = 1;                                //Daylight Saving Time (or as defined in .ini file)
int TZone = -8;                             //Time Zone from UTC (or as defined in .ini file)
char timebuf_s[4];
char timebuf_hm[TBUFLEN];                   //local time format buffer - 21:16
char timebuf_hms[TBUFLEN];                  //local time format buffer - 21:16:43
char timebuf_mdy[TBUFLEN];                  //local time format buffer - 04/01/14
char timebuf_dMyy[TBUFLEN + 4];             //local time format buffer - 01-Apr-2014
char timebuf_Myy[TBUFLEN];                  //local time format buffer - -Apr-2014
char timebuf_dow[1];                        //local time format buffer - 5

//--------------------------------------------------------------------------------------------------------------------------------------//
// Display clock on LCD

#define CLOCKX                  338
#define CLOCKLINE               21

void DispClockLCD() {
    sprintf((char*)lcd_text, "%s %s", timebuf_dMyy, timebuf_hms);
    uint32_t tc = lcd.GetTextColor();
    lcd.SetTextColor(LCD_COLOR_YELLOW);
    lcd.DisplayStringAt(CLOCKX, LINE(CLOCKLINE) + 8, (uint8_t *)&lcd_text, LEFT_MODE);
    lcd.SetTextColor(tc);
}

//--------------------------------------------------------------------------------------------------------------------------------------//
// Update time

void UpdateTimeRegs() {
    strftime(timebuf_s, 4, "%S", localtime(&ctTime));
    strftime(timebuf_mdy, TBUFLEN, "%m/%d/%y", localtime(&ctTime));
    strftime(timebuf_dMyy, TBUFLEN * 2, "%d-%b-%Y", localtime(&ctTime));
    strftime(timebuf_Myy, TBUFLEN, "-%b-%Y", localtime(&ctTime));
    strftime(timebuf_hm, TBUFLEN, "%H:%M", localtime(&ctTime));
    strftime(timebuf_hms, TBUFLEN, "%H:%M:%S", localtime(&ctTime));
    strftime(timebuf_dow, 1, "%u", localtime(&ctTime));
}

void UpdateTime() {
    ctTime = time(NULL) + ((TZone + DST) * 3600);   //timezone and dst offsets
    UpdateTimeRegs();
}

//--------------------------------------------------------------------------------------------------------------------------------------//
//Wait and get a touched character

int readTouch() {
    do{
        wait_ms(50);
        ts.GetState(&TS_State);
    } while(!(TS_State.touchDetected));
    int TouchData = 255;
    uint8_t idx;
    uint16_t x, y;
    do{
        wait_ms(50);
        for (idx = 0; idx < TS_State.touchDetected; idx++) {
            x = TS_State.touchX[idx];
            y = TS_State.touchY[idx];
            TouchData = ss.Touch12Block(x, y);
        }
        ts.GetState(&TS_State);
    } while(TS_State.touchDetected);
    return(TouchData);
}
    
//--------------------------------------------------------------------------------------------------------------------------------------//
//Get 6 decimal characters from the TFT / touchscreen - use for date and time setting

char dtArray[6];

void Get6() {
    while(1) {
        int rt = 255;
        int b = 1;
        bool isNumb = false;
        while(1) {
            do{
                wait_ms(100);
                rt = readTouch();
            } while(rt == 255);
            
            if(rt < 10) {
                ss.TouchClockBlock(b, rt, false);
                isNumb = true;
                dtArray[b - 1] = rt;
                if(b < 6) b++;
            }
            if(rt == '<') {
                ss.TouchClockBlock(b, NULL, true);
                if(b > 1) {
                    b--;
                    ss.TouchClockBlock(b, NULL, true);
                }
                isNumb = false;
            }
            if((rt == '#') && (b == 6) && (isNumb == true)) {
                isNumb = false;
                return;
            }
            if(rt == '#') {
                isNumb = false;
                wait_ms(100);
            }
            if(b > 6) b = 6;
            if(b < 1) b = 1;
            rt = 255;
            wait_ms(300);
        }
    }
}

//--------------------------------------------------------------------------------------------------------------------------------------//
//Manually set the date and time

char DateTime[12]; 

void SetRTClcd() {
    ss.Draw12Block(1, LCD_COLOR_DARKGREEN, LCD_COLOR_YELLOW);  
    ss.DrawClockBlock('/'); 
    Get6();
    for(int i = 0; i < 6; i++) {
        DateTime[i] = dtArray[i];
    }
    ss.Draw12Block(1, LCD_COLOR_DARKGREEN, LCD_COLOR_YELLOW); 
    ss.DrawClockBlock(':'); 
    Get6();
    for(int i = 0; i < 6; i++) {
        DateTime[i + 6] = dtArray[i];
    }

    struct tm t;
    t.tm_year = (2000 + (DateTime[4] * 10) + DateTime[5]);
    t.tm_mon = ((DateTime[0] * 10) + DateTime[1]);
    t.tm_mday = ((DateTime[2] * 10) + DateTime[3]);
    t.tm_hour = ((DateTime[6] * 10) + DateTime[7]);
    t.tm_min = ((DateTime[8] * 10) + DateTime[9]);
    t.tm_sec = ((DateTime[10] * 10) + DateTime[11]);
    
    // adjust for tm structure required values
    t.tm_year = t.tm_year - 1900;
    t.tm_mon = t.tm_mon - 1;    
    // set the time
    time_t ctTime = mktime(&t);
    ctTime = ctTime - (TZone + DST) * 3600;     //take out local time
    set_time(ctTime);
    UpdateTime();
}

//--------------------------------------------------------------------------------------------------------------------------------------//
//main

int main()
{
    //TS_StateTypeDef TS_State;
    uint16_t x, y;
    uint8_t status;
    uint8_t idx;
    uint8_t cleared = 0;
    uint8_t prev_nb_touches = 0;
    
    gled = 1;
    
    pc.baud(230400);
    
    // Setting of button group
    lcd.Clear(LCD_COLOR_BLACK);
    lcd.SetBackColor(LCD_COLOR_BLACK);
    lcd.SetTextColor(LCD_COLOR_CYAN);
    lcd.SetFont(&Font20);
    lcd.DisplayStringAt(0, LINE(5), (uint8_t *)"Set Clock Via LCD Demo", CENTER_MODE);
    wait(3.0);
    
    status = ts.Init(lcd.GetXSize(), lcd.GetYSize());
    if (status != TS_OK) {
        lcd.Clear(LCD_COLOR_RED);
        lcd.SetBackColor(LCD_COLOR_RED);
        lcd.SetTextColor(LCD_COLOR_WHITE);
        lcd.DisplayStringAt(0, LINE(5), (uint8_t *)"TOUCHSCREEN INIT FAIL", CENTER_MODE);
    } else {
        lcd.Clear(LCD_COLOR_GREEN);
        lcd.SetBackColor(LCD_COLOR_GREEN);
        lcd.SetTextColor(LCD_COLOR_WHITE);
        lcd.DisplayStringAt(0, LINE(5), (uint8_t *)"TOUCHSCREEN INIT OK", CENTER_MODE);
    }

    wait(1.5);
    
    //go set the RTC via the LCD
    SetRTClcd();

    lcd.SetFont(&Font12);
    lcd.Clear(LCD_COLOR_BLUE);
    lcd.SetBackColor(LCD_COLOR_BLUE);
    lcd.SetTextColor(LCD_COLOR_WHITE);

    //the touchscreen demo with RTC in the lower-right corner
    while (true) {
        ts.GetState(&TS_State);
        if (TS_State.touchDetected) {
            // Clear lines corresponding to old touches coordinates
            if (TS_State.touchDetected < prev_nb_touches) {
                for (idx = (TS_State.touchDetected + 1); idx <= 5; idx++) {
                    lcd.ClearStringLine(idx);
                }
            }
            prev_nb_touches = TS_State.touchDetected;

            cleared = 0;

            sprintf((char*)lcd_text, "Touched: %d", TS_State.touchDetected);
            lcd.DisplayStringAt(0, LINE(0), (uint8_t *)&lcd_text, LEFT_MODE);

            for (idx = 0; idx < TS_State.touchDetected; idx++) {
                x = TS_State.touchX[idx];
                y = TS_State.touchY[idx];
                sprintf((char*)lcd_text, "Touch %d: x=%d y=%d    ", idx+1, x, y);
                lcd.DisplayStringAt(0, LINE(idx+1), (uint8_t *)&lcd_text, LEFT_MODE);
            }
            lcd.DrawPixel(TS_State.touchX[0], TS_State.touchY[0], LCD_COLOR_ORANGE);
        } else {
            if (!cleared) {
                lcd.Clear(LCD_COLOR_BLUE);
                sprintf((char*)lcd_text, "Touches: 0");
                lcd.DisplayStringAt(0, LINE(0), (uint8_t *)&lcd_text, LEFT_MODE);
                cleared = 1;
            }
        }
        gled = !gled;
        wait_ms(200);
        UpdateTime();
        DispClockLCD();
    }
}
