#ifndef AD9851_H
#define AD9851_H

#include "mbed.h"

/** A AD9851 driver interface to control the DDS-60 from AmQRP
 *
 * @code
 * #include "mbed.h"
 * #include "AD9851.h"
 * 
 * AD9851 freqSyn(p13, p14, p15);          //sdo, clk, len
 * 
 * int main() {
 *     freqSyn.AD9851Enable();                  //turn on AD9851
 *     freqSyn.AD9851Disable();                 //turn off AD9851
 *
 *     float Fdata;
 *     Fdata = freqSyn.GetBaseValue();          //get Base frequency
 *     Fdata = freqSyn.GetIfValue();            //get IF offset frequency
 *
 *     freqSyn.SetBaseValue(7320000.0);         //set new Base frequency (0.0 - 179,999,999.999MHz)
 *     freqSyn.SetIfValue(1620000.0);           //set new IF frequency (0.0 - 179,999,999.999MHz)
 *     freqSyn.SetM6Value('A');                 //set multiplier value 0 = 30MHz max, 6 = 180MHz max, A = auto 
 *
 *     //functions below are results from CalcNewValue()
 *     bool ErFlag;
 *     ErFlag = freqSyn.CalcNewValue();         //calculate new AD9851 value, based on Base, IF and M6 values
 *                                              //if ok, new value output to AD9851
 *                                              //note: reverts to old values if Base + IF * M6 is over limit, sets error flag
 *                                              //ERROR occured with new values used by CalcNewValue()
 *
 *     ErFlag = freqSyn.GetErrFlagValue();      //get error value from last time CalsNewValue() was executed
 *   
 *     unsigned int UIdata; 
 *     UIdata = freqSyn.GetFD32Value();         //get 32 bit hex value of AD9851 data
 *   
 *     char Cdata;
 *     Cdata = freqSyn.GetFortyValue();         //get 8 bit hex value of AD9851 control register (enable & M6). Phase offset always 0
 * }
 * @endcode
 */

class AD9851 /*: public Stream*/ { // or  : public Base

public:
/** Create a frequency selector object 
    * @param SDO serial data out
    * @param CLK serial clock - data clocked on + edge
    * @param LEN data latch enable strobe - high pulse after all 40 bits of data clocked in
    */
    AD9851(PinName SDO, PinName CLK, PinName LEN);

//  Destroys object
    ~AD9851();

//  Returns the stored Base Freq, IF Freq, 32 bit serial word and 5th control byte values    
    double GetBaseValue();
    double GetIfValue();
    unsigned int GetFD32Value();
    char GetFortyValue();
    bool GetErrFlagValue();

//  Set values   
    double SetBaseValue(double bv);
    double SetIfValue(double bv);
    char SetM6Value(char bx);

//  Turn the AD9851 on / off    
    void AD9851Enable();
    void AD9851Disable();
    

//  calculates new AD9851 value - returns to old values if error occurs, sets error flag
    bool CalcNewValue();
    
protected:
    DigitalOut _clk, _sdo, _len;  
    double bv; 
    char bx;
        
private:

//  transfers new 40 bit word to the AD9851
    void OutNewValue();
    void FirstAccess();


};    
#endif