#include "mbed.h"
#include "useful.h"

#include "led.h"        /* Basic On-Board LED driver functions */
#include "digital.h"    /* Test the basic digital I/O of the MBED board */
#include "analog.h"     /* code to do things to the a2d sub system */
#include "i2c.h"        /* basic I2C Functions */
#include "can.h"        /* Basic CAN bus test functions */
#include "ticker.h"     /* Ticker look, flash a led et. al. */
#include "serial.h"     /* Talk to the RS232 line */
#include "network.h"    /* Test code for the TCP/IP network */
#include "main.h"       /* routines in this file */
#include "breakup.h"    /* breakup a string to its bits */
#include "local_defines.h"
#include "pca9685_reg.h"    /* Needed for the led driver init */
#include "scripting.h"  /* Lets see if we can get a simple scripting system going */
#include "SNTPClient.h" /* Time handling code */
#include "SDFileSystem.h"   /* Look at the SD card */
#include "sound.h"      /* Test code for the sound I/I system */

#include "help_data.h"    /* a text file for the command help data */

#include "cmd.h"


SDFileSystem sd(p5, p6, p7, p27, "sd");

LocalFileSystem local("local");

FILE    *exec_fp;

extern char station_id;
extern char pwd[0x20];
extern int set_baud_rate;
extern int sys_state;
extern int var[MAX_VAR];
extern int jmp[MAX_JMP];


/******************************************/
/*                                        */
/* Exec commands, for the Diags Code      */
/*                                        */
/******************************************/

struct    cmd_callback {
    char    *label;            /* command */
    int         cmd;            /* switch value */
    int         args;            /* Number of args for this command, 0 let sub-routine handle */
    void    (*cmd_routine)(int,char **);    /* call */
};
struct cmd_callback cmd_cb[] ={
    /* General commands for the command interpriter */
    {   "exec",     0,      0,    exec_file},           /* Run a script file from the local file store */
    {   "cat",      0,      2,    cat_file},            /* Cat a file on the file system to the current output device */
    {   "cp",       0,      3,    cp_file},             /* copy file from to */
    {   "dir",      0,      1,    dir_file},            /* Get a directory listing from the local file store */
    {   "cd",       0,      0,    cd_file},             /* Change directory */
    {   "pwd",      0,      0,    pwd_file},            /* show the user the current directory */
    {   "wait",     0,      2,    wait_control},        /* wait for X tenths of a second */
    {   "baud",     0,      2,    baud_rate},           /* Set the baudrate of the serial I/O */
    {   "print",    0,      0,    print_string},        /* Print a string to current output device */
    {   "help",     0,      1,    help_display},        /* Show the user some help data */
    {   "state",    0,      0,    show_state},          /* Show the current setup state of the SBC */
    {   "date",     0,      1,    date_cmd},            /* tell the user the data from the RTC */
    {   "dateset",  0,      2,    date_set},            /*Set the RTC to a new time */
    {   "station",  0,      2,    station_set},         /* Set the CAN bus station ID */
    {   "netinit",  0,      1,    netinit},             /* DHCP the network */
    {   "set",      0,      0,    set_var},             /* Set the variable to a value */
    {   "sum",      0,      0,    sum_var},             /* Add some basic maths */
    {   "point",    0,      0,    point_var},           /* Set out loop point */
    {   "tst",      0,      0,    tst_var},             /* test the value of a var */
    {   "clr",      0,      0,    clr_var},             /* Clear the var's to zero */
    /* Commands for the Diags */
    {   "list",     0,      1,    list},                /* List I2C Devices to user */
    {   "i2cw",     0,      3,    i2cw},                /* Write a value to the I2C bus */
    {   "i2cr",     0,      2,    i2cr},                /* Read a value from the I2C bus */
    {   "led",      0,      0,    led_test},            /* Simple test of the MBED LED's */
    {   "sseg",     0,      0,    sseg_test},           /* Simple test of the  Seven Seg displays*/
    {   "rs232",    0,      2,    rs232_test_code},     /* Some test code for the RS232 interface */
    {   "a2d",      0,      2,    a2d_test},            /* Simple test of the  A2D system */
    {   "snd",      0,      1,    snd_test},            /* Sound test code */
    {   "digital",  0,      2,    dig_test},            /* Simple test of the Digital I/O */
    {   "lcd",      0,      2,    lcd_test},            /* Test the attached LCD display */
    {   "lcdprt",   0,      0,    lcd_print},           /* print a string on the Batron LCD */
    {   "lcdclr",   0,      0,    lcd_clear},           /* Clear the Batron screen */
    {   "kbd",      0,      2,    kbd_test},            /* Test the attached keyboard */
    {   "canrx",    0,      1,    canrx},               /* Make this box a CAN Reciever */
    {   "cantx",    0,      0,    cantx},               /* Transmit a CAN message */
    {   "cancmd",   0,      3,    can_cmd},             /* send a command out on the can bus */
    {   "ticker",   0,      1,    ticker_test},         /* Test the ticker calls */
    /* End of command table marker */
    {   "test",      0,      2,    cmd_test},            /* We need somewhere to test commands first :-) */
    {   "END",      -1,    -1,    error_handler}
};


/* Scan down tags, to find the match, else return -1 */
int find_cmd(char *tag)
{
    int     cnt=0,c;
    int  length=strlen(tag);
    char *elements[0x10];

    if(tag[0]=='#') /* See is we have a command or a comment */
        return(0);

    c = breakup(tag,elements,' ');
    while(1){
        if((strcmp(cmd_cb[cnt].label,"END"))==0){
                  if(length>1)
            cmd_cb[cnt].cmd_routine(c,elements);  /* this will call the error routine */
            return(cmd_cb[cnt].cmd);
        } else if((strcmp(cmd_cb[cnt].label,tag))==0){
            if((cmd_cb[cnt].args==c) || (cmd_cb[cnt].args==0)){    /* if arg count set to zero, can be anything */
                cmd_cb[cnt].cmd_routine(c,elements);        /* Call the function with args */
                return(cnt);
            } else {
                lprintf("\n\rWrong number of args passed to %s\n\r",cmd_cb[cnt].label);
                return(-1);
            }
        }
        cnt++;
    }
}


void error_handler(int c, char **a)
{
    int cnt=0;
    lprintf("\n\rCommands available are ");
    while(1){
        if((strcmp(cmd_cb[cnt].label,"END"))==0){
            lprintf("\n\r");
            return;
        } else
            lprintf("%s, ",cmd_cb[cnt].label);
        cnt++;
    }
}

/******************************************/
/*                                        */
/* Command stubs are here                 */
/*                                        */
/******************************************/

void exec_file(int c, char **a)
{
    /* File on a[1] */
    
    char     buf[0x60];
    char     cmd[0x60];
    int      cnt=0;
    
    sprintf(buf,"%s%s",pwd,a[1]);
   
    if((exec_fp = fopen(buf,"r"))==NULL){
        lprintf("Unable to open %s\n\r",buf);
        return;
    }
    sys_state = sys_state | EXEC_CALLED;
    while(!feof(exec_fp)){
        fgets(buf,sizeof(buf),exec_fp);
        cnt++;
        if(c==3)
            lprintf("exec line (%d) is :%s\n",cnt,buf);
        if(strlen(buf) > 2){
            sprintf(cmd,"%s\n",buf);
            find_cmd(cmd);
        }
        buf[0]='\0';
        buf[1]='\0';
    }
    fclose(exec_fp);
    sys_state = TO_USB;
}
void cat_file(int c, char **a)
{
    /* File on a[1] */
    
    FILE    *fp;
    char     buf[0x60];
    
    sprintf(buf,"%s%s",pwd,a[1]);
    
    if((fp = fopen(buf,"r"))==NULL){
        lprintf("Unable to open %s\n\r",buf);
        return;
    }
    
    while(!feof(fp)){
        fgets(buf,sizeof(buf),fp);
        if(strlen(buf) > 2){
            lprintf(buf);
        }
        buf[0]='\0';
        buf[1]='\0';
    }
    fclose(fp);
}

void cp_file(int c, char **a)
{
    /* from File on a[1] to file on a[2], full paths supplied */
    
    FILE    *fi,*fo;
    char     buf[0x60];
    
    if((fi = fopen(a[1],"r"))==NULL){
        lprintf("Unable to open %s to read\n",a[1]);
        return;
    }
    if((fo = fopen(a[2],"w"))==NULL){
        lprintf("Unable to open %s to write\n",a[2]);
        fclose(fi);
        return;
    }
    while(!feof(fi)){
        fgets(buf,sizeof(buf),fi);
        if(strlen(buf) > 2){
            fprintf(fo,buf);
        }
        buf[0]='\0';
        buf[1]='\0';
    }
    fclose(fi);
    fclose(fo);
}

void dir_file(int c, char **a)
{
    DIR *d;
    struct dirent *p;
    char     buf[0x60];
    
    sprintf(buf,"%s%s",pwd,a[1]);
    d = opendir(buf);
    if(d != NULL) {
        while((p = readdir(d)) != NULL) {
            lprintf("- %s\n\r", p->d_name);
        }
    } else {
        error("Could not open directory!");
    }
    closedir(d);
}
void cd_file(int c, char **a)
{
    int x;
    char y;
    
    y = a[1][strlen(a[1])];
    
    if(a[1][0]=='/'){
        if(y!='/')
            sprintf(pwd,"%s/",a[1]);
        else
            sprintf(pwd,a[1]);
    } else
        lprintf("File Path like /local/\nCurrently you are in %s\n",pwd);
    x = 0;
    if((strcmp(pwd,"/sd/"))==0)
        x = 1;
    else  if((strcmp(pwd,"/local/"))==0)
        x = 1;
    if(!x){
        lprintf("Only directories are /local/ or /sd/ (If the SD card is in slot, resetting to /local/\n");
        sprintf(pwd,"/local/");
    }
}
void pwd_file(int c, char **a)
{
    lprintf("%s\n",pwd);
}
void wait_control(int c, char **a)
{
    wait(htoi(a[1])/10);
}

void print_string(int c, char **a)
{
    int cnt = 1;
    int len;
    while(cnt <= c){
        if(a[cnt][0]=='v' || a[cnt][0]=='V'){
            len = strlen(a[cnt]);
            if(len == 2){
                lprintf(" %04x",htoi(a[cnt]));
            } else {
                lprintf("%s",a[cnt]);
            }
        } else
            lprintf("%s",a[cnt]);
        cnt++;
    }
    lprintf("\n");
}

void help_display(int c, char **a)
{
    lprintf("%s",HELP_STRING);
    lprintf("%s",HELP_STRING2);
    lprintf("%s",HELP_STRING3);
    lprintf("%s",HELP_SCRIPT1);
}

/******************************************/
/*                                        */
/* Diag Command stubs are here            */
/*                                        */
/******************************************/

void list(int c, char **a)
{
    i2c_probe();
}

void i2cw(int c, char **a)
{
    pio_write(htoi(a[1]),htoi(a[2]));
}

void i2cr(int c, char **a)
{
    pio_read(htoi(a[1]));
}

void led_test(int c, char **a)
{
    if(c==2){
        test_leds(htoi(a[1]));
    } else  if(c==3){
        if((strcmp(a[1],"on"))==0){
            if((strcmp(a[2],"all"))==0)
                led_on(htoi(0));
            else
                led_on(htoi(a[2]));
        } else if((strcmp(a[1],"off"))==0){
             if((strcmp(a[2],"all"))==0)
                led_off(htoi(0));
            else
                led_off(htoi(a[2]));
        } else
            lprintf("LED Control Error: led on/off <led>\n");
            
    } else {
        lprintf("led <num>           - simple led test, go round loop num times\n");
        lprintf("led <on/off> <num>  - Turn <led num> <on/off>\n");
    }
}

void sseg_test(int c, char **a)
{
    if(c==2){
        lprintf("Testing the Seven Segment Display, will go roung the test loop %d times\n",htoi(a[1]));
        test_seven_seg(htoi(a[1]));
    } else if (c==3 & (strcmp(a[1],"val"))==0){
        sseg_four_digits(htoi(a[2]));
    } else {
        lprintf("sseg <num>     - basic test of sseg, loop num times\n");
        lprintf("sseg val <num> - display number on the sseg display\n");
    }
}

void a2d_test(int c, char **a)
{
    show_a2d();
}

void snd_test(int c, char **a)
{
    sound();
}

void dig_test(int c, char **a)
{
    lprintf("Testing the Digital I/O, will go roung the test loop %d times\n",htoi(a[1]));
    digital_test(htoi(a[1]));
}

/******************************************/
/*                                        */
/* Test routines for the Keyboard and LCD */
/* at the mo we need to know what type of */
/* keyboard and LCD we have, carried on   */
/* the supplied argument                  */
/*                                        */
/******************************************/
void lcd_test(int c, char **a)
{
    switch(htoi(a[1])){
        case    1      :    /* Robot Electronics I2C LCD interface */
            init_lcd();     /* Setup the LCD */
            clear_lcd();    /* Clear the screen */
            write_lcd("Hello");    /* Send it some text */
            break; 
        case    2      :    /* Batron I2C LCD interface */
            batron_lcd_init();
            batron("Hello World");
            break;
        default        :
            lprintf("Unknown LCD type %d\n",htoi(a[1]));
    }
}

void lcd_print(int c, char **a)
{
    batron_lcd_init();
    batron(a[1]);
}
void lcd_clear(int c, char **a)
{
    batron_clear();
}

void kbd_test(int c, char **a)
{
    char    r;
    switch(htoi(a[1])){
        case    3      :    /* Robot Electronics I2C LCD interface */
            lprintf("Will take 4 key presses of the Robot Electronics key pad\n");
            lprintf("Displaying the results to the user\n");
            r = blocking_read_keyboard();
            lprintf("Key 1 %d, or 0x%x, or %c\n",r,r,r);
            r = blocking_read_keyboard();
            lprintf("Key 2 %d, or 0x%x, or %c\n",r,r,r);
            r = blocking_read_keyboard();
            lprintf("Key 3 %d, or 0x%x, or %c\n",r,r,r);
            r = blocking_read_keyboard();
            lprintf("Key 4 %d, or 0x%x, or %c\n",r,r,r);
            break;
        case    4      :    /* Work around, with PCF8574 when with Batron I2C LCD */
            lprintf("Will take 4 key presses of thePCF8574s key pad\n");
            lprintf("Displaying the results to the user\n");
            r = pcf8574_kbd(1);
            lprintf("Key 1 %d, or 0x%x, or %c\n",r,r,r);
            r = pcf8574_kbd(1);
            lprintf("Key 2 %d, or 0x%x, or %c\n",r,r,r);
            r = pcf8574_kbd(1);
            lprintf("Key 3 %d, or 0x%x, or %c\n",r,r,r);
            r = pcf8574_kbd(1);
            lprintf("Key 4 %d, or 0x%x, or %c\n",r,r,r);
            break;
        default        :
            lprintf("Unknown KBD type %d\n",htoi(a[1]));
    }
    
}
/******************************************/
/*                                        */
/* CAN Bus tests, run from here           */
/*                                        */
/******************************************/
void canrx(int c, char **a)
{
    lprintf("Locking into the CAN message feed, till RESET is pressed\n");
    while(1){
        can_receive();
    }
}
void cantx(int c, char **a)
{
    if(c== 2 | c==4){
        if((strcmp(a[1],"prt"))==0){
            can_send_string(htoi(a[2]),a[3]);
        } else if((strcmp(a[1],"raw"))==0){
            can_send_raw(htoi(a[2]),a[3]);
        } else if((strcmp(a[1],"sync"))==0){
            can_send_sync();
        } else if((strcmp(a[1],"int"))==0){
            can_send_int(htoi(a[2]),htoi(a[3]),0);
        } else if((strcmp(a[1],"time"))==0){
            can_send_int(htoi(a[2]),htoi(a[3]),1);
        } else if((strcmp(a[1],"who"))==0){
            can_send_who();
        } else
            lprintf("cantx commnads are, sync, int, raw, who, prt\n");
    } else {
        lprintf("cantx <prt/raw> <remote station> <message>\n");
        lprintf("cantx int <remote station> <value>\n");
        lprintf("cantx time <remote station> <value>\n");
        lprintf("cantx sync\n");
        lprintf("cantx who\n");
    }
}
void can_cmd(int c, char **a)
{
    can_send_string(htoi(a[1]),a[2]);
}
/******************************************/
/*                                        */
/* TCP/IP Networking test tool box        */
/*                                        */
/******************************************/
void netinit(int c, char **a)
{
     network_init();
     lprintf("Starting SNTP\n");
     SNTPReadIniFile("/local/sntp.ini");
//     SNTPWriteIniFile(stdout);
     SNTPClientInit();
}

/******************************************/
/*                                        */
/* Test the Ticker system                 */
/*                                        */
/******************************************/
void ticker_test(int c, char **a)
{
     setup_ticker();
}

/******************************************/
/*                                        */
/* Set the station ID for the CAN bus     */
/*                                        */
/******************************************/
void station_set(int c, char **a)
{
     station_id = htoi(a[1]);
}

/******************************************/
/*                                        */
/* Display the system time to the user    */
/*                                        */
/******************************************/
void date_cmd(int c, char **a)
{
     time_t seconds = time(NULL);
     lprintf("mbed RTC time(%d): %s\n",seconds,ctime(&seconds));
}
void date_set(int c, char **a)
{
    time_t seconds;
    seconds = atoi(a[1]);     /* http://www.onlineconversion.com/unix_time.htm  */
    set_time(seconds);
}
/******************************************/
/*                                        */
/* Display the system state to the user   */
/*                                        */
/******************************************/
#define STATE_VAR   0x0001
#define STATE_I2C   0x0002
#define STATE_BRD   0x0004
#define STATE_JMP   0x0008

void show_state(int c, char **a)
{
     int    cnt = 0;
     int    show=0;
     
     if(c!=1){
        if((strcmp(a[1],"var"))==0)
            show = STATE_VAR;
        else if((strcmp(a[1],"i2c"))==0)
            show = STATE_I2C;
        else if((strcmp(a[1],"board"))==0)
            show = STATE_BRD;
        else if((strcmp(a[1],"jmp"))==0)
            show = STATE_JMP;
        else
            lprintf("Unknown state %s, should be var, jmp, i2c, board\n",a[1]);
    } else {
        show = STATE_VAR | STATE_I2C | STATE_BRD | STATE_JMP;
    }
            
    if(show & STATE_BRD){
        time_t seconds = time(NULL);
        lprintf("mbed RTC time(%d): %s",seconds,ctime(&seconds));
        lprintf("Baudrate Setting is %d\n",set_baud_rate);
        lprintf("Station CAN ID is %d\n",station_id);
        lprintf("Sysem State is %08x\nVariables, ",sys_state);
     }
     if(show & STATE_VAR){
        lprintf("System Variables\n");
        while(cnt != MAX_VAR){
            lprintf("%04x, ",var[cnt]);
            cnt++;
        }
        lprintf("\n");
     }
     if(show & STATE_JMP){
        lprintf("System Jump Table\n");
        while(cnt != MAX_JMP){
            lprintf("%04x, ",jmp[cnt]);
            cnt++;
        }
        lprintf("\n");
     }
     if(show & STATE_I2C){
        lprintf("Attached I2C devices \n");
        i2c_probe();
     }
}

/******************************************/
/*                                        */
/*  SD card file system test              */
/*                                        */
/******************************************/
void sdcard_test(int c, char **a)
{
    sys_state = (sys_state & ~MEMSTICK) | SDCARD;
    lprintf("Set the file system to the SD Card\n");
}   
/******************************************/
/*                                        */
/*  Mem Stick file system test            */
/*                                        */
/******************************************/
void memstick_test(int c, char **a)
{
    sys_state = (sys_state & ~SDCARD) | MEMSTICK;
    lprintf("Set the file system to the Memory Stick\n");
}   
/******************************************/
/*                                        */
/* General Test command for the source    */
/*                                        */
/******************************************/
void cmd_test(int c, char **a)
{
    int b=0;
    while(b!=htoi(a[1])){
        sseg_four_digits(b);
        wait(0.1);
        b++;
    }
}