#include "mbed.h"
#include "breakup.h"
#include "useful.h"
#include "i2c.h"
#include "serial.h"
#include "cmd.h"
#include "local_defines.h"
#include "scripting.h"


LocalFileSystem local("local");
FILE    *exec_fp;
extern int sys_state;
extern int var[MAX_VAR];
extern int jmp[MAX_JMP];



/******************************************/
/*                                        */
/* Exec commands, for the dome            */
/*                                        */
/******************************************/

struct    cmd_callback {
    char    *label;            /* command */
    int     cmd;               /* switch value */
    int     args;              /* Number of args for this command */
    void    (*cmd_routine)(int,char **);    /* call */
};
struct cmd_callback cmd_cb[] ={
    {  "list",     0,   1,    list},                /* List I2C Devices to user */
    {  "list2",    0,   1,    list2},               /* List I2C Devices to user */
    {  "exec",     0,   2,    exec_file},           /* Run a script file from the local file store */
    {  "dir",      0,   1,    dir_file},            /* Get a directory listing from the local file store */
    {  "cat",      0,   2,    cat_file},            /* Cat a file on the file system to the current output device */
    {  "relay",    0,   2,    relay_control},       /* Switch a relay 0-8 */
    {  "light",    0,   3,    light_control},       /* Set a light level on a channel */
    {  "wait",     0,   2,    wait_control},        /* wait for X tenths of a second */
    {  "baud",     0,   2,    baud_rate},           /* Set the baudrate of the serial I/O */
    {  "pca",      0,   2,    init_pca},            /* Init a PCA9685 device from the command line */
    {  "print",    0,   0,    print_string},        /* Print a string to current output device */
    {   "set",      0,      0,    set_var},             /* Set the variable to a value */
    {   "sum",      0,      0,    sum_var},             /* Add some basic maths */
    {   "point",    0,      0,    point_var},           /* Set out loop point */
    {   "tst",      0,      0,    tst_var},             /* test the value of a var */
    {   "clr",      0,      0,    clr_var},             /* Clear the var's to zero */
    {  "END",      -1,  -1,   error_handler}
};


/* Scan down tags, to find the match, else return -1 */
int find_cmd(char *tag)
{
    int     cnt=0,c;
    int  length=strlen(tag);
    char *elements[0x10];

    if(tag[0]=='#') /* See is we have a command or a comment */
        return(0);

    c = breakup(tag,elements,' ');
    while(1){
        if((strcmp(cmd_cb[cnt].label,"END"))==0){
                  if(length>1)
            cmd_cb[cnt].cmd_routine(c,elements);  /* this will call the error routine */
            return(cmd_cb[cnt].cmd);
        } else if((strcmp(cmd_cb[cnt].label,tag))==0){
            if((cmd_cb[cnt].args==c) || (cmd_cb[cnt].args==0)){    /* if arg count set to zero, can be anything */
                cmd_cb[cnt].cmd_routine(c,elements);        /* Call the function with args */
                return(cnt);
            } else {
                lprintf("\n\rWrong number of args passed to %s\n\r",cmd_cb[cnt].label);
                return(-1);
            }
        }
        cnt++;
    }
}


void error_handler(int c, char **a)
{
    int cnt=0;
    lprintf("\n\rCommands available are ");
    while(1){
        if((strcmp(cmd_cb[cnt].label,"END"))==0){
            lprintf("\n\r");
            return;
        } else
            lprintf("%s, ",cmd_cb[cnt].label);
        cnt++;
    }
}

/******************************************/
/*                                        */
/* Command stubs are here                 */
/*                                        */
/******************************************/

void list(int c, char **a)
{
    i2c_probe();
}
void list2(int c, char **a)
{
    i2c_probe2();
}

void exec_file(int c, char **a)
{
    /* File on a[1] */
    
    FILE    *fp;
    char     buf[0x60];
    char     cmd[0x60];
    
    sprintf(buf,"/local/%s",a[1]);
    
    if((fp = fopen(buf,"r"))==NULL){
        lprintf("Unable to open %s\n\r",buf);
        return;
    }
    sys_state = sys_state | EXEC_CALLED;
    while(!feof(fp)){
        fgets(buf,sizeof(buf),fp);
        if(strlen(buf) > 2){
            sprintf(cmd,"%s\n",buf);
            find_cmd(cmd);
        }
        buf[0]='\0';
        buf[1]='\0';
    }
    fclose(fp);
}
void dir_file(int c, char **a)
{
    lprintf("List of files in /local\n\r");

    DIR *d;
    struct dirent *p;
    d = opendir("/local");
    if(d != NULL) {
        while((p = readdir(d)) != NULL) {
            lprintf(" - %s\n\r", p->d_name);
        }
    } else {
        error("Could not open directory!");
    }
}
void cat_file(int c, char **a)
{
    /* File on a[1] */
    
    FILE    *fp;
    char     buf[0x60];
    
    sprintf(buf,"/local/%s",a[1]);
    
    if((fp = fopen(buf,"r"))==NULL){
        lprintf("Unable to open %s\n\r",buf);
        return;
    }
    
    while(!feof(fp)){
        fgets(buf,sizeof(buf),fp);
        if(strlen(buf) > 2){
            lprintf(buf);
        }
        buf[0]='\0';
        buf[1]='\0';
    }
    fclose(fp);
}


void relay_control(int c, char **a)
{
    relay_operate(htoi(a[1]));
}

void wait_control(int c, char **a)
{
    wait(htoi(a[1])/10);
}

void light_control(int c, char **a)
{
     channel_light(htoi(a[1]), htoi(a[2]));
}
void init_pca(int c, char **a)
{
    init_pca9685(htoi(a[1]));
}

void print_string(int c, char **a)
{
    int cnt = 1;
    int len;
    while(cnt <= c){
        if(a[cnt][0]=='v' || a[cnt][0]=='V'){
            len = strlen(a[cnt]);
            if(len == 2){
                lprintf(" %04x",htoi(a[cnt]));
            } else {
                lprintf("%s",a[cnt]);
            }
        } else
            lprintf("%s",a[cnt]);
        cnt++;
    }
    lprintf("\n");
}