#include "mbed.h"

/*
**************************************************************************************
AD5627.  Dual 12-bit DAC with i2C interface and external reference pin.
(The AD56x7R have an internal reference). Power supply: 2.7V to 5.5V.

If needed, the two DACs can be updated simultaneously by LDAC (pin 4)or by software.

Wiring:
pin 3 (GND) to ground. pin 9 (Vdd) to 3.3V.
pin 10 (Vref) to 3.3V, pin 6 (ADDR) to 3.3V (A1=0, A0=0).
4.7k pull-up resistors on pins 8 and 7 (SDA, SCL)(to mBed's pins 9 and 10).
pin 4 (LDAC) and pin 5 (CLR) to mBed's digital output pins 11 and 12.

Attach voltmeter to ground and Vout A (pin 1)  or  Vout B (pin 2).

Command byte: 0 S C2 C1 C0 A2 A1 A0. s=1: multiple block write,  s=0: one block write.

Commands C2 C1 C0:
000: Write to input register.
001: Update DAC register.
010: Write to input register n, update all (software LDAC).
011: Write and update DAC channel n.
100: Power up / power down.
101: Reset.
110: LDAC register setup.
111: Internal reference setup (on/off).

DAC address command A2 A1 A0:
000: DAC A.
001: DAC B.
111: Both DACs.


Author: Lluis Nadal. August 2011.
**************************************************************************************
*/

I2C i2c(p9, p10); // SDA, SCL
Serial pc(USBTX, USBRX);
DigitalOut LDAC (p11);
DigitalOut CLR (p12);

DigitalOut L1(LED1);
DigitalOut L2(LED2);
DigitalOut L3(LED3);
DigitalOut L4(LED4);

const int addr_R = 0x19; // Address to read
const int addr_W = 0x18; // Address to write

const int CA1 = 0x18; // Command: DAC A one block write
const int CB1 = 0x19; // Command: DAC B one block write
const int CA2 = 0x58; // Command: DAC A multiple block write
const int CB2 = 0x59; // Command: DAC B multiple block write
const int CA3 = 0x00; // Command: DAC A using LDAC to update.
const int CB3 = 0x01; // Command: DAC B using LDAC to update.


void write_A(float v) { // 0 < v < 3.3 V
    char H;      // High byte
    char L;      // Low byte
    int n;
    n = (int)(v*4096/3.3);
    pc.printf(" n= %d\r\n", n);
    L = n<<4 & 0xF0;
    H = n>>4;
    pc.printf(" (H, L)=(%d, %d)\r\n", H, L);
    pc.printf("\r\n");
    i2c.start();
    i2c.write(addr_W); // Write address to write
    i2c.write(CA1);    // Write command
    i2c.write(H);      // Write high byte
    i2c.write(L);      // Write low byte
    i2c.stop();
}


void write_B(float v) { // 0 < v < 3.3 V
    char H;      // High byte
    char L;      // Low byte
    int n;
    n = (int)(v*4096/3.3);
    pc.printf(" n= %d\r\n", n);
    L = n<<4 & 0xF0;
    H = n>>4;
    pc.printf(" (H, L)=(%d, %d)\r\n", H, L);
    pc.printf("\r\n");
    i2c.start();
    i2c.write(addr_W); // Write address to write
    i2c.write(CB1);    // Write command
    i2c.write(H);      // Write high byte
    i2c.write(L);      // Write low byte
    i2c.stop();
}

void write_A_LDAC(float v) { // 0 < v < 3.3 V
    char H;      // High byte
    char L;      // Low byte
    int n;
    n = (int)(v*4096/3.3);
    pc.printf(" n= %d\r\n", n);
    L = n<<4 & 0xF0;
    H = n>>4;
    pc.printf(" (H, L)=(%d, %d)\r\n", H, L);
    pc.printf("\r\n");
    i2c.start();
    i2c.write(addr_W); // Write address to write
    i2c.write(CA3);    // Write command
    i2c.write(H);      // Write high byte
    i2c.write(L);      // Write low byte
    i2c.stop();
}

void write_B_LDAC(float v) { // 0 < v < 3.3 V
    char H;      // High byte
    char L;      // Low byte
    int n;
    n = (int)(v*4096/3.3);
    pc.printf(" n= %d\r\n", n);
    L = n<<4 & 0xF0;
    H = n>>4;
    pc.printf(" (H, L)=(%d, %d)\r\n", H, L);
    pc.printf("\r\n");
    i2c.start();
    i2c.write(addr_W); // Write address to write
    i2c.write(CB3);    // Write command
    i2c.write(H);      // Write high byte
    i2c.write(L);      // Write low byte
    i2c.stop();
}


int main() {
    L1 = 0;
    i2c.frequency(100000);

    // Pulsing this pin low allows any or all DAC registers to be updated if they have new data.
    LDAC = 0;

    CLR = 0; // Outputs the two DACs to 0V.
    wait(5);
    CLR = 1;
    L1 = 1;

    // DAC A one block write
    write_A(2.537);
    wait(5);
    
    // DAC A one block write
    write_A(1.777);
    L2 = 1;
    wait(5);

    // DAC B one block write
    write_B(0.666);
    wait(5);
    
    // DAC B one block write
    write_B(3.000);
    wait(5);


    // Using LDAC pin
    LDAC = 1;
    write_A_LDAC(1.000); // DAC register A is not updated
    write_B_LDAC(1.500); // DAC register B is not updated
    L3 = 1;
    wait(5);

    LDAC = 0; // DAC registers A and B are updated simultaneously
    L4 = 1;

}
