/* mbed TextDisplay Library Base Class
 * Copyright (c) 2007-2009 sford
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * A common base class for Text displays
 * To port a new display, derive from this class and implement
 * the constructor (setup the display), character (put a character
 * at a location), rows and columns (number of rows/cols) functions.
 * Everything else (locate, printf, putc, cls) will come for free
 *
 * The model is the display will wrap at the right and bottom, so you can
 * keep writing and will always get valid characters. The location is
 * maintained internally to the class to make this easy
 */

#ifndef MBED_TEXTDISPLAY_H
#define MBED_TEXTDISPLAY_H

#include "mbed.h"
#include "mbed.h"

/** TextDisplay parent class, use one the derived classes
 * Example:
 * @code
 / simple test for TextDisplay class
 #include "mbed.h"
 #include "sed133xLCD.h"
 
 sed133xLCD lcd(p5, p6, p7, p8, p9, p28, p27, p26, p25, p24, p23, p22, p21,256,128, 6,9, "lcd");
 
 
 int main() {
      lcd.printf("Hello TextDisplay world!\r\n");
      lcd.claim(stdout);
      printf("hello stream world\r\n");
 }

 * @endcode
 */
 
class TextDisplay : public Stream {
public:

    // functions needing implementation in derived implementation class

    /** Create a TextDisplay interface
     *
     * @param name The name used in the path to access the strean through the filesystem
     */

    TextDisplay(const char *name = NULL);

    /** output a character at the given position
     *
     * @param column column where charater must be written
     * @param  row where character must be written
     * @param c the character to be written to the TextDisplay
     */

    virtual void character(uint16_t column, uint16_t row, int c) = 0;

    /** return number if rows on TextDisplay
     * @result number of rows
     */
    virtual uint16_t rows() = 0;
    /** return number if columns on TextDisplay
    * @result number of rows
    */

    virtual uint16_t columns() = 0;

    // functions that come for free, but can be overwritten
    /** redirect output from a stream (stoud, sterr) to  display
    * @param stream stream that shall be redirected to the TextDisplay
    */
    virtual bool claim (FILE *stream);
    /** clear screen
    */
    virtual void cls();
    /** locate cursor on given position
    * @param column horizontal position
    * @param row vertical positon
    */
    virtual void locate(uint16_t column, uint16_t row);
     /** set foreground colour if supported on display
     * @param colour Hexadecimal 0xRRGGBB, RR = red level, GG = green level, BB = blue level
     */
    virtual void foreground(uint32_t colour);
     /** set background colour if supported on display
     * @param colour Hexadecimal 0xRRGGBB, RR = red level, GG = green level, BB = blue level
     */
    virtual void background(uint32_t colour);
    // putc (from Stream)
    // printf (from Stream)

protected:

    virtual int _putc(int value);
    virtual int _getc();

    // character location
    uint16_t _column;
    uint16_t _row;

    // colours
    uint32_t _foreground;
    uint32_t _background;
    char *_path;
};

#endif
