// mbed Sed133xLCD Library Base Class
// Based on TextLCD (Copyright (c) 2007-2009 sford)
// Released under the MIT License: http://mbed.org/license/mit
// Author: louis Lagendijk
//
//
#include "TextDisplays.h"

#ifndef MBED_Sed133xLCD_H
#define MBED_Sed133xLCD_H

    enum dataType_t {
        t_data = 0,
        t_command = 1
    };

    enum command_t {
        NO_COMMAND = 0,
        SYSTEM_SET = 0x40,
        SET_MEM = 0x42,
        GET_MEM = 0x43,
        SCROLL_SET = 0x44,
        SET_CURSOR = 0x46,
        GET_CURSOR = 0x47,
        CURSOR_DIR_RIGHT = 0x4c,
        CURSOR_DIR_LEFT = 0x4d,
        CURSOR_DIR_UP = 0x4e,
        CURSOR_DIR_DOWN = 0x4f,
        SLEEP_IN = 0x53,
        DISPLAY_OFF = 0x58,
        DISPLAY_ON = 0x59,
        HORIZONTAL_SCROLL_POSITION = 0x5a,
        OVERLAY_FORMAT = 0x5b,
        CHAR_GEN_RAM_ADDR = 0x5c,
        CURSOR_FORM = 0x5d
    };

    struct commandName_t {
        command_t value;
        char *text;
    };

/** Sed133xLCD Textdisplay
 * 
 * Supports SED1330 (and probably the SED1335 and SED1336).
 * Based on datasheet 268-0.4

 *This code drives the display using the 8080 interface (SEL1 = 0, SEL2 = 0).
 * Required pins:
 * @code
 * 8 bits for the databus +
 * *Reset (note 1)
 * *Read
 * *Write
 * *ChipSelect
 * A0
 *
 * Note1: Reset is a Schmitt trigger input that can normally not be driven from
 * an output pin. The other pins can be directly connected to the MBED 
 *           +5V
 *            _
 *           | | 10k
 *           |_|
 *            |
 * MBED__    |/
 *  -|___|---|  BC547
 *    22k    |\>
 *            |
 *           ___
 * @endcode
 *
 * Character sizes can be set as desired (defaults to 6*8). The SED seems to use a 
 * 5*7 font internally other font sizes use additional white space around the 
 * characters.
 * Font widths of more than 8 pixels are not supported.
 * The actual size of the display in characters is caluclated from the horizontal/ 
 * vertical resolution and the specified font size.
 
 * Example:
 * @code
 / simple test for TextDisplay class
 #include "mbed.h"
 #include "Sed133xLCD.h"
 
 Sed133xLCD lcd(p5, p6, p7, p8, p9, p28, p27, p26, p25, p24, p23, p22, p21,256,128, 6,9, "lcd");
 
 
 int main() {
      lcd.printf("Hello TextDisplay world!\r\n");
      lcd.claim(stdout);
      printf("hello stream world\r\n");
 }

 * @endcode
 */
class Sed133xLCD : public TextDisplay {
public:
    /** Create the TextDisplay interface
     *
     * @param reset PinName for reset
     * @param nRead PinName for /read
     * @param nWrite PinName for /write
     * @param nSelect PinName for /select
     * @param addr0 PinName for addr0
     * @param d0-d7 PinName for D0 -D7
     * @param hor_dots horizontal resolution  of display (in pixels)
     * @param vert_dots vertical resolution of display (in pixels)
     * @param char_width disply width of character (in pixels)
     * @param char_height display height of character (in pixels)8 
     * @param name name to be used in pathname of the stream 
     */

    Sed133xLCD(PinName reset, PinName nRead, PinName nWrite, PinName nSelect,PinName addr0,
               PinName d0, PinName d1, PinName d2, PinName d3,
               PinName d4, PinName d5, PinName d6, PinName d7,
               uint16_t hor_dots = 256, uint16_t vert_dots = 128,
               uint8_t char_width = 6, uint8_t char_height = 8, const char *name = "sed133x");
    virtual void character(uint16_t column, uint16_t row, int c);
    virtual uint16_t rows() {
        return _rows;
    }
    virtual uint16_t columns() {
        return _columns;
    }
    virtual void initializeSed();
    virtual void cls();

    // locate, putc, printf come from parent class

protected:
    commandName_t commandNameTab[];
    char *commandName(command_t command_val);
    void busWait(void);
    void resetSed(void);
    void sendByte(dataType_t type, uint8_t data);
    uint8_t getByte(dataType_t type);
    void sendCommand(uint8_t data);
    void sendData(uint8_t data);
    void systemSet(void);
    void scrollSet(void);
    void horizontalScrollPosition(void);
    void overlayFormat(void);
    void cursorDirection(void);
    void cursorForm(void);
    void displayOn(void);
    void displayOff(void);
    void charGenRamAddr(void);
    void init(void);
    void setCursor(uint8_t column, uint8_t row);
    void setCursor(int offset);
    void printText(uint8_t *text);
    void printData(int len, uint8_t *data);
    void clsText(void);
    void clsGraphics(void);
    void clearCharacterGeneratorRam(void);

    DigitalOut _reset, _nRead, _nWrite, _nSelect, _addr0;
    BusInOut _dataBus;
    uint16_t _hor_dots, _vert_dots, _rows, _columns, _char_width, _char_height;
};
#endif



