/* mbed TextLCD Library
 * Copyright (c) 2007-2009 sford
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * Modified by Ned Konz to provide better support for 4-line LCDs and ones with other controller chips.
 */

#include "TextLCD.h"
#include "mbed.h"

/*
 * useful info found at http://www.a-netz.de/lcd.en.php
 *
 * Initialisation
 * ==============
 *
 * After attaching the supply voltage/after a reset, the display needs to be brought in to a defined state
 *
 * - wait approximately 15 ms so the display is ready to execute commands
 * - Execute the command 0x30 ("Display Settings") three times (wait 1,64ms after each command, the busy flag cannot be queried now).
 * - The display is in 8 bit mode, so if you have only connected 4 data pins you should only transmit the higher nibble of each command.
 * - If you want to use the 4 bit mode, now you can execute the command to switch over to this mode now.
 * - Execute the "clear display" command
 *
 * Timing
 * ======
 *
 * Nearly all commands transmitted to the display need 40us for execution.
 * Exceptions are the commands "Clear Display and Reset" and "Set Cursor to Start Position"
 * These commands need 1.64ms for execution. These timings are valid for all displays working with an
 * internal clock of 250kHz. But I do not know any displays that use other frequencies. Any time you
 * can use the busy flag to test if the display is ready to accept the next command.
 *
 * _e is kept low except when being used.
 * _rw is kept 0 (write) apart from actions that use it differently
 * _rs is set by the data/command writes
 */

TextLCD::TextLCD(PinName rs, PinName rw, PinName e, PinName d0, PinName d1,
                 PinName d2, PinName d3, uint16_t rows, uint16_t cols, const char *name) : 
                 TextDisplay(name), _rw(rw), _rs(rs),
        _e(e), _d(d0, d1, d2, d3), _rows(rows), _columns(cols) {

    _rw = 0;
    wait_us(1);     // min. 100nsec delay
    _e  = 0;
    _rs = 0;    // command mode
    _d.output();

    reset();
    cls();
}

void TextLCD::reset() {
    wait_ms(15);
    // e is low at this point, as is rw.
    // 2. Send 0x3 and wait 150 ms (will stay in 8-bit mode if already there)
    writeHalfByte(0x3);
    wait_ms(5);
    // 3. Send 0x3 and wait 150 ms (will go to 8-bit mode if was in 4-bit without any garbage nibble)
    writeHalfByte(0x3);
    wait_ms(5);
    // 4. Send 0x3 and wait 250 ms (will go to 8-bit mode even if garbage nibble was previously received)
    writeHalfByte(0x3);
    wait_ms(5);
    // 5. Send 0x2 and wait 200 ms (should go to 4-bit mode now)
    writeHalfByte(0x2);
    wait_ms(5);
    // 7. Send LCD setup sequence (eg 0x2, 0x8 (=0x28), 0x0, 0x8 (=0x08), etc.)
    writeCommand(0x28);    // Function set 001 BW N F - -
    wait_ms(15);

    writeCommand(0x08);     // display off, cursor invisible
    wait_ms(15);

    writeCommand(0x01);
    wait_ms(15);      // 1.64ms command

    writeCommand(0x0C);     // display enabled, cursor invisible
    wait_ms(15);

    writeCommand(0x6);  //  Cursor Direction and Display Shift : 0000 01 CD S (CD 0-left, 1-right S(hift) 0-no, 1-yes
    wait_ms(15);

    locate(0,0);
}

// memory starts at 0x80, and is 0x40 chars long per row
// However, rows 2 and 3 of 4-line displays are actually adjacent to rows 0 and 1.
// 16x4 displays are addressed the same way as 20x4 ones.

void TextLCD::character(uint16_t column, uint16_t row, int c) {
    int address;
    address = 0x80 + ((row & ~2) * 0x40) + column;
    if (row > 1)
        address += 20;
    writeCommand(address);
    writeData(c);
}

void TextLCD::writeHalfByte(uint16_t value) {
    _e = 1;
    wait_us(1);
    _d = value & 0x0F;    // send data on bus
    wait_us(1); // setup time
    _e = 0; // strobe
    wait_us(1); // hold time
}

void TextLCD::writeByte(uint16_t value) {
    writeHalfByte(value>>4);
    writeHalfByte(value);
}

void TextLCD::writeCommand(uint16_t command) {
    _rs = 0;
    writeByte(command);
    waitUntilDone();
}

void TextLCD::writeData(uint16_t data) {
    _rs = 1;
    writeByte(data);
    waitUntilDone();
}

void TextLCD::cls() {
    writeCommand(0x01);
    wait_us(2000);      // 1.64ms command
    locate(0,0);
}

// This should be changed to use readAddressAndBusy() when that works.
void TextLCD::waitUntilDone() {
    wait_us(60);
}

// Return the busy/address byte.
// The busy flag is the high bit.
// Not yet working reliably.
uint16_t TextLCD::readAddressAndBusy() {
    _d.input();
    _rw = 1;
    wait_us(1);
    _e = 1;
    wait_us(1);
    _e = 0;

    uint16_t retval = _d.read() << 4;

    wait_us(1);
    _e = 1;
    wait_us(1);
    _e = 0;

    retval |= _d.read();
    _rw = 0;

    _d.output();
    return retval;
}

