//Library includes
#include "mbed.h"
#include "rtos.h"
 
#define MAX_DELAY   10000   //Max size of delay effect sample buffer(too large can crash on compile from memory limit)
 
//mbed pin declarations

AnalogIn pot(p20);
DigitalOut led(LED1);
DigitalOut led2(LED2);
AnalogIn input(p15);
AnalogOut output(p18);
Serial pc(USBTX, USBRX);
Serial blue(p13,p14);
 
//global variable declarations

int inv_gain = 3;                       //scale factor for decreasing delay volume
int delay = 0;                          //store value for delay from smartphone app
float volume = .5;                      //store value for volume from smartphone app
float distortion = 1.0;                 //store value for distortion from smartphone app
unsigned short buffer[MAX_DELAY];       //declare array for storing samples for delay effect
int scaleFactor = 30;                   //global to convert int values from smartphone app sliders

/*******************************************************
 Bluetooth thread for reading input from smartphone app
********************************************************/ 
void threadBluetooth(void const *args)
{
         while(1)
         {
              if(blue.readable())                       //check if new value is available
              {
                   int changeVal;                       //variable to hold value from smartphone app
                   changeVal = blue.getc();             //get new value from app via bluetooth
                   
                   if(changeVal <= scaleFactor)         //check if new value falls in volume slider range
                   {
                       //convert volume to float value in range [0.0-1.0]
                       volume = (float)changeVal/((float)scaleFactor+3);
                   }
                   else if(changeVal <= scaleFactor*2)  //check if value is in delay range
                   {
                       //convert delay to float in range (0.0 - 1.0]
                       delay = ((float)changeVal-((float)scaleFactor+1))/((float)scaleFactor)*MAX_DELAY;
                       
                       if(changeVal == 31)              //if new value is bottom of delay slider
                       {
                           delay = 0;                   //set delay to zero (disable effect)
                       }
                   }
                   else if(changeVal <= scaleFactor*3)  //check if value is in distortion range
                   {
                        //convert distortion to float in range (3.0 - 4.0)
                        distortion = 3.0 + ((float)changeVal-(2*(float)scaleFactor+1))/((float)scaleFactor);
                        
                        if(changeVal == 61)             //if new value is bottom of distortion slider
                        {
                            distortion = 1;             //set distortion to one (disable effect)
                        }
                   }
        }
        Thread::wait(100);
}                                                       //end bluetooth thread

/*****
 Main
*****/
  
int main(void)
{
    Thread threadBlue(threadBluetooth);                 //startup Bluetooth thread for smartphone app
    int oldDelay;                                       //global for delay comparison
    
    while(1)
    {
        int i;                                          //for loop count variable
        
        if(delay > 0)                                   //check if delay effect is on
        {
            if(delay != oldDelay)                       //if delay value has changed re-buffer
            {
                for (i = 0; i < delay; i++)             //buffer loop
                {
                    buffer[i] += input.read_u16();
                    oldDelay = delay;
                }
            }
            while(delay > 0)                                        //while delay is on
            {
                buffer[i] = buffer[i]/inv_gain + input.read_u16();  //scale buffer value and add to input
                output.write_u16(distortion * volume * buffer[i]);  //write to output
                i = (i+1) % delay;                                  //traverse buffer array(use delay number of values)
            }
        }
        else
        {
            //with no delay write input value scaled by volume and distortion 
            output.write_u16(distortion * volume * input.read_u16());    
            wait_us(23);                                             //wait time based on sample rate
        }
    }
}                       //end main