#include "mbed.h"
#include "radio.h"

#include "mbed-trace/mbed_trace.h"
#define TRACE_GROUP   "slav"
 
static Mutex serialOutMutex;
static void serial_out_mutex_wait()
{
    serialOutMutex.lock();
}

static void serial_out_mutex_release()
{
    osStatus s = serialOutMutex.unlock();
    MBED_ASSERT(s == osOK);
}
 
DigitalOut myled(LED1);

#if defined(SX127x_H) || defined(SX126x_H)
    #define BW_KHZ              500
    #define SPREADING_FACTOR    11
    #define CF_HZ               910800000
#elif defined(SX128x_H)
    #define BW_KHZ              200
    #define SPREADING_FACTOR    11
    #define CF_HZ               2487000000
#endif

#ifdef TARGET_DISCO_L072CZ_LRWAN1
    DigitalOut pinCommandExecute(PB_2);
    DigitalOut pin0(PB_5);
    DigitalOut pin1(PB_6);
    DigitalOut pin2(PB_8);
    DigitalOut pin3(PB_9);
    DigitalOut pin4(PB_12);
    DigitalOut pin5(PB_13);
    DigitalOut pin6(PB_14);
    DigitalOut pin7(PB_15);
#else
    DigitalOut pin4(PC_3);
    DigitalOut pin5(PC_2);
    DigitalOut pin6(PC_6);
    DigitalOut pin7(PC_8);
#endif /* !TARGET_DISCO_L072CZ_LRWAN1 */

Timeout timeoutOutputDelay;
Timeout timeoutCommandExecuteSignal;
static uint8_t command = 0;

#define PIN_ASSERT_us       500000
#define COMMAND_EXECUTE_SIGNAL_PERIOD 200000


/**********************************************************************/

void alarm_pin_clr()
{
    
}


static bool commandWrittenFlag = false;
static bool commandAssertedFlag = false;

void onCommandExecuteSignalTimeout(void)
{
    pinCommandExecute = false;
    commandAssertedFlag = true;
}

void writeCommandToDigitalOuts(void)
{
    pin0 = command & 0b00000001;
    pin1 = command & 0b00000010;
    pin2 = command & 0b00000100;
    pin3 = command & 0b00001000;
    pin4 = command & 0b00010000;
    pin5 = command & 0b00100000;
    pin6 = command & 0b01000000;
    pin7 = command & 0b10000000;
    pinCommandExecute = true;
    commandWrittenFlag = true;
    timeoutCommandExecuteSignal.attach_us(onCommandExecuteSignalTimeout, COMMAND_EXECUTE_SIGNAL_PERIOD);
}

static uint16_t crc_ccitt( uint8_t *buffer, uint16_t length )
{
    tr_debug(__FUNCTION__);
    const uint16_t polynom = 0x1021;    // The CRC calculation follows CCITT
    uint16_t crc = 0x0000;              // CRC initial value

    if( buffer == NULL )
    {
        tr_debug("NULL Buffer");
        return 0;
    }

    for(uint16_t i = 0; i < length; ++i)
    {
        crc ^= ( uint16_t ) buffer[i] << 8;
        for( uint16_t j = 0; j < 8; ++j )
        {
            crc = ( crc & 0x8000 ) ? ( crc << 1 ) ^ polynom : ( crc << 1 );
        }
    }
    return crc;
}

void get_alarm()
{
    tr_debug(__FUNCTION__);
    
    uint16_t rx_crc, crc = crc_ccitt(Radio::radio.rx_buf, 5);
    rx_crc = Radio::radio.rx_buf[5];
    rx_crc <<= 8;
    rx_crc += Radio::radio.rx_buf[6];
    //tr_debug("%u) crc rx:%04x, calc:%04x\r\n", lora.RegRxNbBytes, rx_crc, crc);
    
    if (crc != rx_crc)
    {
        tr_debug("crc fail %04x, %04x\r\n", rx_crc, crc);
        return;
    }
    
    CriticalSectionLock::enable();
    command = Radio::radio.rx_buf[0];
    CriticalSectionLock::disable();
    
    unsigned delay;
    delay = Radio::radio.rx_buf[1];
    delay <<= 8;
    delay += Radio::radio.rx_buf[2];
    delay <<= 8;
    delay += Radio::radio.rx_buf[3];
    delay <<= 8;
    delay += Radio::radio.rx_buf[4];
    
    timeoutOutputDelay.attach_us(writeCommandToDigitalOuts, delay);
    tr_debug("output delay:%u\r\n", delay);
}

void txDoneCB()
{
}

void rxDoneCB(uint8_t size, float Rssi, float Snr)
{
    get_alarm();
    tr_debug("%.1fdBm  snr:%.1fdB ", Rssi, Snr);
}

const RadioEvents_t rev =
{
    /* Dio0_top_half */     NULL,
    /* TxDone_topHalf */    NULL,
    /* TxDone_botHalf */    txDoneCB,
    /* TxTimeout  */        NULL,
    /* RxDone  */           rxDoneCB,
    /* RxTimeout  */        NULL,
    /* RxError  */          NULL,
    /* FhssChangeChannel  */NULL,
    /* CadDone  */          NULL
};

 
int main()
{   
    mbed_trace_mutex_wait_function_set( serial_out_mutex_wait );
    mbed_trace_mutex_release_function_set( serial_out_mutex_release );
    mbed_trace_init();
    tr_debug(__FUNCTION__);

    Radio::Init(&rev);
    Radio::Standby();
    Radio::LoRaModemConfig(BW_KHZ, SPREADING_FACTOR, 1);
    Radio::LoRaPacketConfig(8, false, true, false);  // preambleLen, fixLen, crcOn, invIQ
    Radio::SetChannel(CF_HZ);
    Radio::Rx(0);
    
    while(true)
    {     
        Radio::service();
        if(commandWrittenFlag) 
        {
            tr_debug("Command Written: %d", command);
            CriticalSectionLock::enable();
            commandWrittenFlag = false;
            CriticalSectionLock::disable();
        }
        if(commandAssertedFlag)
        {
            tr_debug("Command Asserted: %d", command);
            CriticalSectionLock::enable();
            commandAssertedFlag = false;
            CriticalSectionLock::disable();
        }
    }
}
